#include <HidroAccumulator.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

#include <cstdlib>


HidroAccumulator::HidroAccumulator( 
  TeRaster* lddRaster,
  TeRaster* &accumulatedRaster,
  bool kmArea ) :
    HidroFlowAlgorithm( lddRaster ),    
    lddRaster_(lddRaster),
    accumulatedRaster_(accumulatedRaster),
    kmArea_(kmArea)
{  
}

HidroAccumulator::~HidroAccumulator()  
{
  // Free Memory
  lddMatrix_.clear();
  accumulatedMatrix_.clear();
}

bool
HidroAccumulator::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output Accumulated raster
  
  // accumulatedRaster params
  TeRasterParams accumulatedRasterParams = lddRaster_->params();
  
  // Set dummy
  accumulatedRasterParams.setDummy( -9999 );
  accumulatedRasterParams.useDummy_ = true;  

  // Change mode
  accumulatedRasterParams.setDataType( TeFLOAT );
  accumulatedRasterParams.mode_ = 'w';
  accumulatedRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  accumulatedRasterParams.vmax_[0] = -TeMAXFLOAT;
  accumulatedRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  accumulatedRaster_ = new TeRaster( accumulatedRasterParams );

  // verify if accumulatedRaster created is valid
  if( !accumulatedRaster_->init() )
  {
    errorMessage_ = accumulatedRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = accumulatedRaster_->params().nlines_;
  unsigned int ncolumns = accumulatedRaster_->params().ncols_;  
    
  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating Accumulated Raster step 1 from 4.");
  TeProgress::instance()->setTotalSteps( nlines );

  // init accumulated matrix
  if( kmArea_ )
  {
    if( !initAccumulatedMatrixArea() )
    {
      return false;
    }
  }
  else
  {
    if( !initAccumulatedMatrixPixel() )
    {
      return false;
    }
  }  

  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Accumulated Raster step 2 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  if( !initConnections() )
  {
    return false;
  }

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Accumulated Raster step 3 from 4.");
  TeProgress::instance()->setProgress( 0 );

  // Calculate the accumulated matrix
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineFrom = line;
        columnFrom = column;

        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {
          if( lddMatrix_[lineTo][columnTo] == 255 )
          {
            break;
          }

          accumulatedMatrix_[lineTo][columnTo] = accumulatedMatrix_[lineTo][columnTo] + accumulatedMatrix_[lineFrom][columnFrom];          

          // if connections bigger than one
          if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
          {
            connectionsMatrix_[lineTo][columnTo]--;
            break;
          }
          
          lineFrom = lineTo;
          columnFrom = columnTo;
        }        
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory  
  lddMatrix_.clear();

  // STEP 4

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Creating Accumulated Raster step 4 from 4.");
  if( !copyPDIMatrix2Raster( accumulatedMatrix_, accumulatedRaster_ ) )
  {
    return cancel();
  }

  // Free Memory
  accumulatedMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroAccumulator::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  accumulatedMatrix_.clear();  

  return HidroFlowAlgorithm::cancel();
}

bool
HidroAccumulator::initAccumulatedMatrixPixel()
{
  unsigned int nlines = accumulatedRaster_->params().nlines_;
  unsigned int ncolumns = accumulatedRaster_->params().ncols_;

  accumulatedMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, accumulatedMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( lddMatrix_[line][column] != 255 )
      {
        accumulatedMatrix_[line][column] = 1;
      }
      else
      {
        accumulatedMatrix_[line][column] = -9999;
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  return true;
}

bool
HidroAccumulator::initAccumulatedMatrixArea()
{
  unsigned int nlines = accumulatedRaster_->params().nlines_;
  unsigned int ncolumns = accumulatedRaster_->params().ncols_;

  double A = lddRaster_->params().projection()->datum().radius() / 1000.0;
  double F = lddRaster_->params().projection()->datum().flattening();  

  //double A = 6378.137; //!COMPRIMENTO DO SEMI EIXO MAIOR DO ELIPSIDE (KM)
	//double B = 6356.752; //!COMPRIMENTO DO SEMI EIXO MENOR DO ELIPSIDE (KM)
  //double F = ( A - B ) / A; //!ACHATAMENTO DO ELIPSIDE
	
  double E2 = 2 * F - F * F; //!QUADRADO DA EXCENTRICIDADE

  double resx = lddRaster_->params().resx_;
  double resy = lddRaster_->params().resy_;

  accumulatedMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, accumulatedMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    // A rea varia somente na lina / y / latitude
    
    TeCoord2D index( 0, line );
    TeCoord2D coord = lddRaster_->index2Coord( index );

    double YLAT = coord.y();        
    
    double RN =  A / sqrt( ( 1 - E2 * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) ); //!RAIO DE CURVATURA DA TERRA NA LATITUDE

    double AREACEL = ((M_PI * RN*RN) / 180) * 
                      fabs( sin( (YLAT-resy/2) * M_PI/180) - sin( (YLAT+resy/2) * M_PI/180 ) ) *
                      fabs( resx );

    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( lddMatrix_[line][column] != 255 )
      {        
        accumulatedMatrix_[line][column] = (float) AREACEL;
      }
      else
      {
        accumulatedMatrix_[line][column] = -9999;
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  return true;
}