#include <HidroDrainageExtraction.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

HidroDrainageExtraction::HidroDrainageExtraction(
  TeRaster* lddRaster,
  TeRaster* contributingAreaRaster,
  float accumulatedThreshold,
  TeRaster* &drainageRaster,
  TeRaster* vdnpRaster,
  float vdnpThreshold) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    contributingAreaRaster_(contributingAreaRaster),
    contributingAreaThreshold_(accumulatedThreshold),
    drainageRaster_(drainageRaster),
    vdnpRaster_(vdnpRaster),
    vdnpThreshold_(vdnpThreshold)    
{  
}

HidroDrainageExtraction::~HidroDrainageExtraction()
{   
  accumulatedMatrix_.clear();
  vdnpMatrix_.clear();
  drainageMatrix_.clear();
}


bool
HidroDrainageExtraction::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // load data

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Accumulated data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Accumulated
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Accumulated data.");    
  if( !copyRaster2PDIMatrix( contributingAreaRaster_, accumulatedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load vdnp
  if( vdnpThreshold_ > 0 )
  {
    TeProgress::instance()->reset();
	  TeProgress::instance()->setCaption("TerraHidro");
	  TeProgress::instance()->setMessage("Load VDNP data.");    
    if( !copyRaster2PDIMatrix( vdnpRaster_, vdnpMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      return cancel();
    }
  }

  // create output drainage raster
  
  // drainageRaster params
  TeRasterParams drainageRasterParams = contributingAreaRaster_->params();
  
  // Set dummy
  drainageRasterParams.setDummy( 255 );
  drainageRasterParams.useDummy_ = true;  

  // Change mode
  drainageRasterParams.mode_ = 'w';
  drainageRasterParams.decoderIdentifier_ = "SMARTMEM";
  drainageRasterParams.setDataType( TeUNSIGNEDCHAR );

  // Set Max and Minimal values
  drainageRasterParams.vmax_[0] = -TeMAXFLOAT;
  drainageRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  drainageRaster_ = new TeRaster( drainageRasterParams );

  // verify if drainageRaster created is valid
  if( !drainageRaster_->init() )
  {
    errorMessage_ = drainageRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  nlines_ = drainageRaster_->params().nlines_;
  ncolumns_ = drainageRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Drainage Raster step 1 from 3.");
  TeProgress::instance()->setTotalSteps( nlines_ );

  // incializa matrix de drenagem com dummys (255)
  drainageMatrix_.Reset( nlines_, ncolumns_, TePDIMatrix<unsigned char>::AutoMemPol, drainageMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      drainageMatrix_[line][column] = (unsigned char) 255;     
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 2
  TeProgress::instance()->setMessage("Creating Drainage Raster step 2 from 3.");
  TeProgress::instance()->setTotalSteps( nlines_ );

 
  if( vdnpThreshold_ > 0 )
  {    
    if( !bouthThreshold() )
      return false;
  }
  else
  {
    if( !onlyAccumulatedThreshold() )
      return false;
  }

  // Free Memory  
  accumulatedMatrix_.clear();
  vdnpMatrix_.clear();

  // STEP 3
  // fix the holes
   TeProgress::instance()->setMessage("Creating Drainage Raster step 3 from X.");
  TeProgress::instance()->setTotalSteps( nlines_ );
   if( vdnpThreshold_ > 0 )
  {
    if( !fix() )
      return false;
  }

  // STEP 4

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Creating Drainage Raster step 3 from 3.");
  if( !copyPDIMatrix2Raster( drainageMatrix_, drainageRaster_ ) )
  {
    return cancel();
  }

  // Free Memory 
  drainageMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroDrainageExtraction::cancel()
{
  // Free Memory  
  accumulatedMatrix_.clear();
  vdnpMatrix_.clear();
  drainageMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}

bool HidroDrainageExtraction::onlyAccumulatedThreshold()
{
  // find the drainage
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if(  accumulatedMatrix_[line][column] > contributingAreaThreshold_ )
      {
        drainageMatrix_[line][column] = 1;
      }      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  return true;
}

bool HidroDrainageExtraction::bouthThreshold()
{
  // find the drainage
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if(  (accumulatedMatrix_[line][column] > contributingAreaThreshold_ ) && 
            ( vdnpMatrix_[line][column] > vdnpThreshold_ ) )
      {
        drainageMatrix_[line][column] = 1;
      }      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  return true;
}


bool HidroDrainageExtraction::fix()
{
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  // find the drainage
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if(  drainageMatrix_[line][column] == 1 )
      {
          lineFrom = line;
          columnFrom = column;

          while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
          {
            if( lddMatrix_[lineTo][columnTo] == 255 )
            {
              break;
            }

            if( drainageMatrix_[lineTo][columnTo] == 1 )
            {
              break;
            }

            drainageMatrix_[lineTo][columnTo] = 1;           
          
            lineFrom = lineTo;
            columnFrom = columnTo;
          }
      }      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  return true;
}