/*!
	\brief This is the Hidro Flow Group.
	@defgroup hidroFlow The Group for Plugin Flow Classes.
*/

#ifndef HIDRO_DRAINAGE_EXTRACTION_H
#define HIDRO_DRAINAGE_EXTRACTION_H

/** 
  * \file HidroDrainageExtraction.h
  *
  *
  * \brief This file is a class extract the drainage of a accumulated raster and VDNP raster.
  *
  * \author Alexandre Copertino Jardim <alexandre@dpi.inpe.br>
  * \version 1.0
  *
/*!
  @ingroup hidroFlow
 */

/*
** ----------------------------------------------------------------------------
** Includes:
*/

#include <HidroFlowAlgorithm.h>

#include <TePDIMatrix.hpp>

class TeRaster;

class HidroDrainageExtraction : public HidroFlowAlgorithm
{
public:
  /**
   * @name Constructor and Destructor Methods.
	 *  Methods related to instantiation and destruction.
	**/
	//@{
  
    /**
    * \brief Constructor with parameters.
	  * \param contributingAreaRaster Accumulated raster (input).
    * \param accumulatedThreshold Accumulated Threshold (input).
    * \param vdnpRaster VDNP raster (input).
    * \param vdnpThreshold VDNP Threshold (input).
    * \param drainageRaster Drainage raster (result).
	  **/
    HidroDrainageExtraction(
      TeRaster* lddRaster,
      TeRaster* contributingAreaRaster,
      float accumulatedThreshold,
      TeRaster* &drainageRaster,
      TeRaster* vdnpRaster = NULL,
      float vdnpThreshold = 0 );

    ~HidroDrainageExtraction();
  
  //@}

  /**
   * @name Behavioral Methods
	 * Methods related to hidro Drainage behavior.
	**/
	//@{

    /**
    * \brief Extract the drainage of a accumulated and vdnp raster.    
    * \return True if successful false otherwise.
	  **/
    bool execute();

  //@}

protected:
  
  /**
    * \brief Extract the drainage of a accumulated and vdnp raster.    
    * \return True if successful false otherwise.
	  **/
  bool cancel();

  /**
    * \brief Extract the drainage of a accumulated and vdnp raster.    
    * \return True if successful false otherwise.
	  **/
  bool onlyAccumulatedThreshold();
  bool bouthThreshold();
  bool fix();
  
  /**
   * @name Attributes
	 * Attributes related to hidro Drainage.
	**/
	//@{
    unsigned int nlines_;
    unsigned int ncolumns_;

    TeRaster* lddRaster_;
    TeRaster* contributingAreaRaster_;
    TeRaster* vdnpRaster_;
    TeRaster* &drainageRaster_;

    TePDIMatrix<float> accumulatedMatrix_; //!< accumulated matrix.
    TePDIMatrix<float> vdnpMatrix_; //!< VDNP matrix.
    TePDIMatrix<unsigned char> drainageMatrix_; //!< drainage matrix.

    float contributingAreaThreshold_; //!< Accumulated Threshold.
    float vdnpThreshold_; //!< VDNP Threshold.
  //@}
};

#endif //HIDRO_DRAINAGE_EXTRACTION_H

