#include <HidroMiniBasins.h>

#include <TeRaster.h>

#include <HidroFlowUtils.h>

HidroMiniBasins::HidroMiniBasins(  
  TeRaster* lddRaster,
  TeRaster* segmentedRaster,
  TeRaster* &miniBasinsRaster ) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    segmentedRaster_(segmentedRaster),
    miniBasinsRaster_(miniBasinsRaster)
{  
}

HidroMiniBasins::~HidroMiniBasins()
{
  // Free Memory  
  segmentedMatrix_.clear();
  miniBasinsMatrix_.clear();
}


bool
HidroMiniBasins::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( segmentedRaster_->params().nlines_ != lddRaster_->params().nlines_ || segmentedRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "Segmented lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // load data  

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Segmented
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Segmented data.");    
  if( !copyRaster2PDIMatrix( segmentedRaster_, segmentedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output MiniBasins raster
  
  // miniBasinsRaster params
  TeRasterParams miniBasinsRasterParams = lddRaster_->params();
  
  // Set dummy
  miniBasinsRasterParams.setDummy( -9999 );
  miniBasinsRasterParams.useDummy_ = true;  

  // Change mode
  miniBasinsRasterParams.mode_ = 'w';
  miniBasinsRasterParams.decoderIdentifier_ = "SMARTMEM";
  miniBasinsRasterParams.setDataType( TeINTEGER );

  // Set Max and Minimal values
  miniBasinsRasterParams.vmax_[0] = -TeMAXFLOAT;
  miniBasinsRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  miniBasinsRaster_ = new TeRaster( miniBasinsRasterParams );

  // verify if miniBasinsRaster created is valid
  if( !miniBasinsRaster_->init() )
  {
    errorMessage_ = miniBasinsRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = miniBasinsRaster_->params().nlines_;
  unsigned int ncolumns = miniBasinsRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating Mini Basins Raster step 1 from 4.");
  TeProgress::instance()->setTotalSteps( nlines );

  // init miniBasinsMatrix
  miniBasinsMatrix_.Reset( nlines, ncolumns, TePDIMatrix<int>::AutoMemPol, miniBasinsMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( segmentedMatrix_[line][column] != -9999 )
      {
        miniBasinsMatrix_[line][column] = segmentedMatrix_[line][column];        
      }
      else
      {        
        if( lddMatrix_[line][column] == 255 )
        {
          // dummy
          miniBasinsMatrix_[line][column] = -9999;
        }
        else
        {
          // flag until dummy
          miniBasinsMatrix_[line][column] = -9998;
        }        
      }
    }
    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // free memory
  segmentedMatrix_.clear();

  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Mini Basins Raster step 2 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  if( !initConnections() )
  {
    return false;
  }  

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Mini Basins Raster step 3 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // fill the miniBasinsRaster with segmented values
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  unsigned int lineStop;
  unsigned int columnStop;

  int segment;
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineTo = line;
        columnTo = column;

        segment = -9999; // dummy

        // find the segment value or miniBasinsRasterDummy
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;

          lineStop = lineFrom;
          columnStop = columnFrom;

          if( miniBasinsMatrix_[lineFrom][columnFrom] == -9999 )
          {
            segment = -9999; // dummy
            break;
          }

          if( miniBasinsMatrix_[lineFrom][columnFrom] != -9998 )
          {
            segment = miniBasinsMatrix_[lineFrom][columnFrom];
            break;
          }
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );

        // fill the path with segment value
        lineTo = line;
        columnTo = column;
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;

          miniBasinsMatrix_[lineFrom][columnFrom] = segment;

          if( lineStop == lineFrom && columnStop == columnFrom )
          {
            break;
          }          
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory  
  lddMatrix_.clear();

  // STEP 4

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Mini Basins Raster step 4 from 4.");
  TeProgress::instance()->setProgress( 0 );

  if( !copyPDIMatrix2Raster( miniBasinsMatrix_, miniBasinsRaster_ ) )
  {
    return cancel();
  }  

  // Free Memory  
  miniBasinsMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroMiniBasins::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  segmentedMatrix_.clear();
  miniBasinsMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}