#include <HidroSegmenter.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>


HidroSegmenter::HidroSegmenter(
  TeRaster* lddRaster,
  TeRaster* drainageRaster,
  TeRaster* &segmentedRaster ) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    drainageRaster_(drainageRaster),
    segmentedRaster_(segmentedRaster)
{  
}

HidroSegmenter::~HidroSegmenter()
{
  // Free memory
  drainageMatrix_.clear();
  segmentedMatrix_.clear();
}


bool
HidroSegmenter::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( drainageRaster_->params().nlines_ != lddRaster_->params().nlines_ || drainageRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "Drainage lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Drainage
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Drainage data.");    
  if( !copyRaster2PDIMatrix( drainageRaster_, drainageMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output Segmented raster
  
  // segmentedRaster params
  TeRasterParams segmentedRasterParams = lddRaster_->params();
  
  // Set dummy
  segmentedRasterParams.setDummy( -9999 );
  segmentedRasterParams.useDummy_ = true;  

  // Change mode
  segmentedRasterParams.mode_ = 'w';
  segmentedRasterParams.decoderIdentifier_ = "SMARTMEM";
  segmentedRasterParams.setDataType( TeINTEGER );

  // Set Max and Minimal values
  segmentedRasterParams.vmax_[0] = -TeMAXFLOAT;
  segmentedRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  segmentedRaster_ = new TeRaster( segmentedRasterParams );

  // verify if segmentedRaster created is valid
  if( !segmentedRaster_->init() )
  {
    errorMessage_ = segmentedRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = segmentedRaster_->params().nlines_;
  unsigned int ncolumns = segmentedRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Segmented Raster step 1 from 5.");
  TeProgress::instance()->setTotalSteps( nlines );

  // init segmentedMatrix
  segmentedMatrix_.Reset( nlines, ncolumns, TePDIMatrix<int>::AutoMemPol, segmentedMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {     
      // flag until dummy
      segmentedMatrix_[line][column] = -9999;     
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Segmented Raster step 2 from 5.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  TeProgress::instance()->setTotalSteps( nlines * 2 );

  // init connection matrix with zeros and dummys (255)  
  bool res = connectionsMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, connectionsMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddMatrix_[line][column] != 255 && drainageMatrix_[line][column] == 1 )
      {
        connectionsMatrix_[line][column] = 0;
      }
      else
      {
        connectionsMatrix_[line][column] = 255;
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  
  // calculate connection matrix
  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddCode2LineColumn( line, column, lineTo, columnTo ) )
      {
        if( connectionsMatrix_[lineTo][columnTo] != 255 && drainageMatrix_[line][column] == 1 )
        {
          connectionsMatrix_[lineTo][columnTo]++;
        }
      }     
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( nlines + line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Segmented Raster step 3 from 5.");
  TeProgress::instance()->setProgress( 0 );

  // find the segments
  unsigned int lineFrom;
  unsigned int columnFrom;

  int segmenter = 0;
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineTo = line;
        columnTo = column;

        segmenter++; // new segmenter
        
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;          

          if( lddMatrix_[lineFrom][columnFrom] == 255 )
          {            
            break;
          }          

          if( connectionsMatrix_[lineFrom][columnFrom] > 1 )
          {
            connectionsMatrix_[lineFrom][columnFrom]--;

            // mark as flag;
            segmentedMatrix_[lineFrom][columnFrom] = -9998;
            break;
          }

          if( segmentedMatrix_[lineFrom][columnFrom] == -9998 )
          {
            segmenter++;
          }

          segmentedMatrix_[lineFrom][columnFrom] = segmenter;          
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory  
  lddMatrix_.clear();
  drainageMatrix_.clear();
  connectionsMatrix_.clear();

  // STEP 5

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Creating Segmented Raster step 5 from 5.");
  if( !copyPDIMatrix2Raster( segmentedMatrix_, segmentedRaster_ ) )
  {
    return cancel();
  }

  // Free Memory  
  segmentedMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();


  return true;
}

bool HidroSegmenter::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  drainageMatrix_.clear();
  segmentedMatrix_.clear();
  connectionsMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}
