#include <HidroUpscaleRaster.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

#include<queue>

HidroUpscaleRaster::HidroUpscaleRaster(
    TeRaster* lddRaster,
    TeRaster* accumulatedRaster,
    TeRaster* &scaledRaster,
    TeRaster* &outletPixelsRaster,
    int factor,
    int mufp,
    int at ) :
HidroFlowAlgorithm( lddRaster ),
lddRaster_(lddRaster),
accumulatedRaster_(accumulatedRaster),
scaledRaster_(scaledRaster),
outletPixelsRaster_(outletPixelsRaster),
factor_(factor),
mufp_(mufp),
at_(at)
{
  neighborDirection_[0][0] = 32;
  neighborDirection_[0][1] = 64;
  neighborDirection_[0][2] = 128;
  
  neighborDirection_[1][0] = 16;
  neighborDirection_[1][1] = 0;
  neighborDirection_[1][2] = 1;
  
  neighborDirection_[2][0] = 8;
  neighborDirection_[2][1] = 4;
  neighborDirection_[2][2] = 2;
}

bool
HidroUpscaleRaster::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( accumulatedRaster_->params().nlines_ != lddRaster_->params().nlines_ || accumulatedRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "Accumulated matrix lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // load data

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Accumulated
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Accumulated data.");    
  if( !copyRaster2PDIMatrix( accumulatedRaster_, accumulatedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output Scaled raster
  
  // scaledRaster params
  TeRasterParams scaledRasterParams = lddRaster_->params();
  
  // Set dummy
  scaledRasterParams.setDummy( 255 );
  scaledRasterParams.useDummy_ = true;

  // Change Resolution and nlines ncoloumns
  scaledRasterParams.boundingBoxResolution( lddRaster_->params().boundingBox().x1(),
    lddRaster_->params().boundingBox().y1(),
    lddRaster_->params().boundingBox().x2(),
    lddRaster_->params().boundingBox().y2(),
    lddRaster_->params().resx_ * factor_,
    lddRaster_->params().resy_ * factor_ );    

  // Change mode
  scaledRasterParams.mode_ = 'w';
  scaledRasterParams.decoderIdentifier_ = "SMARTMEM";
  scaledRasterParams.setDataType( TeUNSIGNEDCHAR );

  // Set Max and Minimal values
  scaledRasterParams.vmax_[0] = -TeMAXFLOAT;
  scaledRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  scaledRaster_ = new TeRaster( scaledRasterParams );

  // verify if scaledRaster created is valid
  if( !scaledRaster_->init() )
  {
    errorMessage_ = scaledRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // create output outlet pixels raster
  
  // outletPixelsRaster params
  TeRasterParams outletPixelsRasterParams = lddRaster_->params();
  
  // Set dummy
  outletPixelsRasterParams.setDummy( 255 );
  outletPixelsRasterParams.useDummy_ = true;     

  // Change mode
  outletPixelsRasterParams.mode_ = 'w';
  outletPixelsRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  outletPixelsRasterParams.vmax_[0] = -TeMAXFLOAT;
  outletPixelsRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  outletPixelsRaster_ = new TeRaster( outletPixelsRasterParams );

  // verify if scaledRaster created is valid
  if( !outletPixelsRaster_->init() )
  {
    errorMessage_ = outletPixelsRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // used in for's and progress bar
  unsigned int nlines = scaledRaster_->params().nlines_;
  unsigned int ncolumns = scaledRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 1 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // init scaledMatrix
  scaledMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, scaledMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      scaledMatrix_[line][column] = 255;      
    }    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // used in for's and progress bar
  unsigned int highNines = lddRaster_->params().nlines_;
  unsigned int highNcolumns = lddRaster_->params().ncols_;

  // STEP 2
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 2 from X.");
  TeProgress::instance()->setTotalSteps( highNines );

  // init outletPixelsMatrix  
  outletPixelsMatrix_.Reset( lddRaster_->params().nlines_, lddRaster_->params().ncols_, TePDIMatrix<unsigned char>::AutoMemPol, outletPixelsMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<highNines; line++ )
  {
    for( unsigned int column=0; column<highNcolumns; column++ )
    {      
      outletPixelsMatrix_[line][column] = 255;
    }    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 3
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 3 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // Main loop 1
  // Find the outlet pixels
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      // find the outletPixel
      OutletPixel outletPixel = findOutletPixel( line, column );      

      // store outlet pixel
      if( outletPixel.ldd_ != 0 && outletPixel.ldd_ != 255 )
      {
        outletPixelsMatrix_[outletPixel.line_][outletPixel.column_] = 1;
      }

      scaledMatrix_[line][column] = outletPixel.ldd_;
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 4
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 4 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // Main loop 2
  // Define the LDD for low resolution
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      // re find the outlet pixel
      OutletPixel outletPixel = reFindOutletPixel( line, column );      

      // Check the accumulated area threshold
      if( outletPixel.ldd_ != 0 )
      {
        areaThreshold( line, column, outletPixel );
      }
      else
      {
        outletPixel.ldd_ = 255;
      }

      scaledMatrix_[line][column] = outletPixel.ldd_;
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 5
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 5 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // Main loop 3
  // Correcting the crossing flow path
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      checkCrossing( line, column );      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 6
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 6 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // Main loop 4
  // Correcting the looping flow path
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      checkLooping( line, column );      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 7
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 7 from X.");
  TeProgress::instance()->setTotalSteps( nlines );

  // copy matrix values to raster.  
  if( !copyPDIMatrix2Raster( scaledMatrix_, scaledRaster_ ) )
  {
    return cancel();
  }

  // STEP 8
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 8 from X.");
  TeProgress::instance()->setTotalSteps( lddRaster_->params().nlines_ );

  // copy matrix values to raster.  
  if( !copyPDIMatrix2Raster( outletPixelsMatrix_, outletPixelsRaster_ ) )
  {
    return cancel();
  }

  // Free Memory
  accumulatedMatrix_.clear();
  scaledMatrix_.clear();
  outletPixelsMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}
void
HidroUpscaleRaster::checkCrossing( unsigned int line, unsigned int column )
{
  unsigned int nlines = scaledRaster_->params().nlines_;
  unsigned int ncolumns = scaledRaster_->params().ncols_;

  unsigned char ldd = scaledMatrix_[line][column];

  if( !( ldd == 2 || ldd == 8 || ldd == 32 || ldd == 128 ) )
  {
    return;
  }

  /*if(  ldd == 1 || ldd == 4 || ldd == 16 || ldd == 64  || ldd == 255 || ldd == 0)
  {
    return;
  }*/

  unsigned int lineTo = lineIncrement[ ldd ] + line;
  unsigned int columnTo = columnIncrement[ ldd ] + column;

  // check borders
  if(  lineTo > nlines-1 || columnTo > ncolumns-1  )
  {
    return;
  }
    
  // N1 [line][columnTo]
  // N2 [lineTo][column]

  // check cross N1
  unsigned char lddN1 = scaledMatrix_[line][columnTo];
  unsigned int lineN1To = lineIncrement[ lddN1 ] + line;
  unsigned int columnN1To = columnIncrement[ lddN1 ] + columnTo;

  if( lineN1To == lineTo && columnN1To == column )
  {
    // cross N1
    solveCross( line, column, line, columnTo );    
  }          

  unsigned char lddN2 = scaledMatrix_[lineTo][column];

  unsigned int lineN2To = lineIncrement[ lddN2 ] + lineTo;
  unsigned int columnN2To = columnIncrement[ lddN2 ] + column;

  if( lineN2To == line && columnN2To == columnTo )
  {
    // cross N2
    solveCross( line, column, lineTo, column );    
  }
}

void 
HidroUpscaleRaster::solveCross( unsigned int line1, unsigned int column1, unsigned int line2, unsigned int column2 )
{
  // re find the outlet pixel
  OutletPixel outletPixel1 = reFindOutletPixel( line1, column1 );
  
  OutletPixel outletPixel2 = reFindOutletPixel( line2, column2 );

  if( outletPixel1.accumulatedArea_ > outletPixel2.accumulatedArea_ )
  {
    unsigned char ldd = scaledMatrix_[line1][column1];
    
    unsigned int lineTo = lineIncrement[ ldd ] + line1;
    unsigned int columnTo = columnIncrement[ ldd ] + column1;

    unsigned int l = lineTo - line2 + 1;
    unsigned int c = columnTo - column2 + 1;

    scaledMatrix_[line2][column2] = neighborDirection_[l][c];
  }
  else
  {
    unsigned char ldd = scaledMatrix_[line2][column2];
    
    unsigned int lineTo = lineIncrement[ ldd ] + line2;
    unsigned int columnTo = columnIncrement[ ldd ] + column2;

    unsigned int l = lineTo - line1 + 1;
    unsigned int c = columnTo - column1 + 1;

    scaledMatrix_[line1][column1] = neighborDirection_[l][c];
  }
}

int
HidroUpscaleRaster::calculateDistance( unsigned int line, unsigned int column,  OutletPixel &outletPixel )
{
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines()-1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns() -1;

  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  int distance = 1;

  lineFrom = outletPixel.line_;
  columnFrom = outletPixel.column_;

  while( upStream( lineFrom, columnFrom, lineTo, columnTo ) )
  {
    // check block border
    if( lineTo < blockBeginLine
      || columnTo < blockBeginColumn
      || lineTo > blockEndLine-1
      || columnTo > blockEndColumn-1 )
    {
      break;
    }

    distance++;

    lineFrom = lineTo;
    columnFrom = columnTo;
  }

  return distance;
}

OutletPixel
HidroUpscaleRaster::computeBlockAccumulatedArea( unsigned int line, unsigned int column )
{
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines()-1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns()-1;

  unsigned int blockNLines = blockEndLine-blockBeginLine;
  unsigned int blockNColumns = blockEndColumn-blockBeginColumn;

  // calcular area accumulada da celula
  TePDIMatrix<float> blockAccumulated;
  blockAccumulated.Reset( blockNLines,
    blockNColumns );      

  TePDIMatrix<unsigned char> blockConnectionsMatrix;
  blockConnectionsMatrix.Reset( blockNLines,
    blockNColumns );

  // init conections and accumulated
  for( unsigned int l = 0; l < blockNLines; l++ )
  {
    for( unsigned int c = 0; c < blockNColumns; c++ )
    {
      if( (int)lddMatrix_[blockBeginLine+l][blockBeginColumn+c] != 255 )
      {
        blockConnectionsMatrix[l][c] = 0;
        blockAccumulated[l][c] = 1;
      }
      else
      {
        blockConnectionsMatrix[l][c] = 255;
        blockAccumulated[l][c] = 0;
      }
    }
  }

  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int l = 0; l < blockNLines; l++ )
  {
    for( unsigned int c = 0; c < blockNColumns; c++ )
    {
      if( lddCode2LineColumn( blockBeginLine+l, blockBeginColumn+c, lineTo, columnTo ) )
      {
        // check block border
        if( !(lineTo < blockBeginLine ||
          columnTo < blockBeginColumn || 
          lineTo > blockEndLine-1 ||
          columnTo > blockEndColumn-1) )
        {
          if( (int)blockConnectionsMatrix[lineTo-blockBeginLine][columnTo-blockBeginColumn] != 255 )
          {
            blockConnectionsMatrix[lineTo-blockBeginLine][columnTo-blockBeginColumn]++;
          }
        }
      }
    }
  }

  // accumulate
  // Calculate the accumulated matrix
  unsigned int lineFrom;
  unsigned int columnFrom;
  //unsigned int lineTo;
  //unsigned int columnTo;
  
  for( unsigned int l = 0; l < blockNLines; l++ )
  {
    for( unsigned int c = 0; c < blockNColumns; c++ )
    {      
      if( blockConnectionsMatrix[l][c] == 0 )
      {
        lineFrom = blockBeginLine+l;
        columnFrom = blockBeginColumn+c;

        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {
          if( (int)lddMatrix_[lineTo][columnTo] == 255 )
          {
            break;
          }

          // check block border
          if( lineTo < blockBeginLine ||
            columnTo < blockBeginColumn || 
            lineTo > blockEndLine-1 ||
            columnTo > blockEndColumn-1 )
          {
            break;
          }

          blockAccumulated[lineTo-blockBeginLine][columnTo-blockBeginColumn] = blockAccumulated[lineTo-blockBeginLine][columnTo-blockBeginColumn] + blockAccumulated[lineFrom-blockBeginLine][columnFrom-blockBeginColumn];

          // if connections bigger than one
          if( ((int)blockConnectionsMatrix[lineTo-blockBeginLine][columnTo-blockBeginColumn]) > 1 )
          {
            blockConnectionsMatrix[lineTo-blockBeginLine][columnTo-blockBeginColumn]--;
            break;
          }
          
          lineFrom = lineTo;
          columnFrom = columnTo;
        }        
      }
    }
  }

  OutletPixel blockOutletPixel;
  blockOutletPixel.accumulatedArea_ = 0;
  blockOutletPixel.column_ = 0;
  blockOutletPixel.line_ = 0;
  blockOutletPixel.ldd_ = 0;

  // Lines
  for( unsigned int i = 1; i < blockNColumns-1; i++ )
  {
    // fisrt line UP
    if( (int)lddMatrix_[blockBeginLine][blockBeginColumn+i] == 32 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn+i] == 64 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn+i] == 128 )
    {
      if( blockAccumulated[0][i] > blockOutletPixel.accumulatedArea_ )
      {	        
        blockOutletPixel.accumulatedArea_ = blockAccumulated[0][i];
        blockOutletPixel.column_ = blockBeginColumn+i;
        blockOutletPixel.line_ = blockBeginLine;
        blockOutletPixel.ldd_ = 64;
      }
    }

    // last line DOWN
    if( (int)lddMatrix_[blockEndLine-1][blockBeginColumn+i] == 8 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn+i] == 4 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn+i] == 2 )
    {
      if( blockAccumulated[blockNLines-1][i] > blockOutletPixel.accumulatedArea_ )
      {
        blockOutletPixel.accumulatedArea_ = blockAccumulated[blockNLines-1][i];
        blockOutletPixel.column_ = blockBeginColumn+i;
        blockOutletPixel.line_ = blockEndLine-1;
        blockOutletPixel.ldd_ = 4;
      }
    }
  }
  
  // Columns
  for( unsigned int i = 1; i < blockNLines-1; i++ )
  {
    // fisrt column LEFT
    if( (int)lddMatrix_[blockBeginLine+i][blockBeginColumn] == 32 ||
      (int)lddMatrix_[blockBeginLine+i][blockBeginColumn] == 16 ||
      (int)lddMatrix_[blockBeginLine+i][blockBeginColumn] == 8 )
    {      
      if( blockAccumulated[i][0] > blockOutletPixel.accumulatedArea_ )
      {    
        blockOutletPixel.accumulatedArea_ = blockAccumulated[i][0];
        blockOutletPixel.column_ = blockBeginColumn;
        blockOutletPixel.line_ = blockBeginLine+i;
        blockOutletPixel.ldd_ = 16;
      }      
    }

    // last column RIGHT
    if( (int)lddMatrix_[blockBeginLine+i][blockEndColumn-1] == 128 ||
      (int)lddMatrix_[blockBeginLine+i][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[blockBeginLine+i][blockEndColumn-1] == 2 )
    {
      if( blockAccumulated[i][blockNColumns-1] > blockOutletPixel.accumulatedArea_ )
      {        
        blockOutletPixel.accumulatedArea_ = blockAccumulated[i][blockNColumns-1];
        blockOutletPixel.column_ = blockEndColumn-1;
        blockOutletPixel.line_ = blockBeginLine+i;
        blockOutletPixel.ldd_ = 1;
      }
    }
  }
  
  // Diagonals
  // up left
  if( (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 8 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 16 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 32 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 64 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 128 )
    {
      if( blockAccumulated[0][0] > blockOutletPixel.accumulatedArea_ )
      {
        blockOutletPixel.accumulatedArea_ = blockAccumulated[0][0];
        blockOutletPixel.column_ = blockBeginColumn;
        blockOutletPixel.line_ = blockBeginLine;

        unsigned char lddResult = lddMatrix_[blockBeginLine][blockBeginColumn];
        if( (int)lddResult == 8 )
          lddResult = 16;
        if( (int)lddResult == 128 )
          lddResult = 64;
        
        blockOutletPixel.ldd_ = lddResult;
      }
    }  

    // up rigth
    if( (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 2 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 32 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 64 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 128 )
    {
      if( blockAccumulated[0][blockNColumns-1] > blockOutletPixel.accumulatedArea_ )
      {        
        blockOutletPixel.accumulatedArea_ = blockAccumulated[0][blockNColumns-1];
        blockOutletPixel.column_ = blockEndColumn-1;
        blockOutletPixel.line_ = blockBeginLine;

        unsigned char lddResult = lddMatrix_[blockBeginLine][blockEndColumn-1];
        if( (int)lddResult == 32 )
          lddResult = 64;
        if( (int)lddResult == 2 )
          lddResult = 1;
        
        blockOutletPixel.ldd_ = lddResult;
      }
    }


    // down left    
    if( (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 2 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 4 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 8 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 16 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 32 )
    {
      if( blockAccumulated[blockNLines-1][0] > blockOutletPixel.accumulatedArea_ )
      {
        blockOutletPixel.accumulatedArea_ = blockAccumulated[blockNLines-1][0];
        blockOutletPixel.column_ = blockBeginColumn;
        blockOutletPixel.line_ = blockEndLine-1;

        unsigned char lddResult = lddMatrix_[blockEndLine-1][blockBeginColumn];
        if( (int)lddResult == 32 )
          lddResult = 16;
        if( (int)lddResult == 2 )
          lddResult = 4;
        
        blockOutletPixel.ldd_ = lddResult;
      }
    }

    // down rigth
    if( (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 2 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 4 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 8 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 128 )
    {
      if(  blockAccumulated[blockNLines-1][blockNColumns-1] > blockOutletPixel.accumulatedArea_ )
      {
        blockOutletPixel.accumulatedArea_ = blockAccumulated[blockNLines-1][blockNColumns-1];
        blockOutletPixel.column_ = blockEndColumn-1;
        blockOutletPixel.line_ = blockEndLine-1;

        unsigned char lddResult = lddMatrix_[blockEndLine-1][blockEndColumn-1];
        if( (int)lddResult == 8 )
          lddResult = 4;
        if( (int)lddResult == 128 )
          lddResult = 64;
        
        blockOutletPixel.ldd_ = lddResult;
      }
    }

  return blockOutletPixel;
}

bool
HidroUpscaleRaster::upStream( const unsigned int &lineFrom,
      const unsigned int &columnFrom,
      unsigned int &lineTo,
      unsigned int &columnTo )
{
  //const unsigned char lddCode = lddMatrix_[lineFrom][columnFrom];

  bool find = false;

  //if( lddCode != 255 && lddCode != 0)
  {
    // look neighbors
    float mostAccumulated = 0;
    
    //up left
    if( (int)lddMatrix_[lineFrom-1][columnFrom-1] ==  2 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom-1];
        lineTo = lineFrom - 1;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //up
    if( (int)lddMatrix_[lineFrom-1][columnFrom] ==  4 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom];
        lineTo = lineFrom - 1;
        columnTo = columnFrom;
        find = true;
      }
    }

    //up rigth
    if( (int)lddMatrix_[lineFrom-1][columnFrom+1] ==  8 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom+1];
        lineTo = lineFrom - 1;
        columnTo = columnFrom + 1;
        find = true;
      }
    }

    //left
    if( (int)lddMatrix_[lineFrom][columnFrom-1] ==  1 )
    {
      if( accumulatedMatrix_[lineFrom][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom][columnFrom-1];
        lineTo = lineFrom;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //right
    if( (int)lddMatrix_[lineFrom][columnFrom+1] ==  16 )
    {
      if( accumulatedMatrix_[lineFrom][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom][columnFrom+1];
        lineTo = lineFrom;
        columnTo = columnFrom + 1;
        find = true;
      }
    }

    //down left
    if( (int)lddMatrix_[lineFrom+1][columnFrom-1] ==  128 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom-1];
        lineTo = lineFrom+1;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //down
    if( (int)lddMatrix_[lineFrom+1][columnFrom] ==  64 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom];
        lineTo = lineFrom+1;
        columnTo = columnFrom;
        find = true;
      }
    }

    //down
    if( (int)lddMatrix_[lineFrom+1][columnFrom+1] ==  32 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom+1];
        lineTo = lineFrom+1;
        columnTo = columnFrom+1;
        find = true;
      }
    }
  }

  return find;
}

bool
HidroUpscaleRaster::initOutletPixelsCandidates( unsigned int line, unsigned int column, priority_queue< OutletPixel, deque<OutletPixel>, greater<OutletPixel> > &outletPixelsCandidates )
{
  // Find the outlets
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines() - 1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns() - 1;

  // Lines
  for( unsigned int i = blockBeginColumn+1; i < blockEndColumn-1; i++ )
  {
    // fisrt line UP
    if( (int)lddMatrix_[blockBeginLine][i] == 32 ||
      (int)lddMatrix_[blockBeginLine][i] == 64 ||
      (int)lddMatrix_[blockBeginLine][i] == 128 ||
      (int)lddMatrix_[blockBeginLine][i] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockBeginLine;
      newOutletPixel.ldd_ = 64;

      outletPixelsCandidates.push( newOutletPixel );
    }
    if( (int)lddMatrix_[blockBeginLine][i] == 255 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = (float)TeMAXFLOAT;
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockBeginLine;
      newOutletPixel.ldd_ = 255;

      outletPixelsCandidates.push( newOutletPixel );
      return false;
    }

    // last line DOWN
    if( (int)lddMatrix_[blockEndLine-1][i] == 8 ||
      (int)lddMatrix_[blockEndLine-1][i] == 4 ||
      (int)lddMatrix_[blockEndLine-1][i] == 2 ||
      (int)lddMatrix_[blockEndLine-1][i] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockEndLine-1;
      newOutletPixel.ldd_ = 4;

      outletPixelsCandidates.push( newOutletPixel );
    }
    if( (int)lddMatrix_[blockEndLine-1][i] == 255 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = (float)TeMAXFLOAT;
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockEndLine-1;
      newOutletPixel.ldd_ = 255;

      outletPixelsCandidates.push( newOutletPixel );
      return false;
    }
  }
  
  // Columns
  for( unsigned int i = blockBeginLine+1; i < blockEndLine-1; i++ )
  {
    // fisrt column LEFT
    if( (int)lddMatrix_[i][blockBeginColumn] == 32 ||
      (int)lddMatrix_[i][blockBeginColumn] == 16 ||
      (int)lddMatrix_[i][blockBeginColumn] == 8 ||
      (int)lddMatrix_[i][blockBeginColumn] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockBeginColumn];
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 16;

      outletPixelsCandidates.push( newOutletPixel );
    }
    if( (int)lddMatrix_[i][blockBeginColumn] == 255 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = (float)TeMAXFLOAT;
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 255;

      outletPixelsCandidates.push( newOutletPixel );
      return false;
    }

    // last column RIGHT
    if( (int)lddMatrix_[i][blockEndColumn-1] == 128 ||
      (int)lddMatrix_[i][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[i][blockEndColumn-1] == 2 ||
      (int)lddMatrix_[i][blockEndColumn-1] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockEndColumn-1];
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 1;

      outletPixelsCandidates.push( newOutletPixel );
    }
    if( (int)lddMatrix_[i][blockEndColumn-1] == 255 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = (float)TeMAXFLOAT;
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 255;

      outletPixelsCandidates.push( newOutletPixel );
      return false;
    }
  }
  
  // Diagonals
  // up left
  if( (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 8 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 16 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 32 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 64 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 128 ||
      (int)lddMatrix_[blockBeginLine][blockBeginColumn] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockBeginColumn];
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = blockBeginLine;

      unsigned char lddResult = lddMatrix_[blockBeginLine][blockBeginColumn];
      if( (int)lddResult == 8 )
        lddResult = 16;
      if( (int)lddResult == 128 )
        lddResult = 64;
      if( (int)lddResult == 0 )
        lddResult = 32;
      
      newOutletPixel.ldd_ = lddResult;

      outletPixelsCandidates.push( newOutletPixel );
    }

    // up rigth
  if( (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 2 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 32 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 64 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 128 ||
      (int)lddMatrix_[blockBeginLine][blockEndColumn-1] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockEndColumn-1];
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = blockBeginLine;

      unsigned char lddResult = lddMatrix_[blockBeginLine][blockEndColumn-1];
      if( (int)lddResult == 32 )
        lddResult = 64;
      if( (int)lddResult == 2 )
        lddResult = 1;
      if( (int)lddResult == 0 )
        lddResult = 128;
      
      newOutletPixel.ldd_ = lddResult;

      outletPixelsCandidates.push( newOutletPixel );
    }  

    // down left    
  if( (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 2 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 4 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 8 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 16 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 32 ||
      (int)lddMatrix_[blockEndLine-1][blockBeginColumn] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockBeginColumn];
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = blockEndLine-1;

      unsigned char lddResult = lddMatrix_[blockEndLine-1][blockBeginColumn];
      if( (int)lddResult == 32 )
        lddResult = 16;
      if( (int)lddResult == 2 )
        lddResult = 4;
      if( (int)lddResult == 0 )
        lddResult = 8;
      
      newOutletPixel.ldd_ = lddResult;

      outletPixelsCandidates.push( newOutletPixel );
    }  

    // down rigth
  if( (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 1 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 2 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 4 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 8 ||
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 128 || 
      (int)lddMatrix_[blockEndLine-1][blockEndColumn-1] == 0 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockEndColumn-1];
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = blockEndLine-1;

      unsigned char lddResult = lddMatrix_[blockEndLine-1][blockEndColumn-1];
      if( (int)lddResult == 8 )
        lddResult = 4;
      if( (int)lddResult == 128 )
        lddResult = 64;
      if( (int)lddResult == 0 )
        lddResult = 2;
      
      newOutletPixel.ldd_ = lddResult;

      outletPixelsCandidates.push( newOutletPixel );
    }

    return true;
}

OutletPixel 
HidroUpscaleRaster::findOutletPixel( unsigned int line, unsigned int column )
{
  // Priority with outletPixels candidates in a priority order.
  priority_queue< OutletPixel, deque<OutletPixel>, greater<OutletPixel> > outletPixelsCandidates;

  // init outletPixelsCandidates
  if( !initOutletPixelsCandidates( line, column, outletPixelsCandidates ) )
  {
    OutletPixel dummyOutletPixel;
    dummyOutletPixel = outletPixelsCandidates.top();
    dummyOutletPixel.accumulatedArea_ = 0.0;
    dummyOutletPixel.ldd_ = 255;
    return dummyOutletPixel;
  }

  bool compulteAccumulateArea = true;

  OutletPixel blockOutletPixel;
  blockOutletPixel.accumulatedArea_ = 0;
  blockOutletPixel.line_ = 0;
  blockOutletPixel.column_ = 0;
  blockOutletPixel.ldd_ = 0;

  OutletPixel topOutletPixel;
  topOutletPixel.accumulatedArea_ = 0;
  topOutletPixel.line_ = 0;
  topOutletPixel.column_ = 0;
  topOutletPixel.ldd_ = 0;

  // main loop
  while( !outletPixelsCandidates.empty() )
  {
    topOutletPixel = outletPixelsCandidates.top();

    int distance = calculateDistance( line, column, topOutletPixel );

    // check MUFP parameter
    if( distance > mufp_ )
    {
      return topOutletPixel;
    }

    // Compute block accumulated area only first time.
    if( compulteAccumulateArea )
    {
      blockOutletPixel = computeBlockAccumulatedArea( line, column );

      compulteAccumulateArea = false;
    }

    if( blockOutletPixel == topOutletPixel )
    {
      return topOutletPixel;
    }

    outletPixelsCandidates.pop();
  }

  OutletPixel failPixel;
  failPixel.accumulatedArea_ = 0;
  failPixel.line_ = 0;
  failPixel.column_ = 0;  
  failPixel.ldd_ = 0;
  return failPixel;
}

bool
HidroUpscaleRaster::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  accumulatedMatrix_.clear();
  scaledMatrix_.clear();
  outletPixelsMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}

void
HidroUpscaleRaster::areaThreshold( unsigned int line, unsigned int column, OutletPixel &outletPixel )
{
  if( outletPixel.ldd_ == 0 )
  {
    return;
  }

  // neighborhood border
  unsigned int neighborhoodBeginLine;
  if( line == 0 || line-1 == 0)
    neighborhoodBeginLine = 1;
  else
    neighborhoodBeginLine = (line-1) * factor_;

  unsigned int neighborhoodEndLine = (line+1) * factor_ + factor_;

  if( neighborhoodEndLine > lddMatrix_.GetLines()-1 )
    neighborhoodEndLine = lddMatrix_.GetLines() - 1;

  unsigned int neighborhoodBeginColumn;
  if( column==0 || column-1 == 0 )
    neighborhoodBeginColumn = 1;
  else
    neighborhoodBeginColumn = (column-1) * factor_;
  
  unsigned int neighborhoodEndColumn = (column+1) * factor_ + factor_;

  if( neighborhoodEndColumn > lddMatrix_.GetColumns()-1 )
    neighborhoodEndColumn = lddMatrix_.GetColumns() - 1;

  unsigned int lineFrom = outletPixel.line_;
  unsigned int lineTo;
  unsigned int columnFrom = outletPixel.column_;
  unsigned int columnTo;

  unsigned int l = 0;
  unsigned int c = 0;

  bool outletFounded = false;

  while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
  {
    if( (int)lddMatrix_[lineTo][columnTo] == 255 || (int)lddMatrix_[lineTo][columnTo] == 0 )
    {
      if( outletFounded == true )
      {
        outletPixel.ldd_ = neighborDirection_[l][c];
        return;
      }

      // se tiver nas bordas
      if( line == 0 ||
        column == 0 ||
        line == scaledMatrix_.GetLines()-1 ||
        column == scaledMatrix_.GetColumns()-1 ) 
      {
        // out of high resolution border
        if( lineTo == 0 || lineTo == 1 )
        {
          outletPixel.ldd_ = 64;
        }
        if( lineTo == lddMatrix_.GetLines() - 1 || lineTo == lddMatrix_.GetLines() - 2 )
        {
          outletPixel.ldd_ = 4;
        }

        if( columnTo == 0 ||  columnTo == 1 )
        {
          outletPixel.ldd_ = 16;
        }
        if( columnTo == lddMatrix_.GetColumns() - 1 || columnTo == lddMatrix_.GetColumns() - 2 )
        {
          outletPixel.ldd_ = 1;
        }
        
        return;
      }

      l = (int)(lineFrom / factor_);
      c = (int)(columnFrom / factor_);

      l = l - line + 1;
      c = c - column + 1;

      outletPixel.ldd_ = neighborDirection_[l][c];      
      return;
    }

    // check neighborhood border
    if( lineTo < neighborhoodBeginLine ||
      columnTo < neighborhoodBeginColumn || 
      lineTo > neighborhoodEndLine-1 ||
      columnTo > neighborhoodEndColumn-1 )
    {
      if( outletFounded == true )
      {
        outletPixel.ldd_ = neighborDirection_[l][c];
        return;
      }

      l = (int)(lineFrom / factor_);
      c = (int)(columnFrom / factor_);

      l = l - line + 1;
      c = c - column + 1;

      outletPixel.ldd_ = neighborDirection_[l][c];      
      return;
    }

    if( outletPixelsMatrix_[lineTo][columnTo] == 1 )
    {
      l = (int)(lineTo / factor_);
      c = (int)(columnTo / factor_);

      l = l - line + 1;
      c = c - column + 1;

      outletFounded = true;
      outletPixel.ldd_ = neighborDirection_[l][c];

      if( (accumulatedMatrix_[lineTo][columnTo] - outletPixel.accumulatedArea_) > at_ )
      {
        outletPixel.ldd_ = neighborDirection_[l][c];
        return;        
      }
    }

    lineFrom = lineTo;
    columnFrom = columnTo;
  }

  //outletPixel.ldd_ = neighborDirection_[l][c];
}

void
HidroUpscaleRaster::checkLooping( unsigned int line, unsigned int column )
{
  unsigned int nlines = scaledRaster_->params().nlines_;
  unsigned int ncolumns = scaledRaster_->params().ncols_;
  
  unsigned char ldd = scaledMatrix_[line][column];

  if( ldd == 0 || ldd == 255 )
  {
    return;
  }

  unsigned int lineTo = line + lineIncrement[ldd];
  unsigned int columnTo = column + columnIncrement[ldd];

  if( lineTo > nlines-1 || columnTo > ncolumns-1 )
  {
    return;
  }

  unsigned char lddTo = scaledMatrix_[lineTo][columnTo]; 

  if( (int)lineIncrement[ldd] + (int)lineIncrement[lddTo] == 0 )
  {
    if( columnIncrement[ldd] + columnIncrement[lddTo] == 0 )
    {
      // looping
     solveLooping( line, column, lineTo, columnTo );
    }
  }
}

void 
HidroUpscaleRaster::solveLooping( unsigned int line1, unsigned int column1, unsigned int line2, unsigned int column2 )
{
  // re find the outlet pixel
  OutletPixel outletPixel1 = reFindOutletPixel( line1, column1 );  
  OutletPixel outletPixel2 = reFindOutletPixel( line2, column2 );

  // save original mufp value
  int aux_mufp = mufp_;

  // zero mufp
  mufp_ = 0;

  // check less accumulated area
  unsigned int line;
  unsigned int column;
  OutletPixel outletPixel;

  if( outletPixel1.accumulatedArea_ < outletPixel2.accumulatedArea_ )
  {
    outletPixel = outletPixel1;
    line = line1;
    column = column1;    
  }
  else
  {
    outletPixel = outletPixel2;
    line = line2;
    column = column2;
  }

  // zero o outlet antigo
  outletPixelsMatrix_[outletPixel.line_][outletPixel.column_] = 0;

  // find the outletPixel with mufp = 0
  OutletPixel outletPixelNew = findOutletPixel( line, column );  

  // store outlet pixel
  if( outletPixelNew.ldd_ != 0 && outletPixelNew.ldd_ != 255 )
  {
    outletPixelsMatrix_[outletPixelNew.line_][outletPixelNew.column_] = 1;
  }
  scaledMatrix_[line][column] = outletPixelNew.ldd_;

  // Check the accumulated area threshold
  areaThreshold( line, column, outletPixelNew );

  scaledMatrix_[line][column] = outletPixelNew.ldd_;

  // restore mufp
  mufp_ = aux_mufp;
}


OutletPixel
HidroUpscaleRaster::reFindOutletPixel( unsigned int line, unsigned int column )
{
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines() - 1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns() - 1;

  // Lines
  for( unsigned int i = blockBeginColumn+1; i < blockEndColumn-1; i++ )
  {
    // fisrt line UP
    if( (int)outletPixelsMatrix_[blockBeginLine][i] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockBeginLine;
      newOutletPixel.ldd_ = 64;

      return newOutletPixel;
    }

    // last line DOWN
    if( (int)outletPixelsMatrix_[blockEndLine-1][i] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockEndLine-1;
      newOutletPixel.ldd_ = 4;

      return newOutletPixel;
    }
  }
  
  // Columns
  for( unsigned int i = blockBeginLine+1; i < blockEndLine-1; i++ )
  {
    // fisrt column LEFT
    if( (int)outletPixelsMatrix_[i][blockBeginColumn] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockBeginColumn];
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 16;

      return newOutletPixel;
    }

    // last column RIGHT
    if( (int)outletPixelsMatrix_[i][blockEndColumn-1] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockEndColumn-1];
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 1;

      return newOutletPixel;
    }
  }

  // Diagonals
  // up left
  if( (int)outletPixelsMatrix_[blockBeginLine][blockBeginColumn] == 1 )    
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockBeginColumn];
    newOutletPixel.column_ = blockBeginColumn;
    newOutletPixel.line_ = blockBeginLine;

    unsigned char lddResult = lddMatrix_[blockBeginLine][blockBeginColumn];
    if( (int)lddResult == 8 )
      lddResult = 16;
    if( (int)lddResult == 128 )
      lddResult = 64;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

    // up rigth
  if( (int)outletPixelsMatrix_[blockBeginLine][blockEndColumn-1] == 1 )    
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockEndColumn-1];
    newOutletPixel.column_ = blockEndColumn-1;
    newOutletPixel.line_ = blockBeginLine;

    unsigned char lddResult = lddMatrix_[blockBeginLine][blockEndColumn-1];
    if( (int)lddResult == 32 )
      lddResult = 64;
    if( (int)lddResult == 2 )
      lddResult = 1;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

    // down left    
  if( (int)outletPixelsMatrix_[blockEndLine-1][blockBeginColumn] == 1 )
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockBeginColumn];
    newOutletPixel.column_ = blockBeginColumn;
    newOutletPixel.line_ = blockEndLine-1;

    unsigned char lddResult = lddMatrix_[blockEndLine-1][blockBeginColumn];
    if( (int)lddResult == 32 )
      lddResult = 16;
    if( (int)lddResult == 2 )
      lddResult = 4;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }  

    // down rigth
  if( (int)outletPixelsMatrix_[blockEndLine-1][blockEndColumn-1] == 1 )
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockEndColumn-1];
    newOutletPixel.column_ = blockEndColumn-1;
    newOutletPixel.line_ = blockEndLine-1;

    unsigned char lddResult = lddMatrix_[blockEndLine-1][blockEndColumn-1];
    if( (int)lddResult == 8 )
      lddResult = 4;
    if( (int)lddResult == 128 )
      lddResult = 64;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

  OutletPixel failPixel;
  failPixel.accumulatedArea_ = 0;
  failPixel.line_ = 0;
  failPixel.column_ = 0;  
  failPixel.ldd_ = 0;
  return failPixel;
}

bool
OutletPixel::operator>( const OutletPixel &rightSide ) const
{
  return ( this->accumulatedArea_ < rightSide.accumulatedArea_ );
}

bool
OutletPixel::operator==( const OutletPixel &rightSide ) const
{
  return ( this->line_ == rightSide.line_ && this->column_ == rightSide.column_ );
}
