#include <HidroTreeClassify.h>
#include <HidroFlowUtils.h>

#include <HidroUtils.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>
#include <qstring.h>
#include <math.h>

#define D2R(a) (a*M_PI/180)

HidroTreeClassify::HidroTreeClassify( TeRaster* *Raster, TeRaster* lddRaster, TeRaster* &drainRaster, unsigned int* options ) :
HidroFlowAlgorithm( lddRaster ),
Raster_(Raster),
lddRaster_(lddRaster),
drainRaster_(drainRaster),
options_(options)
{
}

HidroTreeClassify::~HidroTreeClassify()
{
  // Free Memory
  lddMatrix_.clear();
  drainMatrix_.clear();
  varMatrix_.clear();
}

bool HidroTreeClassify::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();  

  // load data
  TeProgress::instance()->reset();
  TeProgress::instance()->setCaption("TerraHidro");

  // compute basic matrix
//  unsigned int nlines = RasterTmp->params().nlines_;
//  unsigned int ncolumns = RasterTmp->params().ncols_;  
  unsigned int nlines = Raster_[0]->params().nlines_;
  unsigned int ncolumns = Raster_[0]->params().ncols_;  
  varMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, varMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  drainMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned int>::AutoMemPol, drainMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  TePDIMatrix<unsigned int> ruleMatrix_;
  ruleMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned int>::AutoMemPol, drainMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
	drainMatrix_[line][0] = 0;
	drainMatrix_[line][ncolumns-1] = 0;
  }
  for( unsigned int column=0; column<ncolumns; column++ )
  {
    drainMatrix_[0][column] = 0;
	drainMatrix_[nlines-1][column] = 0;
  }

  //Applying the rules...
  TeProgress::instance()->setMessage("Load Slope data.");    
  if( !copyRaster2PDIMatrix( Raster_[0], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Slope data.");    
  TeProgress::instance()->setTotalSteps( nlines );
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    ruleMatrix_[line][column] = (varMatrix_[line][column] > 10.245379)?32:0;
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load Minimal Curvature data.");    
  if( !copyRaster2PDIMatrix( Raster_[1], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Minimal Curvature data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > -1.386735)
	    {
	      ruleMatrix_[line][column] += 4;
	      if(varMatrix_[line][column] > -0.925477)
	      {
	        ruleMatrix_[line][column] += 128;
	        if(varMatrix_[line][column] > -0.635774)
	        {
	          ruleMatrix_[line][column] += 512;
	        }
	      }
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load Horizontal Curvature data.");    
  if( !copyRaster2PDIMatrix( Raster_[2], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Horizontal Curvature data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > -1.871114)
	    {
	      ruleMatrix_[line][column] += 16;
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load Contributing Area data.");    
  if( !copyRaster2PDIMatrix( Raster_[3], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Contributing Area data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > 68822.5)
	    {
	      ruleMatrix_[line][column] += 2;
  	      if(varMatrix_[line][column] > 232291.3)
	      {
	        ruleMatrix_[line][column] += 1;
   	        if(varMatrix_[line][column] > 7088892)
	        {
	          ruleMatrix_[line][column] += 256;
	        }
	      }
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load Level Difference in Flow Direction (16 neighboors) data.");    
  if( !copyRaster2PDIMatrix( Raster_[4], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Level Difference (16 neighboors) data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > 65)
	    {
	      ruleMatrix_[line][column] += 8;
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load Level Difference in Flow Direction (64 neighboors) data.");    
  if( !copyRaster2PDIMatrix( Raster_[5], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to Level Difference (64 neighboors) data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > 98)
	    {
	      ruleMatrix_[line][column] += 64;
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  TeProgress::instance()->setMessage("Load HAND (MaxOrdStrahler = 5) data.");    
  if( !copyRaster2PDIMatrix( Raster_[6], varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  TeProgress::instance()->setMessage("Applying Rules related to HAND (MaxOrdStrahler = 5) data.");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if(varMatrix_[line][column] != -9999)
	  {
	    if(varMatrix_[line][column] > 19)
	    {
	      ruleMatrix_[line][column] += 1024;
	    }
	  }
	  else
	  {
		ruleMatrix_[line][column] = 0;
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  //Classifying...
  unsigned char leaf[12] = {3,1,1,3,3,3,2,2,3,3,2,3};
  if(options_[0] == 0)
  {
    for( unsigned int i=0; i<12; i++ ) 
	{
	  leaf[i] = (leaf[i] == 3)?0:1;
	}
  }
  else if(options_[0] == 2)
  {
    for( unsigned int i=0; i<12; i++ )
	{
	  leaf[i] = i+1;
	}
  }
  TeProgress::instance()->setMessage("Classifying...");    
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
	{
	  if((ruleMatrix_[line][column] % 2) == 0) //A <= 232291.5
	  {
	    if(((ruleMatrix_[line][column]/2) % 2) == 0) //A <= 68822.5
	    {
	      drainMatrix_[line][column] = leaf[0]; //F0 ND
	    }
		else //A > 68822.5
		{
	      if(((ruleMatrix_[line][column]/4) % 2) == 0) //B <= -1.386735
	      {
			if(((ruleMatrix_[line][column]/8) % 2) == 0) //C <= 65
			{
			  if(((ruleMatrix_[line][column]/16) % 2) == 0) //D <= 1.871114
			  {
			    drainMatrix_[line][column] = leaf[1]; //F1 N
			  }
			  else //D > 1.871114 
			  {
				if(((ruleMatrix_[line][column]/32) % 2) == 0) //E <= 10.245379
				{
				  drainMatrix_[line][column] = leaf[2]; //F2 N
				}
				else //E > 10.245379 
				{
				  drainMatrix_[line][column] = leaf[3]; //F3 ND
				}
			  }
			}
			else //C > 65
			{
			  drainMatrix_[line][column] = leaf[4]; //F4 ND
			}
	      }
		  else //B > -1.386735
		  {
			drainMatrix_[line][column] = leaf[5]; //F5 ND
		  }
		}
	  }
	  else //A > 232291.5
	  {
		if(((ruleMatrix_[line][column]/64) % 2) == 0) //F <= 98
		{
		  if(((ruleMatrix_[line][column]/128) % 2) == 0) //B <= -0.925477
		  {
		    drainMatrix_[line][column] = leaf[6]; //F6789 N e D
		  }
		  else //B > -0.925477 
		  {
			if(((ruleMatrix_[line][column]/256) % 2) == 0) //A <= 7088892
			{
			  if(((ruleMatrix_[line][column]/512) % 2) == 0) //B <= -0.635774
			  {
				if(((ruleMatrix_[line][column]/1024) % 2) == 0) //H <= 19
				{
				  drainMatrix_[line][column] = leaf[7]; //F10 D
				}
				else //H > 19
				{
				  drainMatrix_[line][column] = leaf[8]; //F11 ND
				}
			  }
			  else //B > -0.635774 
			  {
			    drainMatrix_[line][column] = leaf[9]; //F12 ND
			  }
			}
			else //A > 7088892 
			{
			  drainMatrix_[line][column] = leaf[10]; //F13 D
			}
		  }
		}
		else //F > 98
		{
		  drainMatrix_[line][column] = leaf[11]; //F14 D
		}
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  unsigned ordmax = 1;
  unsigned int lineFrom, columnFrom, lineTo, columnTo;
  if(options_[1])
  {
    // load LDD
	TeProgress::instance()->setMessage("Load LDD data.");    
    if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
	if(options_[2] != 0)
	{
	  //connecting gaps less or equal to options_[2]
      TeProgress::instance()->setMessage("Correcting Gaps...");
	  for( unsigned int line=0; line<nlines; line++ )
	  {
		for( unsigned int column=0; column<ncolumns; column++ )
		{
		  if((drainMatrix_[line][column] != 0) && (lddMatrix_[line][column] != 0) && (lddMatrix_[line][column] != 255))
		  {
			lineFrom = line;
			columnFrom = column;
			if(lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ))
			{
			  if(drainMatrix_[lineTo][columnTo] == 0)
			  {
				unsigned int k = 1;
			    vector <unsigned int> gapline, gapcolumn;
				lineFrom = lineTo;
				columnFrom = columnTo;
				gapline.push_back(lineFrom);
				gapcolumn.push_back(columnFrom);
				while (lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) && (k <= options_[2]))
				{
				  if(lddMatrix_[lineTo][columnTo] == 255)
				  {
					break;
				  }
				  if(drainMatrix_[lineTo][columnTo] != 0)
				  {
					k = 0;
					break;
				  }
				  lineFrom = lineTo;
				  columnFrom = columnTo;
				  gapline.push_back(lineFrom);
				  gapcolumn.push_back(columnFrom);
				  k++;
				}
				if(k == 0)
				{
				  for( unsigned int i=0; i<gapline.size(); i++ )
				  {
					drainMatrix_[gapline[i]][gapcolumn[i]] = 1;
				  }
				}
			  }
			}
		  }
        }
	  }
	}
	if(options_[3] != 0)
	{
      TePDIMatrix<unsigned char> tmpMatrix_;
      tmpMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, tmpMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
	  for( unsigned int line=0; line<nlines; line++ )
	  {
		for( unsigned int column=0; column<ncolumns; column++ )
		{
		  tmpMatrix_[line][column] = (drainMatrix_[line][column] != 0)?1:0;
		}
	  }
	  //eliminating segments less or equal to options_[3]
      TeProgress::instance()->setMessage("Eliminating the Isolated Segments...");
	  for( unsigned int line=1; line<nlines-1; line++ )
	  {
		for( unsigned int column=1; column<ncolumns-1; column++ )
		{
		  if(tmpMatrix_[line][column] == 1)
		  {
			tmpMatrix_[line][column] = 2;
			vector <unsigned int> segline, segcolumn;
			segline.push_back(line);
			segcolumn.push_back(column);
			unsigned int k = 0;
			while(k < segline.size())
			{
			  if(tmpMatrix_[segline[k]-1][segcolumn[k]-1] == 1)
			  {
			    tmpMatrix_[segline[k]-1][segcolumn[k]-1] = 2;
				segline.push_back(segline[k]-1);
				segcolumn.push_back(segcolumn[k]-1);
			  }
			  if(tmpMatrix_[segline[k]-1][segcolumn[k]] == 1)
			  {
			    tmpMatrix_[segline[k]-1][segcolumn[k]] = 2;
				segline.push_back(segline[k]-1);
				segcolumn.push_back(segcolumn[k]);
			  }
			  if(tmpMatrix_[segline[k]-1][segcolumn[k]+1] == 1)
			  {
			    tmpMatrix_[segline[k]-1][segcolumn[k]+1] = 2;
				segline.push_back(segline[k]-1);
				segcolumn.push_back(segcolumn[k]+1);
			  }
			  if(tmpMatrix_[segline[k]][segcolumn[k]-1] == 1)
			  {
			    tmpMatrix_[segline[k]][segcolumn[k]-1] = 2;
				segline.push_back(segline[k]);
				segcolumn.push_back(segcolumn[k]-1);
			  }
			  if(tmpMatrix_[segline[k]][segcolumn[k]+1] == 1)
			  {
			    tmpMatrix_[segline[k]][segcolumn[k]+1] = 2;
				segline.push_back(segline[k]);
				segcolumn.push_back(segcolumn[k]+1);
			  }
			  if(tmpMatrix_[segline[k]+1][segcolumn[k]-1] == 1)
			  {
			    tmpMatrix_[segline[k]+1][segcolumn[k]-1] = 2;
				segline.push_back(segline[k]+1);
				segcolumn.push_back(segcolumn[k]-1);
			  }
			  if(tmpMatrix_[segline[k]+1][segcolumn[k]] == 1)
			  {
			    tmpMatrix_[segline[k]+1][segcolumn[k]] = 2;
				segline.push_back(segline[k]+1);
				segcolumn.push_back(segcolumn[k]);
			  }
			  if(tmpMatrix_[segline[k]+1][segcolumn[k]+1] == 1)
			  {
			    tmpMatrix_[segline[k]+1][segcolumn[k]+1] = 2;
				segline.push_back(segline[k]+1);
				segcolumn.push_back(segcolumn[k]+1);
			  }
			  k++;
			}
			if( segline.size() <= options_[3] )
			{
			  while(segline.size() > 0)
			  {
			    drainMatrix_[segline.back()][segcolumn.back()] = 0;
			    segline.pop_back();
			    segcolumn.pop_back();
			  }
			}
		  }
		}
        TeProgress::instance()->setProgress( line );
        if( TeProgress::instance()->wasCancelled() )
        {
          return cancel();
        }
	  }
	  tmpMatrix_.clear();
	}


//	if(options_[4] != 2) //Strahler or Shreve
//	{
	TeProgress::instance()->setMessage("Connecting Isolated Segments...");
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
		if((drainMatrix_[line][column] != 0) && (lddMatrix_[line][column] != 0) && (lddMatrix_[line][column] != 255))
		{
          lineFrom = line;
          columnFrom = column;
		  while (lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ))
		  {
		    if((drainMatrix_[lineTo][columnTo] != 0) || (lddMatrix_[lineTo][columnTo] == 255))
			{
			  break;
			}
			drainMatrix_[lineTo][columnTo] = 1;
            lineFrom = lineTo;
            columnFrom = columnTo;
		  }
		}
	  }
      TeProgress::instance()->setProgress( line );
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
	if(options_[4] == 0) //Strahler
	{
      TeProgress::instance()->setMessage("Creating Strahler Order Raster 1/3...");
	  if( !findSprings() )
      {
        timeEnd_ = Time::instance().actualTimeString();
        timeTotal_ = Time::instance().partialString();
        return false;
      }
      TeProgress::instance()->setMessage("Creating Strahler Order Raster 2/3...");
	  // init connections matrix
	  if( !initConnections() )
	  {
		timeEnd_ = Time::instance().actualTimeString();
		timeTotal_ = Time::instance().partialString();
		return false;
	  }
      TeProgress::instance()->setMessage("Creating Strahler Order Raster 3/3...");
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {      
          if( connectionsMatrix_[line][column] == 0 )
          {
            lineFrom = line;
            columnFrom = column;
            while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
              if( lddMatrix_[lineTo][columnTo] == 255 )
              {
                break;
              }
			  if( (drainMatrix_[lineTo][columnTo] == drainMatrix_[lineFrom][columnFrom]) && (drainMatrix_[lineFrom][columnFrom] != 0) )
			  {
			    drainMatrix_[lineTo][columnTo]++;
				ordmax = max(ordmax,drainMatrix_[lineTo][columnTo]);
			  }
			  else if( drainMatrix_[lineTo][columnTo] < drainMatrix_[lineFrom][columnFrom] )
			  {
                drainMatrix_[lineTo][columnTo] = drainMatrix_[lineFrom][columnFrom]; 
			  }
              // if connections bigger than one
              if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
              {
                connectionsMatrix_[lineTo][columnTo]--;
                break;
              }
              lineFrom = lineTo;
              columnFrom = columnTo;
            }        
          }
        }
        // atualiza barra de progresso
        TeProgress::instance()->setProgress( line );
        // verifica se usurio cancelou a operao
        if( TeProgress::instance()->wasCancelled() )
        {
          return false;
        }
      }
	} 
	else if(options_[4] == 1)//Shreve
	{
      TeProgress::instance()->setMessage("Creating Shreve Order Raster 1/3...");
//      TeProgress::instance()->setProgress( 0 );
	  if( !findSprings() )
      {
        timeEnd_ = Time::instance().actualTimeString();
        timeTotal_ = Time::instance().partialString();
        return false;
      }
	  // init connections matrix
      TeProgress::instance()->setMessage("Creating Shreve Order Raster 2/3...");
	  if( !initConnections() )
	  {
		timeEnd_ = Time::instance().actualTimeString();
		timeTotal_ = Time::instance().partialString();
		return false;
	  }
      TeProgress::instance()->setMessage("Creating Shreve Order Raster 3/3...");
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {      
          if( connectionsMatrix_[line][column] == 0 )
          {
            lineFrom = line;
            columnFrom = column;
            while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
              if( lddMatrix_[lineTo][columnTo] == 255 )
              {
                break;
              }
			  if(drainMatrix_[lineFrom][columnFrom] != 0)
			  {
			    drainMatrix_[lineTo][columnTo] += drainMatrix_[lineFrom][columnFrom];
				ordmax = max(ordmax,drainMatrix_[lineTo][columnTo]);
			  }
              // if connections bigger than one
              if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
              {
                connectionsMatrix_[lineTo][columnTo]--;
                break;
              }
              lineFrom = lineTo;
              columnFrom = columnTo;
            }        
          }
        }
        // atualiza barra de progresso
        TeProgress::instance()->setProgress( line );
        // verifica se usurio cancelou a operao
        if( TeProgress::instance()->wasCancelled() )
        {
          return false;
        }
	  }
	}
//	}
  }
  TeRasterParams RasterParams = Raster_[0]->params();
  
  // Set dummy
  RasterParams.setDummy( 0 );
  RasterParams.useDummy_ = true;  

  // Change mode
  RasterParams.mode_ = 'w';
  RasterParams.decoderIdentifier_ = "SMARTMEM";
  if(ordmax < 256)
  {
    RasterParams.setDataType( TeUNSIGNEDCHAR );
  }
  else
  {
    RasterParams.setDataType( TeUNSIGNEDLONG );
  }
  RasterParams.nBands( 1 );

  // Set Max and Minimal values
  RasterParams.vmax_[0] = -TeMAXFLOAT;
  RasterParams.vmin_[0] =  TeMAXFLOAT;

  // Lut
  RasterParams.setPhotometric( TeRasterParams::TePallete );
  RasterParams.lutr_.clear();
  RasterParams.lutg_.clear();
  RasterParams.lutb_.clear();
  RasterParams.lutClassName_.clear();

  RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 0 );
  RasterParams.lutClassName_.push_back("");
  if(options_[0] == 0)
  {
	if((options_[4] == 2) || (options_[1] == 0))
	{
      RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 255 );
      RasterParams.lutClassName_.push_back("Drainage");
	}
	else
	{
	  for( unsigned int i=0; i<ordmax; i++ )
	  {
		unsigned char g = (unsigned char) (255*((float) i)/(ordmax-1));
        RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 255 - g ); RasterParams.lutb_.push_back( 255 );
        RasterParams.lutClassName_.push_back("Order "+TeAgnostic::to_string(i+1));
	  }
	}
  }
  else if(options_[0] == 1)
  {
    RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 255 ); RasterParams.lutb_.push_back( 255 );
    RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 255 );
    RasterParams.lutr_.push_back( 120 ); RasterParams.lutg_.push_back( 120 ); RasterParams.lutb_.push_back( 120 );
    RasterParams.lutClassName_.push_back("Spring");
    RasterParams.lutClassName_.push_back("Drainage");
    RasterParams.lutClassName_.push_back("Non Drainage");
  }
  else
  {
    RasterParams.lutr_.push_back( 255 ); RasterParams.lutg_.push_back( 200 ); RasterParams.lutb_.push_back( 200 );
    RasterParams.lutr_.push_back( 100 ); RasterParams.lutg_.push_back( 255 ); RasterParams.lutb_.push_back( 255 );
    RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 210 ); RasterParams.lutb_.push_back( 210 );
    RasterParams.lutr_.push_back( 255 ); RasterParams.lutg_.push_back( 150 ); RasterParams.lutb_.push_back( 150 );
    RasterParams.lutr_.push_back( 255 ); RasterParams.lutg_.push_back( 100 ); RasterParams.lutb_.push_back( 100 );
    RasterParams.lutr_.push_back( 255 ); RasterParams.lutg_.push_back( 50 ); RasterParams.lutb_.push_back( 50 );
    RasterParams.lutr_.push_back( 60 ); RasterParams.lutg_.push_back( 60 ); RasterParams.lutb_.push_back( 255 );
    RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 255 );
    RasterParams.lutr_.push_back( 255 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 0 );
    RasterParams.lutr_.push_back( 200 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 0 );
    RasterParams.lutr_.push_back( 0 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 146 );
    RasterParams.lutr_.push_back( 150 ); RasterParams.lutg_.push_back( 0 ); RasterParams.lutb_.push_back( 0 );
    RasterParams.lutClassName_.push_back("F0");
    RasterParams.lutClassName_.push_back("F1");
    RasterParams.lutClassName_.push_back("F2");
    RasterParams.lutClassName_.push_back("F3");
    RasterParams.lutClassName_.push_back("F4");
    RasterParams.lutClassName_.push_back("F5");
    RasterParams.lutClassName_.push_back("F6789");
    RasterParams.lutClassName_.push_back("F10");
    RasterParams.lutClassName_.push_back("F11");
    RasterParams.lutClassName_.push_back("F12");
    RasterParams.lutClassName_.push_back("F13");
    RasterParams.lutClassName_.push_back("F14");
  }

  // create the raster
  drainRaster_ = new TeRaster( RasterParams );
  if( !drainRaster_->init() )
  {
    errorMessage_ = drainRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Saving Drainage Grid...");    
  if( !copyPDIMatrix2Raster( drainMatrix_, drainRaster_ ) )
  {
    return cancel();
  }

  // Free Memory
  lddMatrix_.clear();
  drainMatrix_.clear();
  varMatrix_.clear();
  ruleMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroTreeClassify::cancel()
{
  // Free Memory
  lddMatrix_.clear();
  drainMatrix_.clear();
  varMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}

/*bool HidroTreeClassify::findSprings()
{
  unsigned int nlines = Raster_[0]->params().nlines_;
  unsigned int ncolumns = Raster_[0]->params().ncols_;  
  // init connections matrix
  if( !initConnections() )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }
	  // looking for springs
  unsigned int lineFrom, columnFrom, lineTo, columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineFrom = line;
        columnFrom = column;
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {
		  if( (drainMatrix_[lineTo][columnTo] == 1) && (drainMatrix_[lineFrom][columnFrom] == 0) )
		  {
            lineFrom = lineTo;
            columnFrom = columnTo;
            while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
			  drainMatrix_[lineTo][columnTo] = 0;
              lineFrom = lineTo;
              columnFrom = columnTo;
			}
		  }
          // if connections bigger than one
          if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
          {
            connectionsMatrix_[lineTo][columnTo]--;
            break;
          }
          lineFrom = lineTo;
          columnFrom = columnTo;
        }        
      }
    }
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return false;
    }
  }
  return true;
}
*/

bool HidroTreeClassify::findSprings()
{
  unsigned int nlines = Raster_[0]->params().nlines_;
  unsigned int ncolumns = Raster_[0]->params().ncols_; 
  // looking for springs
  unsigned int lineFrom, columnFrom, lineTo, columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
	  if(drainMatrix_[line][column] == 1)
	  {
        lineFrom = line;
        columnFrom = column;
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo) )
        {
		  if(drainMatrix_[lineTo][columnTo] != 1)
		  {
			break;
		  }
		  drainMatrix_[lineTo][columnTo] = 0;
          lineFrom = lineTo;
          columnFrom = columnTo;
		}
	  }
	}
    TeProgress::instance()->setProgress( line );
    if( TeProgress::instance()->wasCancelled() )
    {
      return false;
    }
  }
  return true;
}