/******************************************************************************
* Hidro development team
* 
* Terra View Hidro Plugin
* 
* @(#) HidroBoostUIDijkstra.h
*
*******************************************************************************
*
* $Rev: 8870 $:
*
* $Author: alexcj $:
*
* $Date: 2010-08-30 09:18:11 -0300 (seg, 30 ago 2010) $:
*
******************************************************************************/

/*!
	\brief This is the Hidro Interfaces Group.
	@defgroup hidroInterface The Group for Plugin Interaction.
*/

/** 
  * \file HidroBoostUIDijkstra.ui.h
  *
  * \class HidroBoostUIDijkstra
  *
  * \brief This file is a interface generated by qt designer
  *
  * This interface defines the Boost algorithm Dijkstra.
  *		To execute this operation is necessary:
  *												select a hidro graph.
  *												define a edge cost attribute.
  *												select a start vertex
  *												select a end vertex (optional)
  *												define output hidro graph name
  *		
  * \sa HidroMainWindow.ui.h
  *
  * \author Eric Silva Abreu <eric.abreu@funcate.org.br>
  * \version 1.0
  *
/*!
  @ingroup hidroInterface
 */

/*
** ----------------------------------------------------------------------------
** Includes:
*/

#include <HidroUtils.h>
#include <HidroPersister.h>
#include <HidroMetadata.h>
#include <HidroBoostUtils.h>
#include <HidroBoostParams.h>
#include <HidroBoostAlgorithmFactory.h>

#include <qmessagebox.h>

#include <TeQtCanvas.h>
#include <TeAppTheme.h>

/** \brief HidroBoostUIConnectedComponents init function.
    * Used to replace de default constructor from interface;
	* \param parameters		Input parameter used to set the parameters from app
	* \return				True if the input parameters are valids and False in other case.
    */
bool HidroBoostUIDijkstra::init( PluginParameters* parameters )
{
    if(parameters == NULL)
	{
		return false;
	}

	_parameters = parameters;

//reset tools used to get vertex from canvas app
	_getStarVertex = false;
	_getEndVertex  = false;

	return true;
}

/** \brief HidroBoostUIDijkstra set graph name function.
    * Used to set the graph name selected by user to apply this algorithm
	* \param graphName		Input parameter used to define the hidro graph selected
    */
void HidroBoostUIDijkstra::setGraphName( const std::string & graphName )
{
	if(!graphName.empty())
	{
		graphNameLineEdit->setText(graphName.c_str());
	}

	if(_parameters && _parameters->getCurrentDatabasePtr())
	{
		HidroMetadata metadata(_parameters->getCurrentDatabasePtr());

		std::vector<std::string> edgeCosts = metadata.getGraphCostsAttributes(graphName);

		edgeCostComboBox->clear();
		edgeCostComboBox->insertItem("");

		for(unsigned int i = 0; i < edgeCosts.size(); ++i)
		{
			edgeCostComboBox->insertItem(edgeCosts[i].c_str());
		}
	}
}

/** \brief HidroBoostUIDijkstra Select Start Vertex by Click.
    * Used to define the  starter vertex that will be used on boost algorithm
	* Vertex defined by click on canvas app
    */
void HidroBoostUIDijkstra::startVertexByClickRadioButton_clicked()
{
	_getStarVertex = true;
	_getEndVertex  = false;

//connect terraView clicked
	connect(_parameters->teqtcanvas_ptr_, SIGNAL(mousePressed(TeCoord2D&, int, QPoint&)), this, SLOT(getMouseClicked(TeCoord2D&, int, QPoint&)));
}

/** \brief HidroBoostUIDijkstra Select Start Vertex by identification.
    * Used to define the  starter vertex that will be used on boost algorithm
	* Vertex defined by identification on interface
    */
void HidroBoostUIDijkstra::startVertexByIdentRadioButton_clicked()
{
	_getStarVertex = false;
	_getEndVertex  = false;

	disconnect(_parameters->teqtcanvas_ptr_, SIGNAL(mousePressed(TeCoord2D&, int, QPoint&)), this, SLOT(getMouseClicked(TeCoord2D&, int, QPoint&)));
}

/** \brief HidroBoostUIDijkstra Select End Vertex by Click.
    * Used to define the  ended vertex that will be used on boost algorithm
	* Vertex defined by click on canvas app
    */
void HidroBoostUIDijkstra::endVertexByClickRadioButton_clicked()
{
	_getStarVertex = false;
	_getEndVertex  = true;

//connect terraView clicked
	connect(_parameters->teqtcanvas_ptr_, SIGNAL(mousePressed(TeCoord2D&, int, QPoint&)), this, SLOT(getMouseClicked(TeCoord2D&, int, QPoint&)));
}

/** \brief HidroBoostUIDijkstra Select End Vertex by identification.
    * Used to define the  ended vertex that will be used on boost algorithm
	* Vertex defined by identification on interface
    */
void HidroBoostUIDijkstra::endVertexByIdentRadioButton_clicked()
{
	_getStarVertex = false;
	_getEndVertex  = false;

	disconnect(_parameters->teqtcanvas_ptr_, SIGNAL(mousePressed(TeCoord2D&, int, QPoint&)), this, SLOT(getMouseClicked(TeCoord2D&, int, QPoint&)));
}

/** \brief HidroBoostUIDijkstra Ok function.
    * Used to execute the operation.
    */
void HidroBoostUIDijkstra::okPushButton_clicked()
{
	if(graphNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Graph name not defined."));
		return;
	}

	if(edgeCostComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Edge Cost not defined."));
		return;
	}

	if(startVertexIdentLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Start vertex not defined."));
		return;
	}

	if(outputGraphNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output graph name not defined."));
		return;
	}
	

//load graph from database
	HidroGraph graph;

	HidroPersister persister(_parameters->getCurrentDatabasePtr());
	if(!persister.loadGraphFromDatabase(graphNameLineEdit->text().latin1(), graph))
	{
		QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
		return;
	}

//new graph
	HidroGraph outGraph(outputGraphNameLineEdit->text().latin1());

//execute algorithm
	HidroBoostParams pBoost;
	pBoost._boostAlgorithmName	= "Dijkstra Shortest Path";
	pBoost._inputHidroGraph		= &graph;
	pBoost._outputHidroGraph	= &outGraph;
	pBoost._edgeCostAttribute	= edgeCostComboBox->currentText().latin1();
	pBoost._startVertexId		= startVertexIdentLineEdit->text().latin1();

	if(!endVertexIdentLineEdit->text().isEmpty())
	{
		pBoost._endVertexId		= endVertexIdentLineEdit->text().latin1();
	}

	HidroBoostAlgorithms* aBoost = HidroBoostAlgorithmFactory::make(pBoost);

	if(aBoost)
	{
		if(!aBoost->executeAlgorithm())
		{
			std::string errorMessage = aBoost->_errorMessage;
			QMessageBox::warning(this, tr("Warning"), errorMessage.c_str());

			delete aBoost;

			return;
		}
	}
	else
	{
		QMessageBox::warning(this, tr("Warning"), tr("Boost Algorithm not found."));
		return;
	}

	if(outGraph.isGraphValid())
	{
		HidroMetadata metadata(_parameters->getCurrentDatabasePtr());

		TeProjection* proj = metadata.getGraphProjection(graph.getGraphName());

		double resolution = metadata.getGraphResolution(graph.getGraphName());

//save graph in database
		HidroPersister persister(_parameters->getCurrentDatabasePtr());

		if(!persister.saveGraphInDatabase(outGraph, proj, resolution))
		{
			QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
			return;
		}

		if(!persister.updateGraphAttributes(&graph, &outGraph))
		{
			QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
			return;
		}

//preview terralib objects
		std::string mess = "Do you really want to create graph view? \n This operation can take a few minutes.";
		int res = QMessageBox::question(this, tr("Create View"), tr(mess.c_str()), tr("&Yes"), tr("&No"));

		if(res == 0)
		{
			if(!persister.createGraphView(&outGraph, proj))
			{
				QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
				return;
			}

			_parameters->updateTVInterface();
		}
	}

	
	QMessageBox::warning(this, tr("Warning"), tr("Algorithm executed."));

	accept();
}

/** \brief HidroBoostUIDijkstra getMouseClicked function.
    * Function used to get the mouse clicked over the canvas.
	* This signal is emit from application, so we have to connect this
	* signal with this function.
	* \param coord		Input parameter, used to define the geo coord from click (this coord is in canvas projection)
	* \param state		Input parameter, used to indicate the mouse button state
	* \param point		Input parameter, used to define the canvas coord from click
    */
void HidroBoostUIDijkstra::getMouseClicked( TeCoord2D & coord, int /*state*/, QPoint & /*point*/ )
{
	HidroUtils utils(_parameters->getCurrentDatabasePtr());
	
	std::string graphName		= graphNameLineEdit->text().latin1();
	TeTheme* currentTheme		= (TeTheme*)_parameters->getCurrentThemeAppPtr()->getTheme();
	TeProjection* projCanvas	= _parameters->teqtcanvas_ptr_->projection();
	double pixelSize			= _parameters->teqtcanvas_ptr_->pixelSize();

	if(_getStarVertex)
	{
		std::string value = utils.getPointName(graphName, currentTheme, projCanvas, pixelSize, coord);
		startVertexIdentLineEdit->setText(value.c_str());
	}
	else if(_getEndVertex)
	{
		std::string value = utils.getPointName(graphName, currentTheme, projCanvas, pixelSize, coord);
		endVertexIdentLineEdit->setText(value.c_str());
	}

	_getStarVertex = false;
	_getEndVertex  = false;

	disconnect(_parameters->teqtcanvas_ptr_, SIGNAL(mousePressed(TeCoord2D&, int, QPoint&)), this, SLOT(getMouseClicked(TeCoord2D&, int, QPoint&)));	
}

/*
** ----------------------------------------------------------------------------
** End:
*/
