#include <HidroConsistentDEM.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

#include <vector>

HidroConsistentDEM::HidroConsistentDEM(TeRaster* lddRaster,
    TeRaster* demRaster,    
    TeRaster* &outputDEMRaster ) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    demRaster_(demRaster),    
    outputDEMRaster_(outputDEMRaster)    
{
}

HidroConsistentDEM::~HidroConsistentDEM()
{  
}

bool HidroConsistentDEM::cancel()
{
  return HidroFlowAlgorithm::cancel();
}

bool
HidroConsistentDEM::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( demRaster_->params().nlines_ != lddRaster_->params().nlines_ || demRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "DEM lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // init connections matrix
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Consistent DEM");
	TeProgress::instance()->setMessage("Step 1 from x");    
  unsigned int nlines = lddRaster_->params().nlines_;
  unsigned int ncolumns = lddRaster_->params().ncols_;

  TeProgress::instance()->setTotalSteps( nlines * 2 );

  // init connection matrix with zeros and dummys (255)  
  connectionsMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, connectionsMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );

  double ldd;

  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      lddRaster_->getElement( column, line, ldd );
      if( ldd != 255 )
      {
        connectionsMatrix_[line][column] = 0;
      }
      else
      {
        connectionsMatrix_[line][column] = 255;
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  
  // calculate connection matrix

  // create output DEM raster
  
  // output DEM params
  TeRasterParams outputDEMRasterParams = demRaster_->params();

  // Set data type
  outputDEMRasterParams.setDataType( TeFLOAT );
  
  // Set dummy
  outputDEMRasterParams.setDummy( -9999 );
  outputDEMRasterParams.useDummy_ = true;  

  // Change mode
  outputDEMRasterParams.mode_ = 'w';
  outputDEMRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  outputDEMRasterParams.vmax_[0] = -TeMAXFLOAT;
  outputDEMRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  outputDEMRaster_ = new TeRaster( outputDEMRasterParams );

  // verify if outputDEMRaster created is valid
  if( !outputDEMRaster_->init() )
  {
    errorMessage_ = outputDEMRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  double altimetria;
  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      
      demRaster_->getElement( column, line, altimetria );
      outputDEMRaster_->setElement(  column, line, altimetria );

      lddRaster_->getElement( column, line, ldd );
      if( ldd != 255 && ldd != 0 )
      {
        lineTo = line + lineIncrement[(unsigned char)ldd];
        columnTo = column + columnIncrement[(unsigned char)ldd];
        connectionsMatrix_[lineTo][columnTo]++;        
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( nlines + line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }       
  
  // create new one DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Consistent DEM");
  TeProgress::instance()->setMessage("Step 2 from x");
  TeProgress::instance()->setTotalSteps( nlines );  
  
  double altimetriaFrom;
  double altimetriaTo;
  unsigned int lineFrom;
  unsigned int columnFrom;

  for( unsigned int line = 0; line < nlines; line++ )
  {
    for( unsigned int column = 0; column < ncolumns; column++ )
    {
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineFrom = line;
        columnFrom = column;

        bool stop = false;
        
        while( connectionsMatrix_[lineFrom][columnFrom] < 2 && !stop )
        {
          lddRaster_->getElement( columnFrom, lineFrom, ldd );
          if( ldd != 255 &&  ldd != 0 )
          {
            lineTo = lineFrom + lineIncrement[(unsigned char)ldd];
            columnTo = columnFrom + columnIncrement[(unsigned char)ldd];
            outputDEMRaster_->getElement( columnFrom, lineFrom, altimetriaFrom );
            outputDEMRaster_->getElement( columnTo, lineTo, altimetriaTo );
            if( altimetriaFrom <= altimetriaTo )
            {
              altimetriaTo = altimetriaFrom - MINIMUM_DEM_DIFFERENCE;
            }
            outputDEMRaster_->setElement(  columnTo, lineTo, altimetriaTo );
            lineFrom = lineTo;
            columnFrom = columnTo;
          }
          else
          {
            stop = true;
          }          
        }

        if( connectionsMatrix_[lineFrom][columnFrom] > 1 )
        {
          connectionsMatrix_[lineFrom][columnFrom]--;
        }
      }

      // refresh progress bar    
      TeProgress::instance()->setProgress( line );
    
      // check for cancel
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
    }
  }  
  
  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}
