#include <HidroFixPFS.h>

// Hidro Includes
#include <HidroFlowGrid.h>
#include <HidroPFS.h>
#include <HidroFlowD8.h>
#include <HidroFlowUtils.h>
#include <HidroGenerateQueueStatistics.h>
#include <queuex.hpp>

// TerraLib Includes
#include <TeUtils.h>

// Qt Includes
#include <qmessagebox.h>

#define GENERATE_FPS_STATISTICS

bool
HidroFixPFS::pfs()
{
  // Time Statistics
  Time::instance().start();
  starTime_ = Time::instance().actualTimeString();

  // Initialize the pits queue
  //initPitQueue();

  // Eliminar outliers
  if( removeOutliers_ )
  {
     initPitQueue();
     removingDEMoutliers();         
  }

  // Solve plane areas (pits river)
  if( cavePlaneAreas_ )
  {
    fillPlaneAreas();
  }
 
  // Initialize the pits queue
  initPitQueue();
  initialSize_ = pitQueue_.size();

  // Solve the simple pits
  if( singlePits_ )
  {
    solveSimplePits();
  }
  
  // Queue Statistics
#ifdef GENERATE_FPS_STATISTICS
  HidroGenerateQueueStatistics::instance().start( queueStatisticsFileName_, initialSize_ );
#endif

  // Progress Bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Flow Extraction step 2 from 4.");
  TeProgress::instance()->setTotalSteps( initialSize_ );

  // Initialize the PFS algorithm
  HidroPFS pfs( demMatrix_, hasDummy_, demDummy_ );
  
  // Para cada fosso
  while( !pitQueue_.empty() )
  {
    // retrieve the line and column of first pit 
    int *pitCoord = pitQueue_.front();
    unsigned int x = pitCoord[0];
    unsigned int y = pitCoord[1];    
    pitQueue_.pop();

    unsigned char lddVal;
    lddVal = lddMatrix_[y][x];
    
    // while pit wans't solved
    while( lddVal == 0 )
    {
      // sove the pit using the PFS algorithm.
      pfs.solvePit( x, y );
            
      // recomput the LDD for changed cells and neighboors      
      for( unsigned int i=0; i < pfs.solution_path_.size(); i++ )
      {
        recomputeNeighborhoodLDD( pfs.solution_path_[i].first, pfs.solution_path_[i].second );      
      }

      // check if it keep a pit
      lddVal = lddMatrix_[y][x];
      if( lddVal == 0 )
      {
        // probably a precission error (zero computational)         
        demMatrix_[y][x] = demMatrix_[y][x] - double_problem_increment_;            
        double_problem_++;          
      }
      else
      {
        // FPS Statistics
        solvedPitsNumber_++;
        path_length_mean_ += pfs.solution_path_.size();        

        if( pfs.solution_path_.size() > big_path_ )
          big_path_ = pfs.solution_path_.size();
      }
      
      // check pits in the path
      for( unsigned int i=0; i<pfs.solution_path_.size(); i++ )
      {          
        // Compute the start and end of window (3x3)
        unsigned int startColumn = pfs.solution_path_[i].first - 1;
        unsigned int endColumn = pfs.solution_path_[i].first + 2;
        unsigned int startLine = pfs.solution_path_[i].second - 1;
        unsigned int endLine = pfs.solution_path_[i].second + 2;

        if( pfs.solution_path_[i].first < 2 )
          startColumn = 1;
        if( endColumn > rasterColumns_-1 )
          endColumn = rasterColumns_-1;
        if( pfs.solution_path_[i].second < 2 )
          startLine = 1;
        if( endLine > rasterLines_-1 )
          endLine = rasterLines_-1;          

	      for( unsigned int line = startLine; line < endLine; ++line )
	      {
		      for( unsigned int column = startColumn; column < endColumn; ++column )
		      {
            if( lddMatrix_[line][column] == 0 )
            {
              if( !boundaryFixRegion( column, line) )
                pitQueue_.newPit( column, line );
                pits_in_path_++;
            }
          }          
        }
      }      
    }    

    // Update Progress
    TeProgress::instance()->setProgress( initialSize_ - pitQueue_.size() );

    // Storage the partial results hour by hour
    if( Time::instance().hasPassedOneHouer() )    
    {
      // Record the parcial results
      if( partialResult_ )
      {        
        /*std::string partial = outputDirectory_;
        partial.append( "\\" );
        partial.append( flowGrid_->partialResultName_ );
        partial.append( ".tif" );
        //demMatrix_.saveTiff( partial, TeFLOAT );*/
        flowGrid_->saveTiff();
      }
      #ifdef GENERATE_FPS_STATISTICS
        // Update the Queue Statistics File.
        HidroGenerateQueueStatistics::instance().increment( pitQueue_.size() );
      #endif
    }

    // The user cancel the proccess
    if( TeProgress::instance()->wasCancelled() )
    {
      unsolvedPitsNumber_ += pitQueue_.size();        
      QMessageBox::warning(0, "Warning", "The operation fix LDD was cancelled by the user.");
      // Use insted return to save the results.
      break;
    }   
  }

  // FPS Statistics  
  path_length_mean_ = path_length_mean_ / solvedPitsNumber_;
  visited_number_mean_ = visited_number_mean_ / solvedPitsNumber_;
  border_path_gradient_ = BORDER_PATH_GRADIENT_;  

  // Finishi progress bar and statistics
  TeProgress::instance()->reset();

  // Time Statistics
  endTime_ = Time::instance().actualTimeString();
  partialTime_ = Time::instance().partialString();
  #ifdef GENERATE_FPS_STATISTICS
    print();
    // Queue Statistics
    HidroGenerateQueueStatistics::instance().finsh( pitQueue_.size() );
  #endif

  return true;
}


void
HidroFixPFS::fillPlaneAreas()
{
  // Fila das celulas planas
  HidroSmartQueue planeQueue_;

  // Initialize the progress window
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
  TeProgress::instance()->setMessage("Initialize plane areas.");
  TeProgress::instance()->setTotalSteps( rasterLines_ - 2 );
  TeProgress::instance()->setProgress( 0 );
  double centerAltimetria;
  double neighboorAltimetria;
  for( unsigned int line = 1; line < rasterLines_-1; ++line )
	{
		for( unsigned int column = 1; column < rasterColumns_-1; ++column )
    {
      if( pitsNeighbors( column, line ) > 7 )
      {
        lddMatrix_[line][column] = 9;
        
        centerAltimetria = demMatrix_[line][column];
        // Janela 5x5        
        unsigned int startColumn = column - 2;
        unsigned int endColumn = column + 3;
        unsigned int startLine = line - 2;
        unsigned int endLine = line + 3;
        // Check the borders
        if( column < 2 )
          startColumn = 0;
        if( endColumn > rasterColumns_ )
          endColumn = rasterColumns_;
        if( startLine < 2 )
          startLine = 0;
        if( endLine > rasterLines_ )
          endLine = rasterLines_;
	      for( unsigned int l = startLine; l < endLine; ++l )
	      {
		      for( unsigned int c = startColumn; c < endColumn; ++c )
          {
            neighboorAltimetria = demMatrix_[l][c];
            if( neighboorAltimetria == centerAltimetria )
            {              
              if( !boundaryRegion( c, l ) )
              {
                lddMatrix_[l][c] = 9;
                planeQueue_.newPit( c, l );
              }
            }
          }
        }
      }
    }
    TeProgress::instance()->setProgress( line );
  }
  // Finalize the progress window
  TeProgress::instance()->reset();

  // Initialize the progress window
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
  TeProgress::instance()->setMessage("Caving plane Areas.");
  TeProgress::instance()->setTotalSteps( planeQueue_.size() );
  TeProgress::instance()->setProgress( 0 );

  // Fila da borda das areas planas  
  //std::vector<BreachingNode> borderQueue_; //  por isso que cai!!!
  queuex<BreachingNode> borderQueue_;

  unsigned int counter = 0;  
  double decrease = 0;

  while( !planeQueue_.empty() )
  {
    // Varro a fila inteira procurando pelas bordas
    // Acho as bordas e calculo a nova altimetria
    unsigned int planeQueueSize = planeQueue_.size();
    for( unsigned int count=0; count<planeQueueSize; count++ )
    {
      // retrieve the line and column of first plane cell 
      int *pitCoord = planeQueue_.front();
      unsigned int x = pitCoord[0];
      unsigned int y = pitCoord[1];
      planeQueue_.pop();

      if( isPlaneBorder( x, y ) )
      {
        // Se for borda adiciona na fila de borda
        BreachingNode border;
        border.x_ = x;
        border.y_ = y;        
        border.altimetria_ = demMatrix_[y][x] - decrease;
        borderQueue_.push( border );
        
        counter++;
        TeProgress::instance()->setProgress( counter );
      }
      else
      {
        // Se nao adiciona na fila de planos novamente
        planeQueue_.newPit( x, y );
      }
    }
    decrease = decrease + caving_increment_;

    // Altero o DEM e recalculo o LDD da vizinhana
    while( !borderQueue_.empty() )
    {
      // retrieve the line and column of first border cell 
      BreachingNode border = borderQueue_.front();
      unsigned int x = border.x_;
      unsigned int y = border.y_;
      borderQueue_.pop();
      
      // atualizo o DEM com a media das altimetrias (3x3) excluindo o centro
      demMatrix_[y][x] = border.altimetria_;

      // Marco para calcular novamente o LDD      
      lddMatrix_[y][x] = 11;
    }    
  }
  // Finalize the progress window
  TeProgress::instance()->reset();

  //*/ zera o ldd dnovo
  for( unsigned int line = 1; line < rasterLines_-1; ++line )
	{
		for( unsigned int column = 1; column < rasterColumns_-1; ++column )
    {
      if( lddMatrix_[line][column] == 11 )
        recomputeNeighborhoodLDD( column, line );        
    }
  }
  // Bordas
  for( unsigned int column = 0; column < rasterColumns_; ++column )
  {
    lddMatrix_[0][column] = 255;
    lddMatrix_[rasterLines_-1][column] = 255;    
  }
  for( unsigned int line = 0; line < rasterLines_; ++line )
  {
    lddMatrix_[line][0] = 255;
    lddMatrix_[line][rasterColumns_-1] = 255;    
  }
  /**/
}

void
HidroFixPFS::solveSimplePits()
{
  // Initialize the progress window
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
  TeProgress::instance()->setMessage("Simple Pits.");
  TeProgress::instance()->setTotalSteps( initialSize_ );
  TeProgress::instance()->setProgress( 0 );

  for( unsigned int counter = 0; counter < initialSize_; counter++ )
  {
    // retrieve the line and column of first pit 
    int *pitCoord = pitQueue_.front();
    unsigned int x = pitCoord[0];
    unsigned int y = pitCoord[1];    
    pitQueue_.pop();

    // Guardo a altimetria antiga
    double oldAltimetria = demMatrix_[y][x];

    // change the DEM altimetria with mean altimetria
    //double meanAltimetria = meanNeigbhors( x, y, 3, false ); //trocado pelo incremento minimo
    double meanAltimetria = smallerNeighbor( y, x ) + 0.0001;
    demMatrix_[y][x] = meanAltimetria;

    recomputeNeighborhoodLDD( x, y );

    // verifico se algum vizinho  fosso
    bool anyPit = false;
    // Compute the start and end of window (3x3)
    unsigned int startColumn = x - 1;
    unsigned int endColumn = x + 2;
    unsigned int startLine = y - 1;
    unsigned int endLine = y + 2;
	  for( unsigned int line = startLine; line < endLine; ++line )
	  {
		  for( unsigned int column = startColumn; column < endColumn; ++column )
      {        
        if( lddMatrix_[line][column] == 0 )
        {
          anyPit = true;
        }       
      }
    }
        
    if( anyPit )
    {
      //if( ( meanAltimetria - oldAltimetria ) < thresholdAltimetria_ )
      {
        // volto ao que era antes
        demMatrix_[y][x] = oldAltimetria;        
      }
      
      recomputeNeighborhoodLDD( x, y );
      pitQueue_.newPit( x, y );
    }
    else
    {
      // pit solved
      simplePits_++;
    }

    TeProgress::instance()->setProgress( counter );
  }
}


HidroFixPFS::HidroFixPFS( HidroFlowGrid* flowGrid,
    TePDIMatrix<double>& demMatrix,
    TePDIMatrix<unsigned char>& lddMatrix,    
    std::string demName,
    std::string lddName,
    bool cavePlaneAreas,
    bool singlePits,
    bool partialResult,
    bool removeOutliers,
    double lowOutlierThreshold,
    std::string outputDirectory) :
      flowGrid_(flowGrid),
      demMatrix_(demMatrix),
      lddMatrix_(lddMatrix),      
      caving_increment_(0.0001), // 10-6 0.000001 // 10-4 0.0001
      double_problem_increment_(0.01),
      solvedPitsNumber_(0),
      addPitsNumber_(0),
      unsolvedPitsNumber_(0),
      simplePits_(0),
      demName_(demName),
      lddName_(lddName),
      cavePlaneAreas_(cavePlaneAreas),
      singlePits_(singlePits),
      partialResult_(partialResult),
      removeOutliers_(removeOutliers),
      lowOutlierThreshold_(lowOutlierThreshold),
      outputDirectory_(outputDirectory),
      path_length_mean_(0.0),
      visited_number_mean_(0.0),
      pits_solved_(0),
      pits_solved_order_(0),
      big_path_(0),
      big_path_number_(0),
      double_problem_(0),
      pits_in_path_(0),
      unsolved_double_problem_(0),
      thresholdAltimetria_(45)
{
  rasterLines_ = demMatrix.GetLines();
  rasterColumns_ = demMatrix.GetColumns();
  demDummy_ = flowGrid_->demDummy_;
  hasDummy_ = flowGrid_->demRaster_->params().useDummy_;

  demName_ = flowGrid_->partialResultName_;
  outputDirectory_ = flowGrid_->outputDirectory_;
  lddName_ = demName_;  
  queueStatisticsFileName_ = outputDirectory_ + "/" + lddName_ + "_Queue_Size_Stats.txt";
  pitCorrectionStatisticsFileName_ = outputDirectory_+ "/" + lddName_ + "_Pit_Correction_Stats.txt";
}


HidroFixPFS::~HidroFixPFS()
{
}


bool
HidroFixPFS::initPitQueue()
{
  // Initialize the progress window
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
  TeProgress::instance()->setMessage("Initialize main pit queue.");
  TeProgress::instance()->setTotalSteps( rasterLines_-4 );
  TeProgress::instance()->setProgress( 0 );  

  for( unsigned int line = 2; line < rasterLines_-2; ++line )
	{
		for( unsigned int column = 2; column < rasterColumns_-2; ++column )
    {      
      if( lddMatrix_[line][column] == 0 )
      {
        if( !boundaryFixRegion( column, line ) )
        {
          pitQueue_.newPit( column, line );         
        }
      }
      TeProgress::instance()->setProgress( line );
    }
  }  

  // Finalize the progress window
  TeProgress::instance()->reset();
  
  initialSize_ = pitQueue_.size();

  return true;
}


int 
HidroFixPFS::pitsNeighbors( unsigned int x, unsigned int y )
{
  // Compute the start and end of window (3x3)
  unsigned int startColumn = x - 1;
  unsigned int endColumn = x + 2;
  unsigned int startLine = y - 1;
  unsigned int endLine = y + 2;  
  
  int pitsNumber = 0;
  unsigned char lddVal;

	for( unsigned int line = startLine; line < endLine; ++line )
	{				
		for( unsigned int column = startColumn; column < endColumn; ++column )
		{
      lddVal = lddMatrix_[line][column];      
      if( lddVal == 0 || lddVal == 9 )
      {
        ++pitsNumber;
      }
    }
  }

  lddVal = lddMatrix_[y][x];
  if( lddVal == 0 || lddVal == 9 )
    --pitsNumber;

  return pitsNumber;
}


double
HidroFixPFS::meanNeigbhors( unsigned int x, unsigned int y, int windowSize = 3, bool includeCenter = true )
{
  // Compute the start and end of window
  unsigned int startColumn = x - (int)( windowSize/2 );
  unsigned int endColumn = x + (int)( windowSize/2 ) +1;
  unsigned int startLine = y - (int)( windowSize/2 );
  unsigned int endLine = y + (int)( windowSize/2 ) + 1;

  // Check the borders
  if( (int)x < windowSize/2 )
    startColumn = 0;
  if( endColumn > rasterColumns_ )
    endColumn = rasterColumns_;
  if( (int)y < windowSize/2 )
    startLine = 0;
  if( endLine > rasterLines_ )
    endLine = rasterLines_;

  double total = 0;
  int elementsNumber = 0; // number of computed altimetrias  
  double altimetria;

  for( unsigned int line = startLine; line < endLine; ++line )
	{
		for( unsigned int column = startColumn; column < endColumn; ++column )
    {
      altimetria = demMatrix_[line][column];
      if( altimetria != demDummy_ )
      {
        total = total + altimetria;
        ++elementsNumber;
      }
    }
  }

  if( !includeCenter )
  {
    altimetria = demMatrix_[y][x];
    total =  total - altimetria;
    --elementsNumber;
  }

  double mean;

  if( elementsNumber > 0 )
  {
    mean = total / elementsNumber;
  }
  else
  {
    mean = -9999;
  }

  return mean;  
}

float
HidroFixPFS::smallerNeighbor( unsigned int line, unsigned int column )
{
  float smallerNeighbor = FLT_MAX;

  // never will be a pit on matrix border, so don't need to check the matrix limits.

  // line and column to be checked
  unsigned int l, c;

  l = line - 1;
  c = column - 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line - 1;
  c = column;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line - 1;
  c = column + 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line;
  c = column - 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line;
  c = column + 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line + 1;
  c = column - 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line + 1;
  c = column;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  l = line + 1;
  c = column + 1;
  if( demMatrix_[l][c] != demDummy_ )
  {
    if( demMatrix_[l][c] < smallerNeighbor )
    {
      smallerNeighbor = demMatrix_[l][c];
    }
  }

  return smallerNeighbor;  
}



void
HidroFixPFS::recomputeNeighborhoodLDD( unsigned int x, unsigned int y )
{
  // Recompute the LDD for the neighbors
  // Compute the start and end of window (3x3)
  unsigned int startColumn = x - 1;
  unsigned int endColumn = x + 2;
  unsigned int startLine = y - 1;
  unsigned int endLine = y + 2;

  // Check the borders
  if( x < 2 )
    startColumn = 1;
  if( endColumn > rasterColumns_-1 )
    endColumn = rasterColumns_-1;
  if( y < 2 )
    startLine = 1;
  if( endLine > rasterLines_-1 )
    endLine = rasterLines_-1;

	for( unsigned int line = startLine; line < endLine; ++line )
	{
		for( unsigned int column = startColumn; column < endColumn; ++column )
    {
      flowGrid_->d8( line, column );
    }
  }
}


void
HidroFixPFS::print()
{
  QFile pitCorrectionStatisticsFile( pitCorrectionStatisticsFileName_.c_str() );  

  if( pitCorrectionStatisticsFile.open( IO_ReadWrite  ) )
  {
    QTextStream pitCorrectionStatisticsStream;
    pitCorrectionStatisticsStream.setDevice( &pitCorrectionStatisticsFile );

    pitCorrectionStatisticsStream << demName_.c_str() << "\n";
    pitCorrectionStatisticsStream << "Image size: " << Te2String(rasterLines_*rasterColumns_).c_str() << "\n";
    pitCorrectionStatisticsStream << "Lines: " << Te2String(rasterLines_).c_str() << "\n";
    pitCorrectionStatisticsStream << "Columns: " << Te2String(rasterColumns_).c_str() << "\n \n";

    pitCorrectionStatisticsStream << "Start Time: " << starTime_.c_str();
    pitCorrectionStatisticsStream << "End Time: " << endTime_.c_str();
    pitCorrectionStatisticsStream << "Processing time: " << partialTime_.c_str() << "\n \n";    

    pitCorrectionStatisticsStream << "Initial pit number: " << Te2String(initialSize_).c_str() << "\n";
    pitCorrectionStatisticsStream << "Solved pit number: " << Te2String(solvedPitsNumber_).c_str() << "\n";    
    pitCorrectionStatisticsStream << "Unsolved pit number: " << Te2String(unsolvedPitsNumber_).c_str() << "\n";
    pitCorrectionStatisticsStream << "Simple pit number: " << Te2String(simplePits_).c_str() << "\n \n";

    pitCorrectionStatisticsStream << "-- Priority First Search Parameters --" << "\n";
    pitCorrectionStatisticsStream << "border_path_gradient_: " << Te2String(border_path_gradient_, 9).c_str() << "\n";    
    pitCorrectionStatisticsStream << "caving_increment_: " << Te2String(caving_increment_,9).c_str() << "\n";
    pitCorrectionStatisticsStream << "double_problem_increment_: " << Te2String(double_problem_increment_,9).c_str() << "\n";
    pitCorrectionStatisticsStream << "thresholdAltimetria_: " << Te2String(thresholdAltimetria_,9).c_str() << "\n \n";

    if( removeOutliers_ )
    {
      pitCorrectionStatisticsStream << "-- Remove Outliers Parameters --" << "\n";
      pitCorrectionStatisticsStream << "Low Outlier Threshold: " << Te2String(lowOutlierThreshold_, 9).c_str() << "\n";    
    }
      

    pitCorrectionStatisticsStream << "-- Priority First Search numbers --" << "\n";
    pitCorrectionStatisticsStream << "Path mean: " << Te2String(path_length_mean_, 0).c_str() << "\n";    
    pitCorrectionStatisticsStream << "Big Path: " << Te2String(big_path_).c_str() << "\n";    
    pitCorrectionStatisticsStream << "Precision Problem: " << Te2String(double_problem_).c_str() << "\n \n";
    //pitCorrectionStatisticsStream << "Pits in the path: " << Te2String(pits_in_path_).c_str() << "\n \n";    
  }
}


bool
HidroFixPFS::boundaryFixRegion( unsigned int x, unsigned int y )
{
  // se esta na borda do retangulo envolvente
  if( (x < 2) ||  (x > (rasterColumns_-3)) ||
    (y < 2) || (y > (rasterLines_-3)) )
  {
    return true;
  }

  // celulas vizinhas de dummy
  if( hasDummy_ )
  {
    // Compute the start and end of window (3x3)
    unsigned int startColumn = x - 1;
    unsigned int endColumn = x +  2;
    unsigned int startLine = y - 1;
    unsigned int endLine = y + 2;    

	  for( unsigned int line = startLine; line < endLine; ++line )
	  {				
		  for( unsigned int column = startColumn; column < endColumn; ++column )
		  {
        if( (int)demMatrix_[line][column] == (int)demDummy_ )
        {          
          return true;
        }
      }
    }
  }

  return false;
}


bool
HidroFixPFS::boundaryRegion( unsigned int x, unsigned int y )
{ 
  if( (x == 0) ||  (x == rasterColumns_-1) ||
    (y == 0) || (y == (rasterLines_-1) ) )
  {
    return true;
  }
  if( hasDummy_ )
  {
    // Compute the start and end of window (3x3)
    unsigned int startColumn = x - 1;
    unsigned int endColumn = x + 2;
    unsigned int startLine = y - 1;
    unsigned int endLine = y + 2;
  	

	  for( unsigned int line = startLine; line < endLine; ++line )
	  {				
		  for( unsigned int column = startColumn; column < endColumn; ++column )
		  {
        if( (int)demMatrix_[line][column] == (int)demDummy_ )
        {          
          return true;
        }
      }
    }
  }

  return false;
}



bool
HidroFixPFS::isPlaneBorder( unsigned int x, unsigned int y )
{
  // Tratar dados com Dummy tambm.
  if( !hasDummy_ )
  {
    // up
    if( y > 0)
    if( lddMatrix_[y-1][x] != 9 )
      return true;
    // right
    if( lddMatrix_[y][x+1] != 9 )
      return true;
    // down
    if( lddMatrix_[y+1][x] != 9 )
      return true;
    // left
    if( x > 0 )
    if( lddMatrix_[y][x-1] != 9 )
      return true;
  }
  else
  {
    unsigned char lddVal = 255;    
    
    // up
    if( y > 0 )
    {
      lddVal = lddMatrix_[y-1][x];
      if( lddVal != 255 )
      {
        if( lddVal != 9 )
          return true;
      }
    }
    
    // right
    lddVal = lddMatrix_[y][x+1];
    if( lddVal != 255 )
    {
      if( lddVal != 9 )
        return true;
    }
    
    // down
    lddVal = lddMatrix_[y+1][x];
    if( lddVal != 255 )
    {
      if( lddVal != 9 )
        return true;
    }    
    
    
    // left
    if( x > 0 )
    {
      lddVal = lddMatrix_[y][x-1];
      if( lddVal != 255 )
      {
        if( lddVal != 9 )
          return true;
      }
    }    
  }  

  return false;
}

void
HidroFixPFS::removingDEMoutliers()
{
  // Initialize the progress window
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
  TeProgress::instance()->setMessage("Removing low outlier");
  TeProgress::instance()->setTotalSteps( initialSize_ );
  TeProgress::instance()->setProgress( 0 );

  unsigned int counter = 0;

  while( !pitQueue_.empty() )
  {
    // retrieve the line and column of first pit 
    int *pitCoord = pitQueue_.front();
    unsigned int x = pitCoord[0];
    unsigned int y = pitCoord[1];    
    pitQueue_.pop();

    double smallerNeighbor = 1000000.0;
       
    // Compute the start and end of window (3x3)
    unsigned int startColumn = x - 1;
    unsigned int endColumn = x + 2;
    unsigned int startLine = y - 1;
    unsigned int endLine = y + 2;
	  for( unsigned int line = startLine; line < endLine; ++line )
	  {
		  for( unsigned int column = startColumn; column < endColumn; ++column )
      {        
        if( demMatrix_[line][column] < smallerNeighbor && line!=y && column!=x )
        {
          smallerNeighbor = demMatrix_[line][column];
        }       
      }
    }

    if( ( smallerNeighbor - demMatrix_[y][x] ) > lowOutlierThreshold_ )
    {
      demMatrix_[y][x] = smallerNeighbor;
      recomputeNeighborhoodLDD( x, y );
    }

    TeProgress::instance()->setProgress( counter++ );
  }

  initialSize_ = pitQueue_.size();
}
