#include <HidroFlowAlgorithm.h>

#include <HidroTime.h>

#include <TeRaster.h>

HidroFlowAlgorithm::HidroFlowAlgorithm( TeRaster* lddRaster )
{
  init();

  lddNLines_ = lddRaster->params().nlines_ - 1;
  lddNColumns_ = lddRaster->params().ncols_ - 1;
}

HidroFlowAlgorithm::~HidroFlowAlgorithm()
{
  lddMatrix_.clear();

  connectionsMatrix_.clear();
}

void
HidroFlowAlgorithm::init()
{
  // right
  lineIncrement[1] = 0;
  columnIncrement[1] = 1;

  // down right
  lineIncrement[2] = 1;
  columnIncrement[2] = 1;

  // down
  lineIncrement[4] = 1;
  columnIncrement[4] = 0;

  // down left
  lineIncrement[8] = 1;
  columnIncrement[8] = -1;

  // left
  lineIncrement[16] = 0;
  columnIncrement[16] = -1;

  // up left
  lineIncrement[32] = -1;
  columnIncrement[32] = -1;

  // up
  lineIncrement[64] = -1;
  columnIncrement[64] = 0;

  // up right
  lineIncrement[128] = -1;
  columnIncrement[128] = 1;
}

bool
HidroFlowAlgorithm::lddCode2LineColumn(    
    const unsigned int &lineFrom,
    const unsigned int &columnFrom,
    unsigned int &lineTo,
    unsigned int &columnTo )
{
  const unsigned char lddCode = lddMatrix_[lineFrom][columnFrom];

  if( lddCode != 255 && lddCode != 0)
  {
    lineTo = lineFrom + lineIncrement[lddCode];    
    // check border
    if( lineTo > lddNLines_ )
    {
      return false;
    }

    columnTo = columnFrom + columnIncrement[lddCode];
    // check border
    if( columnTo > lddNColumns_ )
    {
      return false;
    }    

    return true;
  }  

  return false;
}

bool
HidroFlowAlgorithm::initConnections()
{
  unsigned int nlines = lddMatrix_.GetLines();
  unsigned int ncolumns = lddMatrix_.GetColumns();

  TeProgress::instance()->setTotalSteps( nlines * 2 );

  // init connection matrix with zeros and dummys (255)  
  connectionsMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, connectionsMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddMatrix_[line][column] != 255 )
      {
        connectionsMatrix_[line][column] = 0;
      }
      else
      {
        connectionsMatrix_[line][column] = 255;
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  
  // calculate connection matrix
  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddCode2LineColumn( line, column, lineTo, columnTo ) )
      {        
        if( connectionsMatrix_[lineTo][columnTo] != 255 )
        {
          connectionsMatrix_[lineTo][columnTo]++;
        }
      }     
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( nlines + line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  return true;
}

bool
HidroFlowAlgorithm::cancel()
{  
  errorMessage_= "The operation was canceled by user.";
  
  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();
  
  return false;
}

const std::string& 
HidroFlowAlgorithm::getErrorMessage()
{
  return errorMessage_;
}

const std::string& 
HidroFlowAlgorithm::getTimeStart()
{
  return timeStart_;
}

const std::string& 
HidroFlowAlgorithm::getTimeEnd()
{
  return timeEnd_;
}

const std::string& 
HidroFlowAlgorithm::getTimeTotal()
{
  return timeTotal_;
}