#include <HidroFlowGraph2Raster.h>

#include <HidroFlowUtils.h>
#include <TeRaster.h>
#include <TeLayer.h>

HidroGraph2Raster::HidroGraph2Raster(
    TeRaster* lddRaster,    
    TeLayer* vertexLayer,
    TeLayer* edgeLayer,
    TeRaster* &outLDDRaster ) :
HidroFlowAlgorithm( lddRaster ),
lddRaster_(lddRaster),
vertexLayer_(vertexLayer),
edgeLayer_(edgeLayer),
outLDDRaster_(outLDDRaster)
{
  neighborDirection_[0][0] = 32;
  neighborDirection_[0][1] = 64;
  neighborDirection_[0][2] = 128;
  
  neighborDirection_[1][0] = 16;
  neighborDirection_[1][1] = 0;
  neighborDirection_[1][2] = 1;
  
  neighborDirection_[2][0] = 8;
  neighborDirection_[2][1] = 4;
  neighborDirection_[2][2] = 2;
}


bool
HidroGraph2Raster::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // create output LDD raster
  TeRasterParams lddRasterParams = lddRaster_->params();

  // Change mode
  lddRasterParams.mode_ = 'w';
  lddRasterParams.decoderIdentifier_ = "SMARTMEM";
  lddRasterParams.setDataType( TeUNSIGNEDCHAR );

  // Set Max and Minimal values
  lddRasterParams.vmax_[0] = -TeMAXFLOAT;
  lddRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  outLDDRaster_ = new TeRaster( lddRasterParams );

  // verify if scaledRaster created is valid
  if( !outLDDRaster_->init() )
  {
    errorMessage_ = lddRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // used in for's and progress bar
  unsigned int nlines = outLDDRaster_->params().nlines_;
  unsigned int ncolumns = outLDDRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Scaled LDD Raster step 1 from 3.");
  TeProgress::instance()->setTotalSteps( nlines );

  // init scaledMatrix
  outLDDMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, outLDDMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    outLDDMatrix_[line][0] = 16;
    outLDDMatrix_[line][ncolumns-1] = 1;
  }
  for( unsigned int column=0; column<ncolumns; column++ )
  {
    outLDDMatrix_[0][column] = 64;
    outLDDMatrix_[nlines-1][column] = 4;
  }
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
    {      
      outLDDMatrix_[line][column] = 255;      
    }    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 2
  TeLineSet lineSet;  
  if( !edgeLayer_->getLines( lineSet ) )
  {
    errorMessage_= "No lines.";
    
    return false;
  }
  
  TeProgress::instance()->setMessage("Creating Scaled LDD Raster step 2 from 3.");
  TeProgress::instance()->setTotalSteps( lineSet.size() );
  unsigned int cout = 0;

  for(TeLineSet::iterator lineIt = lineSet.begin(); lineIt != lineSet.end(); lineIt++)
  {
    TeCoord2D coordFrom = lddRaster_->coord2Index( lineIt->first() );
    TeCoord2D coordTo = lddRaster_->coord2Index( lineIt->last() );    

    unsigned int l = TeRound(coordTo.y_) - TeRound(coordFrom.y_) + 1;
    unsigned int c = TeRound(coordTo.x_) - TeRound(coordFrom.x_) + 1;

    outLDDMatrix_[TeRound(coordFrom.y_)][TeRound(coordFrom.x_)] = neighborDirection_[l][c];
    
    cout++;
    // refresh progress bar    
    TeProgress::instance()->setProgress( cout );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 3
  TeProgress::instance()->setMessage("Creating Scaled LDD Raster step 3 from 3.");  
  
  // copy matrix values to raster.  
  if( !copyPDIMatrix2Raster( outLDDMatrix_, outLDDRaster_ ) )
  {
    return cancel();
  }

  // Free Memory
  lddMatrix_.clear();
  outLDDMatrix_.clear();  

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;

}

bool
HidroGraph2Raster::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  outLDDMatrix_.clear();  

  return HidroFlowAlgorithm::cancel();
}