#include <HidroLengthSlope.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

#include<queue>

HidroLengthSlope::HidroLengthSlope(
    TeRaster* lddRaster,
    TeRaster* demRaster,
    TeRaster* accumulatedRaster,
    TeRaster* outletPixelsRaster,
    TeRaster* &lengthRaster,
    TeRaster* &slopeRaster,
    TeRaster* &visitedPixelsRaster,
    int factor,
    bool useMinimumSlope,
    double minimumSlope,
    double headSlope ) :
HidroFlowAlgorithm( lddRaster ),
lddRaster_(lddRaster),
demRaster_(demRaster),
accumulatedRaster_(accumulatedRaster),
outletPixelsRaster_(outletPixelsRaster),
lengthRaster_(lengthRaster),
slopeRaster_(slopeRaster),
visitedPixelsRaster_(visitedPixelsRaster),
factor_(factor),
useMinimumSlope_(useMinimumSlope),
minimumSlope_(minimumSlope),
headSlope_(headSlope)
{
  neighborDirection_[0][0] = 32;
  neighborDirection_[0][1] = 64;
  neighborDirection_[0][2] = 128;
  
  neighborDirection_[1][0] = 16;
  neighborDirection_[1][1] = 0;
  neighborDirection_[1][2] = 1;
  
  neighborDirection_[2][0] = 8;
  neighborDirection_[2][1] = 4;
  neighborDirection_[2][2] = 2;

  A_ = lddRaster_->params().projection()->datum().radius() / 1000.0;
  F_ = lddRaster_->params().projection()->datum().flattening();

  E2_ = 2 * F_ - F_ * F_;

  resx_ = lddRaster_->params().resx_;
  resy_ = lddRaster_->params().resy_;

  if( useMinimumSlope_ )
  {
    headSlope_ = 0.0;
  }
  else
  {
    headSlope_ = -0.99;
  }
}

bool
HidroLengthSlope::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( accumulatedRaster_->params().nlines_ != lddRaster_->params().nlines_ || accumulatedRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "Accumulated matrix lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // load data

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Accumulated
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Accumulated data.");    
  if( !copyRaster2PDIMatrix( accumulatedRaster_, accumulatedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load OutletPixels
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load OutletPixels data.");    
  if( !copyRaster2PDIMatrix( outletPixelsRaster_, outletPixelsMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output length raster
  
  // lengthPixelsRaster params
  TeRasterParams lengthRasterParams = outletPixelsRaster_->params();
  
  // Set dummy
  lengthRasterParams.setDummy( -9999 );
  lengthRasterParams.useDummy_ = true;
  
  // Change Resolution and nlines ncoloumns
  lengthRasterParams.boundingBoxResolution( lddRaster_->params().boundingBox().x1(),
    lddRaster_->params().boundingBox().y1(),
    lddRaster_->params().boundingBox().x2(),
    lddRaster_->params().boundingBox().y2(),
    lddRaster_->params().resx_ * factor_,
    lddRaster_->params().resy_ * factor_ );

  // Change mode
  lengthRasterParams.mode_ = 'w';
  lengthRasterParams.decoderIdentifier_ = "SMARTMEM";
  lengthRasterParams.setDataType( TeFLOAT );
  
  // Set Max and Minimal values
  lengthRasterParams.vmax_[0] = -TeMAXFLOAT;
  lengthRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  lengthRaster_ = new TeRaster( lengthRasterParams );

  // verify if lengthPixelsRaster created is valid
  if( !lengthRaster_->init() )
  {
    errorMessage_ = lengthRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // create output Slope raster
  
  // SlopePixelsRaster params
  TeRasterParams slopeRasterParams = lddRaster_->params();
  
  // Set dummy
  slopeRasterParams.setDummy( -9999 );
  slopeRasterParams.useDummy_ = true;
  
  // Change Resolution and nlines ncoloumns
  slopeRasterParams.boundingBoxResolution( lddRaster_->params().boundingBox().x1(),
    lddRaster_->params().boundingBox().y1(),
    lddRaster_->params().boundingBox().x2(),
    lddRaster_->params().boundingBox().y2(),
    lddRaster_->params().resx_ * factor_,
    lddRaster_->params().resy_ * factor_ );

  // Change mode
  slopeRasterParams.mode_ = 'w';
  slopeRasterParams.decoderIdentifier_ = "SMARTMEM";
  slopeRasterParams.setDataType( TeFLOAT );
  
  // Set Max and Minimal values
  slopeRasterParams.vmax_[0] = -TeMAXFLOAT;
  slopeRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  slopeRaster_ = new TeRaster( slopeRasterParams );

  // verify if SlopePixelsRaster created is valid
  if( !slopeRaster_->init() )
  {
    errorMessage_ = slopeRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // create output Visited raster
  
  // visitedPixelsRaster params
  TeRasterParams visitedPixelsRasterParams = outletPixelsRaster_->params();

  visitedPixelsRasterParams.setDummy( 255 );
  visitedPixelsRasterParams.vmax_[0] = 3;
  visitedPixelsRasterParams.vmin_[0] = 1;

  // Change mode
  visitedPixelsRasterParams.mode_ = 'w';
  visitedPixelsRasterParams.decoderIdentifier_ = "SMARTMEM";

  // create the raster
  visitedPixelsRaster_ = new TeRaster( visitedPixelsRasterParams );

  // verify if visitedPixelsRaster created is valid
  if( !visitedPixelsRaster_->init() )
  {
    errorMessage_ = visitedPixelsRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }
  
  // used in for's and progress bar
  unsigned int nlines = (int)(lddRaster_->params().nlines_ / (float)factor_ + 0.5);
  unsigned int ncolumns = (int)(lddRaster_->params().ncols_ / (float)factor_ + 0.5);

  // used in for's and progress bar High Resolution
  unsigned int nHlines = lddRaster_->params().nlines_;
  unsigned int nHcolumns = lddRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Length and Slope Rasters step 1 from X.");
  TeProgress::instance()->setTotalSteps( nHlines );
  
  // init length and Slope Rasters  
  for( unsigned int line=0; line<nHlines; line++ )
  {
    for( unsigned int column=0; column<nHcolumns; column++ )
    {
      lengthRaster_->setElement( column, line, -9999 );
      slopeRaster_->setElement( column, line, -9999 );
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 2
  TeProgress::instance()->setMessage("Creating Length and Slope Rasters step 2 from X.");
  TeProgress::instance()->setTotalSteps( nlines );
  
  // re find the outlet pixel
  // and store in a priority_queue
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      // re find the outlet pixel
      OutletPixel outletPixel = reFindOutletPixel( line, column );

      outletPixelsQueue_.push( outletPixel );
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  } 

  // STEP 3
  TeProgress::instance()->setMessage("Creating Length and Slope Rasters step 3 from X.");
  TeProgress::instance()->setTotalSteps( (int)outletPixelsQueue_.size() );

  unsigned int step = 0;

  // find donwstream and upstream
  while( !outletPixelsQueue_.empty() )
  {
    unsigned int line = (int)outletPixelsQueue_.top().line_ / factor_;
    unsigned int column = (int)outletPixelsQueue_.top().column_ / factor_;    

    unsigned int upLine;
    unsigned int upColumn;

    double upstream = calculateUpstream( line, column, outletPixelsQueue_.top(), upLine, upColumn );

    unsigned int downLine;
    unsigned int downColumn;

    double downstream = calculateDownstream( line, column, outletPixelsQueue_.top(), downLine, downColumn );

    // store length
    double length = upstream + downstream;

    if( upstream == 0.0 )
      length = 0.0;

    lengthRaster_->setElement( column, line, length );

    // compute slope

    if( upstream == 0.0 )
    {
      slopeRaster_->setElement( column, line, headSlope_ );
    }
    else
    {
      double altUp, altDown;

      demRaster_->getElement( upColumn, upLine, altUp );
      demRaster_->getElement( downColumn, downLine, altDown );

      double slope = ( altUp - altDown ) / length / 1000;

      if( useMinimumSlope_ )
      {
        if( !(slope > minimumSlope_) )
        {
          slope = minimumSlope_; // float tendendo a zero 1e-45;
        }
      }
        
      slopeRaster_->setElement( column, line, slope );
    }

    outletPixelsQueue_.pop();

    // refresh progress bar    
    TeProgress::instance()->setProgress( step++ );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }  

  // STEP 3
  TeProgress::instance()->setMessage("Creating Sclaled LDD Raster step 3 from X.");
  TeProgress::instance()->setTotalSteps( nlines );  

  // copy matrix values to raster.  
  if( !copyPDIMatrix2Raster( outletPixelsMatrix_, visitedPixelsRaster_ ) )
  {
    return cancel();
  } 

  // Free Memory
  accumulatedMatrix_.clear();  
  outletPixelsMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

double
HidroLengthSlope::calculateUpstream( unsigned int line,
    unsigned int column,
    const OutletPixel &outletPixel,
    unsigned int &upLine,
    unsigned int &upColumn )
{
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines()-1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns() -1;

  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  double distance = 0;

  lineFrom = outletPixel.line_;
  columnFrom = outletPixel.column_;

  bool touchBorder = false;

  while( upStream( lineFrom, columnFrom, lineTo, columnTo ) )
  {
    distance += distanceKm( lineFrom, columnFrom, lineTo, columnTo );

    // check block border
    if( lineTo < blockBeginLine
      || columnTo < blockBeginColumn
      || lineTo > blockEndLine-1
      || columnTo > blockEndColumn-1 )
    {
      touchBorder = true;
      break;
    }

    // mark upstream pixels
    outletPixelsMatrix_[lineTo][columnTo] = 3;    

    lineFrom = lineTo;
    columnFrom = columnTo;
  }

  upLine = lineFrom;
  upColumn = columnFrom;

  // Calular mesmo para as clulas de cabeceira - Javier 11/02/2014
  //if( !touchBorder )
  //{    
  //  distance = 0.0;
  //}
  //else
  {
    // neighborhood border
    unsigned int neighborhoodBeginLine;
    if( line == 0 || line-1 == 0)
      neighborhoodBeginLine = 1;
    else
      neighborhoodBeginLine = (line-1) * factor_;

    unsigned int neighborhoodEndLine = (line+1) * factor_ + factor_;

    if( neighborhoodEndLine > lddMatrix_.GetLines()-1 )
      neighborhoodEndLine = lddMatrix_.GetLines() - 1;

    unsigned int neighborhoodBeginColumn;
    if( column==0 || column-1 == 0 )
      neighborhoodBeginColumn = 1;
    else
      neighborhoodBeginColumn = (column-1) * factor_;
    
    unsigned int neighborhoodEndColumn = (column+1) * factor_ + factor_;

    if( neighborhoodEndColumn > lddMatrix_.GetColumns()-1 )
      neighborhoodEndColumn = lddMatrix_.GetColumns() - 1;

    bool touchOutletPixel = false;
    while( upStream( lineFrom, columnFrom, lineTo, columnTo ) )
    {
      //check raster bordes
      if( lineTo < 0
        || columnTo < 0
        || lineTo > lddMatrix_.GetLines()-1
        || columnTo > lddMatrix_.GetColumns()-1 )
      {
        break;
      }

      // check neighborhood border
      if( lineTo < neighborhoodBeginLine ||
        columnTo < neighborhoodBeginColumn || 
        lineTo > neighborhoodEndLine-1 ||
        columnTo > neighborhoodEndColumn-1 )
      {
        touchOutletPixel = true;
        break;
      }

      unsigned char lddCode = lddMatrix_[lineFrom][columnFrom];
      if( lddCode == 255 && lddCode == 0)
      {
        return false;
      }      

      if( outletPixelsMatrix_[lineTo][columnTo] != 255 )
      {
        touchOutletPixel = true;
        break;
      }

      lineFrom = lineTo;
      columnFrom = columnTo;
    }
    // Calular mesmo para as clulas de cabeceira - Javier 11/02/2014
    //if( !touchOutletPixel )
    //{
    //  distance = 0.0;
    //}    
  }

  return distance;
}

double 
HidroLengthSlope::calculateDownstream( unsigned int line,
    unsigned int column,
    const OutletPixel &outletPixel,
    unsigned int &downLine,
    unsigned int &downColumn )
{
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  double distance = 0;

  lineFrom = outletPixel.line_;
  columnFrom = outletPixel.column_;

  while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
  {    
    if( outletPixelsMatrix_[lineTo][columnTo] != 255 )
    {
      break;
    }

    lineFrom = lineTo;
    columnFrom = columnTo;
    
    distance += distanceKm( lineFrom, columnFrom, lineTo, columnTo );

    // mark downstream pixels
    outletPixelsMatrix_[lineTo][columnTo] = 2;
  }

  downLine = lineFrom;
  downColumn = columnFrom;

  return distance;
}

bool
HidroLengthSlope::upStream( const unsigned int &lineFrom,
      const unsigned int &columnFrom,
      unsigned int &lineTo,
      unsigned int &columnTo )
{
  //const unsigned char lddCode = lddMatrix_[lineFrom][columnFrom];

  bool find = false;

  //if( lddCode != 255 && lddCode != 0)
  {
    // look neighbors
    float mostAccumulated = 0;
    
    //up left
    if( (int)lddMatrix_[lineFrom-1][columnFrom-1] ==  2 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom-1];
        lineTo = lineFrom - 1;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //up
    if( (int)lddMatrix_[lineFrom-1][columnFrom] ==  4 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom];
        lineTo = lineFrom - 1;
        columnTo = columnFrom;
        find = true;
      }
    }

    //up rigth
    if( (int)lddMatrix_[lineFrom-1][columnFrom+1] ==  8 )
    {
      if( accumulatedMatrix_[lineFrom-1][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom-1][columnFrom+1];
        lineTo = lineFrom - 1;
        columnTo = columnFrom + 1;
        find = true;
      }
    }

    //left
    if( (int)lddMatrix_[lineFrom][columnFrom-1] ==  1 )
    {
      if( accumulatedMatrix_[lineFrom][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom][columnFrom-1];
        lineTo = lineFrom;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //right
    if( (int)lddMatrix_[lineFrom][columnFrom+1] ==  16 )
    {
      if( accumulatedMatrix_[lineFrom][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom][columnFrom+1];
        lineTo = lineFrom;
        columnTo = columnFrom + 1;
        find = true;
      }
    }

    //down left
    if( (int)lddMatrix_[lineFrom+1][columnFrom-1] ==  128 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom-1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom-1];
        lineTo = lineFrom+1;
        columnTo = columnFrom - 1;
        find = true;
      }
    }

    //down
    if( (int)lddMatrix_[lineFrom+1][columnFrom] ==  64 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom];
        lineTo = lineFrom+1;
        columnTo = columnFrom;
        find = true;
      }
    }

    //down
    if( (int)lddMatrix_[lineFrom+1][columnFrom+1] ==  32 )
    {
      if( accumulatedMatrix_[lineFrom+1][columnFrom+1] > mostAccumulated )
      {
        mostAccumulated = accumulatedMatrix_[lineFrom+1][columnFrom+1];
        lineTo = lineFrom+1;
        columnTo = columnFrom+1;
        find = true;
      }
    }
  }

  return find;
}

bool
HidroLengthSlope::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  accumulatedMatrix_.clear();  
  outletPixelsMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}


OutletPixel
HidroLengthSlope::reFindOutletPixel( unsigned int line, unsigned int column )
{
  unsigned int blockBeginLine = line * factor_;
  unsigned int blockEndLine = line * factor_ + factor_;

  if( blockBeginLine == 0 )
    blockBeginLine = 1;

  if( blockEndLine > lddMatrix_.GetLines()-1 )
    blockEndLine = lddMatrix_.GetLines() - 1;

  unsigned int blockBeginColumn = column * factor_;
  unsigned int blockEndColumn = column * factor_ + factor_;

  if( blockBeginColumn == 0 )
    blockBeginColumn = 1;

  if( blockEndColumn > lddMatrix_.GetColumns()-1 )
    blockEndColumn = lddMatrix_.GetColumns() - 1;

  // Lines
  for( unsigned int i = blockBeginColumn+1; i < blockEndColumn-1; i++ )
  {
    // fisrt line UP
    if( (int)outletPixelsMatrix_[blockBeginLine][i] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockBeginLine;
      newOutletPixel.ldd_ = 64;

      return newOutletPixel;
    }

    // last line DOWN
    if( (int)outletPixelsMatrix_[blockEndLine-1][i] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][i];
      newOutletPixel.column_ = i;
      newOutletPixel.line_ = blockEndLine-1;
      newOutletPixel.ldd_ = 4;

      return newOutletPixel;
    }
  }
  
  // Columns
  for( unsigned int i = blockBeginLine+1; i < blockEndLine-1; i++ )
  {
    // fisrt column LEFT
    if( (int)outletPixelsMatrix_[i][blockBeginColumn] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockBeginColumn];
      newOutletPixel.column_ = blockBeginColumn;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 16;

      return newOutletPixel;
    }

    // last column RIGHT
    if( (int)outletPixelsMatrix_[i][blockEndColumn-1] == 1 )
    {
      OutletPixel newOutletPixel;
      newOutletPixel.accumulatedArea_ = accumulatedMatrix_[i][blockEndColumn-1];
      newOutletPixel.column_ = blockEndColumn-1;
      newOutletPixel.line_ = i;
      newOutletPixel.ldd_ = 1;

      return newOutletPixel;
    }
  }

  // Diagonals
  // up left
  if( (int)outletPixelsMatrix_[blockBeginLine][blockBeginColumn] == 1 )    
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockBeginColumn];
    newOutletPixel.column_ = blockBeginColumn;
    newOutletPixel.line_ = blockBeginLine;

    unsigned char lddResult = lddMatrix_[blockBeginLine][blockBeginColumn];
    if( (int)lddResult == 8 )
      lddResult = 16;
    if( (int)lddResult == 128 )
      lddResult = 64;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

    // up rigth
  if( (int)outletPixelsMatrix_[blockBeginLine][blockEndColumn-1] == 1 )    
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockBeginLine][blockEndColumn-1];
    newOutletPixel.column_ = blockEndColumn-1;
    newOutletPixel.line_ = blockBeginLine;

    unsigned char lddResult = lddMatrix_[blockBeginLine][blockEndColumn-1];
    if( (int)lddResult == 32 )
      lddResult = 64;
    if( (int)lddResult == 2 )
      lddResult = 1;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

    // down left    
  if( (int)outletPixelsMatrix_[blockEndLine-1][blockBeginColumn] == 1 )
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockBeginColumn];
    newOutletPixel.column_ = blockBeginColumn;
    newOutletPixel.line_ = blockEndLine-1;

    unsigned char lddResult = lddMatrix_[blockEndLine-1][blockBeginColumn];
    if( (int)lddResult == 32 )
      lddResult = 16;
    if( (int)lddResult == 2 )
      lddResult = 4;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }  

    // down rigth
  if( (int)outletPixelsMatrix_[blockEndLine-1][blockEndColumn-1] == 1 )
  {
    OutletPixel newOutletPixel;
    newOutletPixel.accumulatedArea_ = accumulatedMatrix_[blockEndLine-1][blockEndColumn-1];
    newOutletPixel.column_ = blockEndColumn-1;
    newOutletPixel.line_ = blockEndLine-1;

    unsigned char lddResult = lddMatrix_[blockEndLine-1][blockEndColumn-1];
    if( (int)lddResult == 8 )
      lddResult = 4;
    if( (int)lddResult == 128 )
      lddResult = 64;
    
    newOutletPixel.ldd_ = lddResult;

    return newOutletPixel;
  }

  OutletPixel failPixel;
  failPixel.accumulatedArea_ = 0;
  failPixel.line_ = 0;
  failPixel.column_ = 0;  
  failPixel.ldd_ = 0;
  return failPixel;
}

double
HidroLengthSlope::distanceKm( const unsigned int &lineFrom,
      const unsigned int &columnFrom,
      const unsigned int &lineTo,
      const unsigned int &columnTo )
{
  TeCoord2D indexFrom( 0, lineFrom );
  TeCoord2D coordFrom = lddRaster_->index2Coord( indexFrom );  
  double YLATFrom = fabs(coordFrom.y());

  TeCoord2D indexTo( 0, lineTo );
  TeCoord2D coordTo = lddRaster_->index2Coord( indexTo );  
  double YLATTo = fabs(coordTo.y());

  double YLAT;
  if( YLATFrom < YLATTo )
    YLAT = YLATFrom;
  else
    YLAT = YLATTo;  

  // mesma coluna diferentes linhas
  if( lineFrom != lineTo && columnFrom == columnTo )
  {
    YLAT = YLAT + resy_ / 2;
    double RN =  A_ / sqrt( ( 1 - E2_ * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) );
    double dy = RN * sin( resy_ * M_PI/180 );
    return dy;
  }
  // mesma linha colunas diferentes
  else if( lineFrom == lineTo && columnFrom != columnTo )
  {
    double RN =  A_ / sqrt( ( 1 - E2_ * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) );
    double dx = RN * sin( resx_ * M_PI/180 ) * cos( YLAT * M_PI/180 );
    return dx;
  }
  // linha e coluna diferentes
  else
  {
    YLAT = YLAT + resy_ / 2;
    double RN =  A_ / sqrt( ( 1 - E2_ * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) );
    double dy = RN * sin( resy_ * M_PI/180 );
    double dx = RN * sin( resx_ * M_PI/180 ) * cos( YLAT * M_PI/180 );
    double diagonal = sqrt( dx*dx + dy*dy );
    return diagonal;
  }

  return 0.0;  
}
