/**
 * queuex.hpp file
 * This class is a STL queue that use swap in phisical disk.
 */

#ifndef _QUEUEX_
#define _QUEUEX_

#include <queue>
#include <TeUtils.h>

/**
 * @class queuex
 * @brief This class is a STL queue that use swap in disk.
 * @author Alexandre Copertino Jardim <alexcj@dpi.inpe.br> 
 */

template< class _Ty >
class queuex
{
public:
  typedef typename std::queue<_Ty>::size_type size_type;

  queuex()
  {
    setMaxMemPercentUsage( 5 );
  };

  ~queuex(){};

  /** \brief insert element at beginning
    */
  void push( _Ty v )
  {
    // Se tem Back
    if( queue_back_.size() )
    {
      // Se Back no ta cheia
      if( queue_back_.size() < max_queue_size_ )
      {
        queue_back_.push( v );
      }
      // Se Back ta cheia
      else
      {
        copyBack2File();       
        queue_back_.push( v );
      }
    }
    else
    {
      // Se tem File
      if( !queue_file_.empty() )
      {
        queue_back_.push( v );
      }
      else
      {
        // Se Front no ta cheia
        if( queue_front_.size() < max_queue_size_ )
        {
          queue_front_.push( v );
        }
        // Se Front ta cheia
        else
        {       
          queue_back_.push( v );
        }
      }
    }
  }
  /** \brief erase element at end
    */
  void pop()
  {
    // Se tem Front
    if( !queue_front_.empty() )
    {    
      queue_front_.pop();    
    }
    // Se tem File
    else if ( !queue_file_.empty() )
    {
      copyFile2Front();    
      queue_front_.pop();    
    }
    else
    {
      // So sobrou o Back
      queue_back_.pop();
    }
  }

   /** \brief return first element of mutable queue
     */
  _Ty front()
  {
    // Se tem Front
    if( !queue_front_.empty() )
    {
      return queue_front_.front();
    }
    // Se tem File
    else if ( !queue_file_.empty() )
    {
      copyFile2Front();
      return queue_front_.front();
    }
    // So sobrou o Back
    return queue_back_.front();
  }

  /** \brief test if queue is empty
    */
  bool empty() const
  {
    return queue_front_.empty() && queue_back_.empty() && queue_file_.empty();
  }

  /** \brief return length of queue
    */
  size_type size() const
  {
    return queue_front_.size() + queue_back_.size() + queue_file_.size() * max_queue_size_;
  }

  /**
    * @brief Set the max of memory that the queue can use.
    *
    * \param maxMemPercentUsage the percentage of memory that the queue can use.
    *    
    */
  void setMaxMemPercentUsage( double maxMemPercentUsage )
  {
    const double totalPhysMem = (double)TeGetTotalPhysicalMemory();
            
    const double totalVMem = (double)TeGetTotalVirtualMemory();
    
    const double totalMem = ( ((double)maxMemPercentUsage) / 100.0 ) *
      MIN( totalVMem, totalPhysMem );
      
    const double freeMem = MIN( totalMem, (double)TeGetFreePhysicalMemory() );   
        
    max_queue_size_ = (unsigned int)MAX( 100, freeMem / ((double)(sizeof(_Ty)+4) * 8) );
  }

protected:

  /** \brief Copy the back queue to the disk.
	  */
  void copyBack2File()
  {
    char *name = tempnam( 0, "queuex_tmp_" );
    FILE *stream = fopen( name, "wb");

    while( queue_back_.size() )
    {
      _Ty v = queue_back_.front();

      fwrite( (void*)&v, sizeof(v), 1, stream );      
      
      queue_back_.pop();
    }

    fclose( stream );
    queue_file_.push( name );
  }

  /** \brief Copy the first disk queue to the front queue.
	  */
  void copyFile2Front()
  {
    char *name = queue_file_.front();
    FILE *stream = fopen( name, "rb");
    rewind( stream );

    for(unsigned int i=0; i<max_queue_size_; i++)
    {
      _Ty v;
      fread( (void*)&v, sizeof(v), 1, stream );      
      queue_front_.push( v );
    }

    fclose( stream );
    remove( name );
    free( name );

    queue_file_.pop();
  }

  /** \brief Front Memory queue.
	  */
  std::queue< _Ty > queue_front_;

  /** \brief Back Memory queue.
	  */
  std::queue< _Ty > queue_back_;

  /** \brief This queue store the file names.
	  */
  std::queue< char* > queue_file_;

  /** \brief Max size of queue tiles.
	  */
  unsigned int max_queue_size_;

  /** \brief Max ram size of queue.
	  */
  unsigned int max_ram_;
};


#endif // _QUEUEX_
