#include <HidroUtils.h>

#include <HidroMetadata.h>

#include <TeRaster.h>
#include <TeImportRaster.h>
#include <TeVectorRemap.h>

#include <HidroPersister.h>
#include <HidroVertex.h>
#include <HidroEdge.h>
#include <HidroGraph.h>

//#include <qstring.h>
//#include <qmessagebox.h>

void HidroUtils::rasterTheme( TeView* view, TeLayer* layer )
{
  TeAbstractTheme* rstTheme = new TeTheme( layer->name(), layer );

  // make visible: the raster representation  
	rstTheme->visibleRep(TeRASTER | 0x40000000);													
	
  // add theme to the view
  view->add(rstTheme);
	
  // save theme definition to the database
  if( !rstTheme->save() )
	{
		 cout << "Error saving theme: " << _database->errorMessage() << endl;		 
		 return;
	}
}

void HidroUtils::rasterLegendOneColor( TeView* view, TeLayer* layer )
{
  TeAbstractTheme* rstTheme = new TeTheme( layer->name(), layer );

  // make visible: the raster representation
  // and the legend (0x4000000)
	rstTheme->visibleRep(TeRASTER | 0x40000000);													
	
  // add theme to the view
  view->add(rstTheme);
	
  // save theme definition to the database
  if( !rstTheme->save() )
	{
		 cout << "Error saving theme: " << _database->errorMessage() << endl;		 
		 return;
	}

  // create the slices
	vector<TeSlice> slicesVector;

  TeSlice slice("0.9","1.1"); 
	slicesVector.push_back(slice);  

  // Define a slicing mode: 3 slices
	TeGrouping group;
	group.groupMode_ = TeRasterSlicing;
	group.groupNumSlices_ = 1;
	group.groupPrecision_ = 2;

	if( !rstTheme->buildGrouping(group,slicesVector) ) 
	{
		 cout << "Error grouping theme: " << _database->errorMessage() << endl;		 
		 return;
	}

  // Assign to each slice one color	
  for(int ng=0; ng < rstTheme->grouping().groupNumSlices_; ng++)
	{    
		TeColor color(0,0,255);
		TeVisual* visual = TeVisualFactory::make("tevisual");            
		visual->color(color);
		visual->transparency(rstTheme->defaultLegend().visual(TePOLYGONS)->transparency());
		visual->contourStyle(rstTheme->defaultLegend().visual(TePOLYGONS)->contourStyle());
		visual->contourWidth(rstTheme->defaultLegend().visual(TePOLYGONS)->contourWidth());
		visual->contourColor(rstTheme->defaultLegend().visual(TePOLYGONS)->contourColor());	
		rstTheme->setGroupingVisual(ng+1,visual,TePOLYGONS);
	}

	if (!rstTheme->saveGrouping())
	{
		 cout << "Error saving theme slicing: " << _database->errorMessage() << endl;		 
		 return;
	}
}

void HidroUtils::rasterLegendRandomColors( TeView* view, TeLayer* layer )
{
  TeAbstractTheme* rstTheme = new TeTheme( layer->name(), layer );

  // make visible: the raster representation
  // and the legend (0x4000000)
	rstTheme->visibleRep(TeRASTER | 0x40000000);													
	
  // add theme to the view
  view->add(rstTheme);
	
  // save theme definition to the database
  if( !rstTheme->save() )
	{
		 cout << "Error saving theme: " << _database->errorMessage() << endl;		 
		 return;
	}

  // create the slices

  // slices number
  unsigned int slicesNumber = (unsigned int)layer->raster()->params().vmax_[0];

	vector<TeSlice> slicesVector;

  for( unsigned int i=0; i < slicesNumber; i++ )
  {
    TeSlice slice(Te2String(i+0.9),Te2String(i+1.1)); 
	  slicesVector.push_back(slice);
  }	

  // Define a slicing mode: 3 slices
	TeGrouping group;
	group.groupMode_ = TeRasterSlicing;
	group.groupNumSlices_ = slicesNumber;
	group.groupPrecision_ = 1;

	if( !rstTheme->buildGrouping(group,slicesVector) ) 
	{
		 cout << "Error grouping theme: " << _database->errorMessage() << endl;		 
		 return;
	}

  // Assign to each slice one color
  
  // Inicializa numeros aleatorios
  srand(time(NULL));
	
  for(int ng=0; ng < rstTheme->grouping().groupNumSlices_; ng++)
	{    
		TeColor color(rand()%256,rand()%256,rand()%256);
		TeVisual* visual = TeVisualFactory::make("tevisual");            
		visual->color(color);
		visual->transparency(rstTheme->defaultLegend().visual(TePOLYGONS)->transparency());
		visual->contourStyle(rstTheme->defaultLegend().visual(TePOLYGONS)->contourStyle());
		visual->contourWidth(rstTheme->defaultLegend().visual(TePOLYGONS)->contourWidth());
		visual->contourColor(rstTheme->defaultLegend().visual(TePOLYGONS)->contourColor());	
		rstTheme->setGroupingVisual(ng+1,visual,TePOLYGONS);
	}

	if (!rstTheme->saveGrouping())
	{
		 cout << "Error saving theme slicing: " << _database->errorMessage() << endl;		 
		 return;
	}
}

HidroUtils::HidroUtils(TeDatabase* database) : _database(database)
{
}

HidroUtils::~HidroUtils()
{
}

std::vector<std::string>
HidroUtils::listLayers(const bool& onlyRasters,
                       const bool& onlyPolygons,
                       const bool& onlyPoints )
{
	std::vector<std::string> values;

	TeLayerMap::iterator itlay = _database->layerMap().begin();
	
	while ( itlay != _database->layerMap().end() )
	{
		if(onlyRasters)
		{
			if((*itlay).second->hasGeometry(TeRASTER))
			{
				values.push_back((*itlay).second->name());
			}
		}
		else if(onlyPolygons)
		{
			if((*itlay).second->hasGeometry(TePOLYGONS))
			{
				values.push_back((*itlay).second->name());
			}
		}
    else if(onlyPoints)
		{
			if((*itlay).second->hasGeometry(TePOINTS))
			{
				values.push_back((*itlay).second->name());
			}
		}
		else
		{
			values.push_back((*itlay).second->name());
		}

		++itlay;
	}

	return values;
}

std::vector<std::string> HidroUtils::listLayersLUT()
{
	std::vector<std::string> values;

	TeLayerMap::iterator itlay = _database->layerMap().begin();
	
	while ( itlay != _database->layerMap().end() )
	{
		if((*itlay).second->hasGeometry(TeRASTER))
		{
			if((*itlay).second->raster()->params().photometric_[0] == TeRasterParams::TePallete)
			{
				values.push_back((*itlay).second->name());
			}
		}
		
		++itlay;
	}

	return values;
}

std::vector<std::string> HidroUtils::listLayerThemes(const std::string& layerName)
{
	TeThemeMap::iterator it = _database->themeMap().begin();

	std::vector<std::string> values;
	
	while(it != _database->themeMap().end())
	{
		TeTheme* theme = dynamic_cast<TeTheme*>(it->second);

		if(theme)
		{
			if(theme->layer()->name() == layerName)
			{
				values.push_back(theme->name());
			}
		}

		++it;
	}

	return values;
}

bool HidroUtils::checkLayerName(const std::string& layerName, std::string& errorMessage)
{
	bool changed = false;

//check if the name is invalid
	TeCheckName(layerName, changed, errorMessage);

	if(changed)
	{
		return false;
	}
	
//check if the name already exist in database
	if(_database->layerExist(layerName))
	{
		errorMessage = "Output layer name already exist.";
		return false;
	}

	return true;
}

TeLayer* HidroUtils::getLayerByName(const std::string& layerName)
{
	TeLayerMap::iterator it = _database->layerMap().begin();

	while(it != _database->layerMap().end())
	{
		if(it->second->name() == layerName)
		{
			return it->second;
		}

		 ++it;
	}

	return NULL;
}

TeTheme* HidroUtils::getThemeByName(const std::string& themeName)
{
	TeThemeMap::iterator it = _database->themeMap().begin();

	while(it != _database->themeMap().end())
	{
		TeTheme* theme = dynamic_cast<TeTheme*>(it->second);

		if(theme)
		{
			if(theme->name() == themeName)
			{
				return theme;
			}
		}
		++it;
	}

	return NULL;
}

bool HidroUtils::saveOutputRaster(const std::string& layerName, TeRaster* raster, bool autoMultiresolution )
{
	TeLayer* layer = new TeLayer(layerName, _database, raster->projection());
  
  unsigned int blockH = raster->params().nlines_;
	unsigned int blockW = raster->params().ncols_;	

	if (raster->params().nlines_ > 512 && raster->params().ncols_ > 512)
	{
		blockW = 512;
		blockH = 512;
	}

	TeProgress::instance()->reset();

	std::string caption = "Importing Raster...";
	TeProgress::instance()->setCaption(caption);

	std::string msg = "Importing raster generated by Terra Hidro Plugin.";
	TeProgress::instance()->setMessage(msg);

	bool res = TeImportRaster(layer, raster, blockW, blockH, raster->params().compression_[0], 
    "01", raster->params().dummy_[0], raster->params().useDummy_,
    TeRasterParams::TeNoExpansible /*raster->params().tiling_type_*/);
    

	TeProgress::instance()->reset();

  // Multiresolution
  if( autoMultiresolution )
  {
    /*unsigned int rasterColumns = raster->params().ncols_; // number of columns from input raster
    unsigned int rasterLines = raster->params().nlines_; // number of rows from input raster

    if( rasterColumns*rasterLines > 1000*1000 )
    {    
      vector<int> extraRes;
      unsigned int i;
      unsigned int nivel = 1;
      for (i = 500*500; i < rasterColumns*rasterLines && nivel < 11; i=i*4 ) 
      {
        extraRes.push_back(nivel);
        nivel++;
      }

		  TeProgress::instance()->reset();	
  		
      // build extra resolutions
		  for (i=0; i<extraRes.size();i++)
		{
			if(TeProgress::instance())
			{
				QString caption = ("Creating multiresolution");
				TeProgress::instance()->setCaption(caption.latin1());
				QString msg = "Level " + QString("%1").arg(extraRes[i]);
				msg += ". Please, wait!";
				TeProgress::instance()->setMessage(msg.latin1());
			}
			int resFac = (int)pow(2.0,extraRes[i]);
			if (!TeBuildLowerResolution(layer,raster,resFac,"01"))
			{
				QMessageBox::warning(0, "Warning",
					"Fail to build the resolution!");
				break;
			}
		}
		  if (TeProgress::instance())
			  TeProgress::instance()->reset();
    }/**/
  }

	if(!res)
	{
		_database->deleteLayer(layer->id());

		return false;
	}

	return true;
}

bool HidroUtils::saveOutputRasterCreatingTheme(const std::string& layerName, TeRaster* raster, TeView* view)
{
	TeLayer* layer = new TeLayer(layerName, _database, raster->projection());
    
	unsigned int blockW = raster->params().ncols_;
	unsigned int blockH = raster->params().nlines_;
  if (raster->params().ncols_ > 512)
	{
		blockW = 512;	
	}
	if (raster->params().nlines_ > 512)
  {
    blockH = 512;
  }
 
	TeProgress::instance()->reset();

	std::string caption = "Importing Raster...";
	TeProgress::instance()->setCaption(caption);

	std::string msg = "Importing raster generated by Terra Hidro Plugin.";
	TeProgress::instance()->setMessage(msg);

	bool res = TeImportRaster(layer, raster, blockW, blockH, raster->params().compression_[0], 
		"", raster->params().dummy_[0], raster->params().useDummy_, raster->params().tiling_type_);

	TeProgress::instance()->reset();

	if(!res)
	{
		_database->deleteLayer(layer->id());

		return false;
	}

	HidroPersister hidro(_database);

  if( view )
    hidro.createTheme(layer->name(), "", view, layer);

	return true;
}

std::string HidroUtils::getPointName(const std::string& graphName, TeTheme* vertexTheme, TeProjection* projCanvas, double& pixelSize, TeCoord2D& coord)
{
	std::string name = "";

	double tolerance = getToleranceValue(vertexTheme->layer()->projection(), projCanvas, pixelSize, coord);

	TePoint pt;

	if(vertexTheme->locatePoint(coord, pt, tolerance))
	{
		std::string id = pt.objectId();

		HidroMetadata metadata(_database);

		name = metadata.getVertexName(graphName, id);
	}

	return name;
}

TeCoord2D HidroUtils::getPointCoord(const std::string& graphName, TeTheme* vertexTheme, TeProjection* projCanvas, double& pixelSize, TeCoord2D& coord)
{
	TeCoord2D coordOut;

	double tolerance = getToleranceValue(vertexTheme->layer()->projection(), projCanvas, pixelSize, coord);

	TePoint pt;

	if(vertexTheme->locatePoint(coord, pt, tolerance))
	{
		coordOut = pt.elem();
	}

	return coordOut;
}

std::string HidroUtils::getLineName(const std::string& graphName, TeTheme* edgeTheme, TeProjection* projCanvas, double& pixelSize, TeCoord2D& coord)
{
	std::string name = "";

	double tolerance = getToleranceValue(edgeTheme->layer()->projection(), projCanvas, pixelSize, coord);

	TeLine2D line;

	if(edgeTheme->locateLine(coord, line, tolerance))
	{
		std::string id = line.objectId();

		HidroMetadata metadata(_database);

		name = metadata.getEdgeName(graphName, id);
	}

	return name;
}

HidroVertex* HidroUtils::createVertex(TeRaster* raster, const int& line, const int& col, const int& idx, HidroGraph* graph, const bool& checkVertex)
{
	if(line >= raster->params().nlines_ || col >= raster->params().ncols_)
	{
		return NULL;
	}

	if(line < 0  || col < 0)
	{
		return NULL;
	}

	HidroVertex* vertex = NULL;

	std::string vertexName = "vertex_" + Te2String(line) + "_" + Te2String(col);
	
	if(checkVertex)
	{
		if(graph && graph->isGraphValid())
		{
			vertex = graph->getGraphVertex(vertexName);

			if(vertex)
			{
				return vertex;
			}
		}
	}
	
	vertex = new HidroVertex();
	
//set vertex id
	vertex->setVertexId(vertexName);

//set vertex coord
	TeCoord2D coord = raster->index2Coord(TeCoord2D(col, line));
	vertex->setVertexCoord(coord);

//set vertex idx
	vertex->setVertexIdx(idx);	

	return vertex;
}

HidroEdge* HidroUtils::createEdge(HidroVertex* vertexFrom, HidroVertex* vertexTo)
{
	HidroEdge* edge = new HidroEdge();

//set edge id
	std::string edgeName = "edge_" + vertexFrom->getVertexId() + "_" + vertexTo->getVertexId();
	edge->setEdgeId(edgeName);

//set edge vertex from
	edge->setEdgeVertexFrom(vertexFrom);

//set edge vertex to
	edge->setEdgeVertexTo(vertexTo);

	return edge;
}

bool HidroUtils::generateGraph(TeRaster* inputRaster, HidroGraph& graph, const bool& checkVertex)
{
	unsigned int nLines = inputRaster->params().nlines_;
	unsigned int nCols  = inputRaster->params().ncols_;
	unsigned int count  = 0;
	
	unsigned int i, j, value;
	double val;

	TeCoord2D coord;

	if(TeProgress::instance())
	{
		TeProgress::instance()->setCaption("TerraHidro");
		TeProgress::instance()->setMessage("Generating Graph Vertex");
		TeProgress::instance()->setTotalSteps(nLines);
	}

//create all vertex
//include border vertexs
	for(i = 0; i < nLines; ++i)
	{
		for(j = 0; j < nCols; ++j)
		{
			inputRaster->getElement(j, i, val);
			value = (int)val;
	
			/*if(value == 255)
			{
				continue;
			}*/

		
			HidroVertex* vertex = createVertex(inputRaster, i, j, count, &graph, false);

			++count;

			graph.addGraphVertex(vertex);
		}

		if(TeProgress::instance())
		{
			TeProgress::instance()->setProgress(i);
		}
	}

	if(TeProgress::instance())
	{
		TeProgress::instance()->setCaption("TerraHidro");
		TeProgress::instance()->setMessage("Generating Graph Edges");
		TeProgress::instance()->setTotalSteps(nLines);
	}

//create all edges
	for(i = 0; i < nLines; ++i)
	{
		for(j = 0; j < nCols; ++j)
		{
			inputRaster->getElement(j, i, val);
			value = (int)val;
	
			if(value == 255)
			{
				continue;
			}
			
			std::string vertexName = "vertex_" + Te2String(i) + "_" + Te2String(j);
		
			HidroVertex* vertexFrom = graph.getGraphVertex(vertexName);

			if(vertexFrom)
			{
	//right
				if(value & 1)
				{
					std::string vertexName = "vertex_" + Te2String(i) + "_" + Te2String(j + 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);
					
					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
		
	//lower right
				if(value & 2)
				{
					std::string vertexName = "vertex_" + Te2String(i + 1) + "_" + Te2String(j + 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);
					
					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
				
	//down
				if(value & 4)
				{
					std::string vertexName = "vertex_" + Te2String(i + 1) + "_" + Te2String(j);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}

	//lower left
				if(value & 8)
				{
					std::string vertexName = "vertex_" + Te2String(i + 1) + "_" + Te2String(j - 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
				
	//left
				if(value & 16)
				{
					std::string vertexName = "vertex_" + Te2String(i) + "_" + Te2String(j - 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
		
	//upper left
				if(value & 32)
				{
					std::string vertexName = "vertex_" + Te2String(i - 1) + "_" + Te2String(j - 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
				
	//up
				if(value & 64)
				{
					std::string vertexName = "vertex_" + Te2String(i - 1) + "_" + Te2String(j);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}
		
	//upper right
				if(value & 128)
				{
					std::string vertexName = "vertex_" + Te2String(i - 1) + "_" + Te2String(j + 1);
					HidroVertex* vertex = graph.getGraphVertex(vertexName);

					if(vertex)
					{
						HidroEdge* edge = createEdge(vertexFrom, vertex);
						graph.addGraphEdge(edge);
					}
				}

			}
		}

		if(TeProgress::instance())
		{
			TeProgress::instance()->setProgress(i);
		}
	}

	if(TeProgress::instance())
	{
		TeProgress::instance()->reset();
	}

	return true;
}


double HidroUtils::getToleranceValue(TeProjection* projTheme, TeProjection* projCanvas, double& pixelSize, TeCoord2D& pt)
{
	double val = 0.;

	if(!(*projTheme == *projCanvas))
	{
		TeCoord2D pl(pt.x()-pixelSize/2., pt.y()-pixelSize/2.);
		TeCoord2D pu(pt.x()+pixelSize/2., pt.y()+pixelSize/2.);

		TeBox	box(pl, pu);

		box = TeRemapBox(box, projCanvas, projTheme);

		pixelSize = box.x2_ - box.x1_;

		val = 3.* pixelSize;
	}
	else
	{
		val = 3.* pixelSize;
	}

	return val;
}