#include <HidroVDNP.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

HidroVDNP::HidroVDNP(
  TeRaster* demRaster,
  TeRaster* lddRaster,
  TeRaster* &vdnpRaster ) :
    HidroFlowAlgorithm( lddRaster ),
    demRaster_(demRaster),
    lddRaster_(lddRaster),
    vdnpRaster_(vdnpRaster)
{  
}

HidroVDNP::~HidroVDNP()
{
  demMatrix_.clear();  
  vdnpMatrix_.clear();
}

bool
HidroVDNP::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( demRaster_->params().nlines_ != lddRaster_->params().nlines_ || demRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "DEM lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // load data

  // load DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load DEM data.");    
  if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output VDNP raster
  
  // vdnpRaster params
  TeRasterParams vdnpRasterParams = demRaster_->params();
  
  // Set dummy
  vdnpRasterParams.setDummy( -9999 );
  vdnpRasterParams.useDummy_ = true;  

  // Change mode
  vdnpRasterParams.mode_ = 'w';
  vdnpRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  vdnpRasterParams.vmax_[0] = -TeMAXFLOAT;
  vdnpRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  vdnpRaster_ = new TeRaster( vdnpRasterParams );

  // verify if vdnpRaster created is valid
  if( !vdnpRaster_->init() )
  {
    errorMessage_ = vdnpRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = vdnpRaster_->params().nlines_;
  unsigned int ncolumns = vdnpRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating VDNP Raster step 1 from 4.");
  TeProgress::instance()->setTotalSteps( nlines );

  unsigned char lddDummy = 255;
  if( lddRaster_->params().useDummy_ )
  {
    lddDummy = (unsigned char)lddRaster_->params().dummy_[0];
  }

  // init vdnpMatrix
  vdnpMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, vdnpMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddMatrix_[line][column] != lddDummy )
      {
        vdnpMatrix_[line][column] = demMatrix_[line][column];        
      }
      else
      {
        // flag until dummy
        vdnpMatrix_[line][column] = -9999;
      }
    }
    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  
  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating VDNP Raster step 2 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  if( !initConnections() )
  {
    return false;
  }  

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating VDNP Raster step 3 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init the vdnpRaster with drainage altimetria values
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  unsigned int lineStop;
  unsigned int columnStop;

  float valDEM;
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineFrom = line;
        columnFrom = column;

        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {
          if( lddMatrix_[lineTo][columnTo] == 255 )
          {
            break;
          }

          if( vdnpMatrix_[lineFrom][columnFrom] >  vdnpMatrix_[lineTo][columnTo] )
          {
            vdnpMatrix_[lineTo][columnTo] = vdnpMatrix_[lineFrom][columnFrom];            
          }

          // if connections bigger than one
          if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
          {
            connectionsMatrix_[lineTo][columnTo]--;
            break;
          }
          
          lineFrom = lineTo;
          columnFrom = columnTo;
        }        
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Freee Memory
  lddMatrix_.clear();
  
   // STEP 4

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating VDNP Raster step 4 from 4.");
  TeProgress::instance()->setProgress( 0 );

  // DEM - VDNP and dummys
  double valVDNP;  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      valVDNP = vdnpMatrix_[line][column];
      if( valVDNP != -9999 )
      {
        valVDNP = valVDNP - demMatrix_[line][column];
        vdnpRaster_->setElement( column, line, valVDNP );       
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Freee Memory
  demMatrix_.clear();

  // Free Memory  
  vdnpMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroVDNP::cancel()
{
  // Free Memory
  demMatrix_.clear();
  lddMatrix_.clear();
  vdnpMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}