
/******************************************************************************
* Hidro development team
* 
* Terra View Hidro Plugin
* 
* @(#) HidroMainWindow.h
*
*******************************************************************************
*
* $Rev: 8882 $:
*
* $Author: alexcj $:
*
* $Date: 2010-08-30 17:15:59 -0300 (seg, 30 ago 2010) $:
*
******************************************************************************/

/*!
	\brief This is the Hidro Interfaces Group.
	@defgroup hidroInterface The Group for Plugin Interaction.
*/

/** 
  * \file HidroMainWindow.ui.h
  *
  * \class HidroMainWindow
  *
  * \brief This file is a interface generated by qt designer
  *
  * This interface is the main interface from hidro application, 
  * will be used to manipulate graphs (create, load and save in 
  * terraLib database). Also will be used to execute algorithms
  * over the graph.
  *
  * \author Eric Silva Abreu <eric.abreu@funcate.org.br>
  * \version 1.0
  *
/*!
  @ingroup hidroInterface
 */

/*
** ----------------------------------------------------------------------------
** Includes:
*/
#include <TeDatabase.h>
#include <TeTheme.h>
#include <TeView.h>
#include <TeLayer.h>
#include <TeProjection.h>

#include <HidroPersister.h>
#include <HidroMetadata.h>
#include <HidroUtils.h>
#include <HidroGraph.h>
#include <HidroEdge.h>
#include <HidroVertex.h>

#include <HidroBoostUtils.h>
#include <HidroFlowUtils.h>
#include <HidroHand.h>
#include <HidroSlope.h>
#include <HidroFlowD8.h>
#include <HidroAccumulator.h>
#include <HidroDrainage.h>
#include <HidroSegmenter.h>
#include <HidroMiniBasins.h>
#include <HidroSubwatershed.h>
#include <HidroExtractByRasterMask.h>
#include <HidroSegmenter2Vector.h>
#include <HidroUpscaleRaster.h>
#include <HidroLengthSlope.h>
#include <HidroFlowGraph2Raster.h>
#include <HidroFlowGrid.h>

#include <HidroMetadata.h>

#include <HidroUpscale.h>
#include <HidroConvexHull.h>

#include <HidroBoostParams.h>
#include <HidroBoostAlgorithmFactory.h>

#include <HidroBatchPointsWindow.h>

#include <qmessagebox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qsettings.h>
#include <qfiledialog.h>
#include <qdir.h>

/** \brief HidroMainWindow init function.
    * Used to replace de default constructor from interface;
	* \param parameters		Input parameter used to set the parameters from app
	* \return				True if the input parameters are valids and False in other case.
    */
bool HidroMainWindow::init( PluginParameters* parameters )
{
  // remove pages unused pages
  {
    tabWidget->setCurrentPage( 16 );
    QWidget* currentePage = tabWidget->currentPage();
    tabWidget->removePage( currentePage);
    tabWidget->setCurrentPage( 16 );
    currentePage = tabWidget->currentPage();
    tabWidget->removePage( currentePage);
    tabWidget->setCurrentPage( 16 );
    currentePage = tabWidget->currentPage();
    tabWidget->removePage( currentePage);
    tabWidget->setCurrentPage( 0 );
  }

    if(parameters == NULL)
	{
		return false;
	}

	_parameters = parameters;

//list all layers from current database
	listLayers();

//list all graphs from current database
	listGraphs();

//list all flow algorithms implemented
	listFlowAlgorithms();

//list post processing algorithms names
	listPostProcessingAlgorithms();

//list boost algorithms
//	listBoostAlgorithms();

//boost interfaces
	_uiConnectedComponents = NULL;
	_uiDijkstra = NULL;
	_uiSearch = NULL;
	_uiBidirGraph = NULL;
	_maskWindow = NULL;

//convert interface defaults
	inputTerraHidroRadioButton_clicked();
	outputHandRadioButton_clicked();
  
	return true;
}

/** \brief HidroMainWindow destroy function.
    * Used to replace de default destructor from interface;
    */
void HidroMainWindow::destroy()
{
	if(_uiConnectedComponents)
	{
		delete _uiConnectedComponents;
		_uiConnectedComponents = 0;
	}

	if(_uiDijkstra)
	{
		delete _uiDijkstra;
		_uiDijkstra = 0;
	}

	if(_uiSearch)
	{
		delete _uiSearch;
		_uiSearch = 0;
	}

	if(_uiBidirGraph)
	{
		delete _uiBidirGraph;
		_uiBidirGraph = 0;
	}

	if(_maskWindow)
	{
		delete _maskWindow;
		_maskWindow = 0;
	}
}

/** \brief HidroMainWindow listLayers function.
    * Used to list all layer names from current database
	* that has TeRASTER representation inside a combo box
	* in interface.
    */
void HidroMainWindow::listLayers()
{
	if(!_parameters->getCurrentDatabasePtr())
	{
		return;
	}

//clear all layers names inside the combo box
  inputLayerComboBox->clear();
	inputLayerComboBox->insertItem("");

	flowInputLayerComboBox->clear();
	flowInputLayerComboBox->insertItem("");

	upScaleInputLayerComboBox->clear();
	upScaleInputLayerComboBox->insertItem("");

	maskInputLayerComboBox->clear();
	maskInputLayerComboBox->insertItem("");

	converterInputLDDComboBox->clear();
	converterInputLDDComboBox->insertItem("");

  accumulatorLDDComboBox->clear();
  accumulatorLDDComboBox->insertItem("");

  accumulatedComboBox->clear();
  accumulatedComboBox->insertItem("");  

  segmenterLDDComboBox->clear();
  segmenterLDDComboBox->insertItem("");

  segmenterDrainageComboBox->clear();
  segmenterDrainageComboBox->insertItem("");

  microBasinLDDComboBox->clear();
  microBasinLDDComboBox->insertItem("");

  microBasinSegmentedComboBox->clear();
  microBasinSegmentedComboBox->insertItem("");

  delimiterLDDComboBox->clear();
  delimiterLDDComboBox->insertItem("");

  delimiterAcuComboBox->clear();
  delimiterAcuComboBox->insertItem("");

  cutLimiterComboBox->clear();
  cutLimiterComboBox->insertItem("");

  cutInputComboBox->clear();
  cutInputComboBox->insertItem("");

  maskComboBox->clear();
  maskComboBox->insertItem("");

  lddComboBox->clear();
  lddComboBox->insertItem("");

  segmentedComboBox->clear();
  segmentedComboBox->insertItem("");

  demComboBox->clear();
  demComboBox->insertItem("");
  
  lddComboBox_2->clear();
  lddComboBox_2->insertItem("");
  
  drainageComboBox->clear();
  drainageComboBox->insertItem("");

  slopeDEMComboBox->clear();
  slopeDEMComboBox->insertItem("");

  upLDDLineEdit->clear();
  upLDDLineEdit->insertItem("");
  
  upAccumulatedLineEdit->clear();
  upAccumulatedLineEdit->insertItem("");

  lsLDDComboBox->clear();
  lsLDDComboBox->insertItem("");

  lsDEMComboBox->clear();
  lsDEMComboBox->insertItem("");

  lsACUComboBox->clear();
  lsACUComboBox->insertItem("");

  lsOutletComboBox->clear();
  lsOutletComboBox->insertItem("");

  g2rLDDComboBox->clear();
  g2rLDDComboBox->insertItem("");

  HidroUtils utils(_parameters->getCurrentDatabasePtr());

	std::vector<std::string> layerNames = utils.listLayers(true, false);

	for(unsigned int i = 0; i < layerNames.size(); ++i)
	{
		flowInputLayerComboBox->insertItem(layerNames[i].c_str());
		inputLayerComboBox->insertItem(layerNames[i].c_str());
		upScaleInputLayerComboBox->insertItem(layerNames[i].c_str());
		maskInputLayerComboBox->insertItem(layerNames[i].c_str());
		converterInputLDDComboBox->insertItem(layerNames[i].c_str());
    accumulatorLDDComboBox->insertItem(layerNames[i].c_str());
    accumulatedComboBox->insertItem(layerNames[i].c_str());
    segmenterLDDComboBox->insertItem(layerNames[i].c_str());
    segmenterDrainageComboBox->insertItem(layerNames[i].c_str());
    microBasinLDDComboBox->insertItem(layerNames[i].c_str());
    microBasinSegmentedComboBox->insertItem(layerNames[i].c_str());
    delimiterLDDComboBox->insertItem(layerNames[i].c_str());
    delimiterAcuComboBox->insertItem(layerNames[i].c_str());
    cutLimiterComboBox->insertItem(layerNames[i].c_str());
    cutInputComboBox->insertItem(layerNames[i].c_str());
    maskComboBox->insertItem(layerNames[i].c_str());
    lddComboBox->insertItem(layerNames[i].c_str());
    segmentedComboBox->insertItem(layerNames[i].c_str());
    demComboBox->insertItem(layerNames[i].c_str());
    lddComboBox_2->insertItem(layerNames[i].c_str());
    drainageComboBox->insertItem(layerNames[i].c_str());
    slopeDEMComboBox->insertItem(layerNames[i].c_str());
    upLDDLineEdit->insertItem(layerNames[i].c_str());
    upAccumulatedLineEdit->insertItem(layerNames[i].c_str());
    
    lsLDDComboBox->insertItem(layerNames[i].c_str());
    lsDEMComboBox->insertItem(layerNames[i].c_str());
    lsACUComboBox->insertItem(layerNames[i].c_str());
    lsOutletComboBox->insertItem(layerNames[i].c_str());

    g2rLDDComboBox->insertItem(layerNames[i].c_str());
	}

  batchPointsComboBox->clear();
  batchPointsComboBox->insertItem("");

  std::vector<std::string> pointLayerNames = utils.listLayers( false, false, true );
  
  for(unsigned int i = 0; i < pointLayerNames.size(); ++i)
	{
		batchPointsComboBox->insertItem(pointLayerNames[i].c_str());
  }

  lastLayers();
}


/** \brief HidroMainWindow listGraphs function.
    * Used to list all graph names from current database
	* inside a combo box in interface.
    */
void HidroMainWindow::listGraphs()
{
	graphNameComboBox->clear();
	graphNameUpScaleComboBox->clear();
	maskGraphNameComboBox->clear();
  g2rGraphNameComboBox->clear();

	HidroMetadata metadata(_parameters->getCurrentDatabasePtr());

	std::vector<std::string> graphList = metadata.getGraphNameListFromDB();

	graphNameComboBox->insertItem("");
	graphNameUpScaleComboBox->insertItem("");
	maskGraphNameComboBox->insertItem("");
  g2rGraphNameComboBox->insertItem("");

	for(unsigned int i =0; i < graphList.size(); ++i)
	{
		graphNameComboBox->insertItem(graphList[i].c_str());
		graphNameUpScaleComboBox->insertItem(graphList[i].c_str());
		maskGraphNameComboBox->insertItem(graphList[i].c_str());
    g2rGraphNameComboBox->insertItem(graphList[i].c_str());
	}
}

/** \brief HidroMainWindow listFlow Algorithms function.
    * Used to list all flow algorithms to extract flow and put
	* inside a combo box in interface.
    */
void HidroMainWindow::listFlowAlgorithms()
{
/*	flowAlgorithmComboBox->clear();
  flowAlgorithmComboBox->insertItem( "8 Connected Flow Common" );
  flowAlgorithmComboBox->setCurrentText( "8 Connected Flow Common" );	*/
}

void HidroMainWindow::listPostProcessingAlgorithms()
{
	/*postProcessingComboBox->clear();
  postProcessingComboBox->insertItem("Priority First Search - PFS");
  postProcessingComboBox->setCurrentText("Priority First Search - PFS");*/
}

/** \brief HidroMainWindow listBoost Algorithms function.
    * Used to list all boost algorithms implemented in terra hidro plugin
	* inside a combo box in interface.
    */
void HidroMainWindow::listBoostAlgorithms()
{
	algorithmComboBox->clear();
	//algorithmComboBox->insertItem("");

	std::vector<std::string> boostAlgorithms = getBoostAlgorithmsNames();

	for(unsigned int i = 0; i < boostAlgorithms.size(); ++i)
	{
		algorithmComboBox->insertItem(boostAlgorithms[i].c_str());
	}

  algorithmComboBox->setCurrentText( boostAlgorithms[0].c_str() );
}

/** \brief HidroMainWindow checkLayerName function.
    * Used to check the input layer name defined by user in interface
	* the name can not be exist in database and has to be a valid name.
	* \return	True if the name is valid and False in other case
    */
bool HidroMainWindow::checkLayerName(std::string layerName)
{
	if(!_parameters->getCurrentDatabasePtr())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Database not found."));
		return false;
	}

	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	std::string errorMessage = "";

	if(!utils.checkLayerName(layerName, errorMessage))
	{
		QMessageBox::warning(this, tr("Warning"), errorMessage.c_str());
		return false;
	}  

	return true;   
}

/** \brief HidroMainWindow interface button Create.
    * Used to create a new flow in terraLib database from a input
	* layer with TeRASTER representation. It's necessary to define
	* the output layer name to execute this operation.
    */
void HidroMainWindow::flowGeneratePushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }
  
  // DEM
	if( flowInputLayerComboBox->currentText().isEmpty() )
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input DEM grid not defined."));
		return;
	}

  // OutputNames
	if(flowOutputLayerNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Flow grid name not defined."));
		return;
	}
	if(!checkLayerName(flowOutputLayerNameLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Flow grid name already exist in this database."));
		return;
	}

  if(flowFixedDEMLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Fixed DEM grid name not defined."));
		return;
	}
  if(!checkLayerName(flowFixedDEMLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Fixed DEM grid name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // DEM
  TeLayer* demLayer = NULL;
	demLayer = utils.getLayerByName( flowInputLayerComboBox->currentText().latin1() );
  TeRaster* demRaster = NULL;
  demRaster = demLayer->raster();

  // output raster
  TeRaster* lddRaster = NULL;
  TeRaster* fixedDEMRaster = NULL;

  bool cavePlaneAreas = true;//cavePlaneAreasCheckBox->isChecked();
  bool singlePits = true; //singlePitsCheckBox->isChecked();
  bool partialResult = partialResultCheckBox->isChecked();
  bool removeOutliers = removeOutliersCheckBox->isChecked();
  double lowOutlierThreshold = lowOutlierThresholdLineEdit->text().toDouble();
  std::string outputDurectory = lddDirectoryOutputLineEdit->text().latin1();

  // FlowGrid
  HidroFlowGrid hidroFlowGrid( demRaster, lddRaster, fixedDEMRaster, cavePlaneAreas, singlePits, partialResult, removeOutliers,  lowOutlierThreshold, flowFixedDEMLineEdit->text().latin1(), outputDurectory);

  // execut the algorithm
  if( !hidroFlowGrid.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroFlowGrid.getErrorMessage().c_str()));
    
    if( fixedDEMRaster )
    {
      delete fixedDEMRaster;
    }

    if( lddRaster )
    {
      delete lddRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving LDD Grid into database.");
  utils.saveOutputRaster( flowOutputLayerNameLineEdit->text().latin1(), lddRaster );

  TeProgress::instance()->setMessage("Saving Fixed DEM Grid into database.");
  utils.saveOutputRaster( flowFixedDEMLineEdit->text().latin1(), fixedDEMRaster );

  // Create Themes
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( flowOutputLayerNameLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, lddLayer );
  }

  TeLayer* demFixedLayer = NULL;
  demFixedLayer = utils.getLayerByName( flowFixedDEMLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, demFixedLayer );
  }
	
  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastLDD", flowOutputLayerNameLineEdit->text() );
  settings.writeEntry( "lastDEM", flowInputLayerComboBox->currentText() );
  settings.writeEntry( "removeOutlierThreshold", lowOutlierThresholdLineEdit->text().latin1() );

  QString saveP;
  if( partialResult )
  {
    saveP = "1";
  }
  else
  {
    saveP = "0";
  }  
  settings.writeEntry( "savePartialResult", saveP );

  QString removeOutlier;
  if( removeOutlier )
  {
    removeOutlier = "1";
  }
  else
  {
    removeOutlier = "0";
  }
  settings.writeEntry( "removeOutlier", removeOutlier );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Flow Grid\n");
  message.append("Start Time: ");
  message.append( hidroFlowGrid.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroFlowGrid.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroFlowGrid.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );

  /*
  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select the output view in TerraView interface."));
    return;
  }

	//check input parameters from interface
	if(flowInputLayerComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

	if(flowOutputLayerNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output Layer name not defined."));
		return;
	}

	if(!checkLayerName(flowOutputLayerNameLineEdit->text().latin1()))
	{
		return;
	}

	TeLayer* inputLayer = NULL;

	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	inputLayer = utils.getLayerByName(flowInputLayerComboBox->currentText().latin1());

	if(!inputLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}

 //execute operation
	//if(flowDemRadioButton->isChecked())
  if(true)
	{
		///std::string algorithm = flowAlgorithmComboBox->currentText().latin1();		

	//post processing
		//bool pitcorrec = false;
		//std::string postProcName = postProcessingComboBox->currentText().latin1();
		std::string layerName = flowInputLayerComboBox->currentText().latin1();    

		TeDatabase* db = _parameters->getCurrentDatabasePtr();

    std::string lddName = flowOutputLayerNameLineEdit->text().latin1();
    std::string fixedName = flowFixedDEMLineEdit->text().latin1();
    
    
    if( !inputLayer->hasGeometry( TeRASTER ) )
    {
      QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

      return;
    }

    bool cavePlaneAreas = true;//cavePlaneAreasCheckBox->isChecked();
    bool singlePits = true; //singlePitsCheckBox->isChecked();
    bool partialResult = partialResultCheckBox->isChecked();
    std::string outputDurectory = lddDirectoryOutputLineEdit->text().latin1();

		if( !generateFlowD8(db, view, inputLayer, layerName, lddName, cavePlaneAreas, singlePits, partialResult, fixedName, outputDurectory ) )
		{
			QMessageBox::warning(this, tr("Warning"), tr("Error generating flow."));
			return;
		}    

  // Create Themes
    TeLayer* lddLayer = NULL;
	  lddLayer = utils.getLayerByName( lddName );
    if( view )
    {
      utils.rasterTheme( view, lddLayer );
    }

    TeLayer* demFixedLayer = NULL;
    demFixedLayer = utils.getLayerByName( layerName + "_Fixed" );
    if( view )
    {
      utils.rasterTheme( view, demFixedLayer );
    }
    
	}

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastLDD", flowOutputLayerNameLineEdit->text() );
  settings.writeEntry( "lastDEM", flowInputLayerComboBox->currentText() );


//list all layers from current database
	listLayers();

	_parameters->updateTVInterface();  */
}

/** \brief HidroMainWindow interface button Generate.
    * Used to create a new graph in terraLib database from a input
	* layer with TeRASTER representation. It's necessary to define
	* the output layer name to execute this operation.
    */
void HidroMainWindow::generatePushButton_clicked()
{
//check input parameters from interface
	if(inputLayerComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

	if(outputLayerNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output Layer name not defined."));
		return;
	}

	if(!checkLayerName(outputLayerNameLineEdit->text().latin1()))
	{
		return;
	}

	TeLayer* inputLayer = NULL;

	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	inputLayer = utils.getLayerByName(inputLayerComboBox->currentText().latin1());

	if(!inputLayer || !inputLayer->raster())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}

	TeProjection* proj = TeProjectionFactory::make(inputLayer->projection()->params());

//generate graph
	HidroGraph graph(outputLayerNameLineEdit->text().latin1());

	if(!utils.generateGraph(inputLayer->raster(), graph))
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error generating graph."));
		return;
	}

//persister graph
	HidroPersister persister(_parameters->getCurrentDatabasePtr());
	if(!persister.saveGraphInDatabase(graph, proj, inputLayer->raster()->params().resx_))
	{
		QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
		return;
	}

//preview terralib objects
  // Alexandre
	//std::string mess = "Do you really want to create graph view? \n This operation can take a few minutes.";
	//int res = QMessageBox::question(this, tr("Create View"), tr(mess.c_str()), tr("&Yes"), tr("&No"));

  int res = 0;

	if(res == 0)
	{
		if(!persister.createGraphView(&graph, proj))
		{
			QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
			return;
		}

		_parameters->updateTVInterface();
	}

	listGraphs();

	QMessageBox::information(this, tr("Information"), tr("Graph Generated."));
}

/** \brief HidroMainWindow interface button Delete.
    * Used to delete a graph from current database.
	* This operation also will be delete from database
	* the layer points (vertex) and layer lines (edges)
	* from this graph.
    */
void HidroMainWindow::deletePushButton_clicked()
{
	if(graphNameComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Graph not selected."));
		return;
	}

	int response = QMessageBox::question(this, tr("Delete Graph"),
		tr("Do you really want to delete selected graph from database:") + " " + graphNameComboBox->currentText() + "?",
			 tr("Yes"), tr("No"));

	if (response != 0)
		return;

//verify if graph exist
	HidroMetadata metadata(_parameters->getCurrentDatabasePtr());
	if(!metadata.existGraph(graphNameComboBox->currentText().latin1()))
	{
		QMessageBox::warning(this, tr("Warning"), tr("Graph not found in database."));
		return;
	}

//remove graph and layers
	HidroPersister persister(_parameters->getCurrentDatabasePtr());
	if(!persister.deleteGraphFromDatabase(graphNameComboBox->currentText().latin1()))
	{
		QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
		return;
	}

	listGraphs();

	_parameters->updateTVInterface();

	QMessageBox::warning(this, tr("Warning"), tr("Graph removed from database."));
}

/** \brief HidroMainWindow interface button Execute.
    * Used to execute a selected boost algorithm over
	* a selected graph.
    */
void HidroMainWindow::executePushButton_clicked()
{
//verify interface parameters
	if(graphNameComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Graph name not defined."));
		return;
	}

//get algorithm name
	if(algorithmComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Algorithm Name not defined."));
		return;
	}

	std::string boostAalgorithm = algorithmComboBox->currentText().latin1();

	if(boostAalgorithm == "Connected Components")
	{
		if(_uiConnectedComponents)
		{
			delete _uiConnectedComponents;
		}

		_uiConnectedComponents = new HidroBoostUIConnectedComponents(this, "CC", true);

		if(_uiConnectedComponents->init(_parameters, false))
		{
			_uiConnectedComponents->setGraphName(graphNameComboBox->currentText().latin1());
			_uiConnectedComponents->exec();

			delete _uiConnectedComponents;
			_uiConnectedComponents = NULL;
		}
	}
	else if(boostAalgorithm == "Strongly Connected Components")
	{
		if(_uiConnectedComponents)
		{
			delete _uiConnectedComponents;
		}

		_uiConnectedComponents = new HidroBoostUIConnectedComponents(this, "CC", true);

		if(_uiConnectedComponents->init(_parameters, true))
		{
			_uiConnectedComponents->setGraphName(graphNameComboBox->currentText().latin1());
			_uiConnectedComponents->exec();

			delete _uiConnectedComponents;
			_uiConnectedComponents = NULL;
		}
	}
	else if(boostAalgorithm == "Breadth First Search")
	{
		if(_uiSearch)
		{
			delete _uiSearch;
		}

		_uiSearch = new HidroBoostUISearch(this, "Search Algorithms", false);

		if(_uiSearch->init(_parameters, true))
		{
			_uiSearch->setGraphName(graphNameComboBox->currentText().latin1());
			_uiSearch->show();
		}
	}
	else if(boostAalgorithm == "Dijkstra Shortest Path")
	{
		if(_uiDijkstra)
		{
			delete _uiDijkstra;
		}

		_uiDijkstra = new HidroBoostUIDijkstra(this, "Dijkstra", false);

		if(_uiDijkstra->init(_parameters))
		{
			_uiDijkstra->setGraphName(graphNameComboBox->currentText().latin1());
			_uiDijkstra->show();
		}
	}
	else if(boostAalgorithm == "Bidirectional Graph")
	{
		if(_uiBidirGraph)
		{
			delete _uiBidirGraph;
		}

		_uiBidirGraph = new HidroBoostUIBidirectionalGraph(this, "Bidirectional Graph", false);

		if(_uiBidirGraph->init(_parameters, true))
		{
			_uiBidirGraph->setGraphName(graphNameComboBox->currentText().latin1());
			_uiBidirGraph->show();
		}
	}
	else if(boostAalgorithm == "Incidence Graph")
	{
		if(_uiBidirGraph)
		{
			delete _uiBidirGraph;
		}

		_uiBidirGraph = new HidroBoostUIBidirectionalGraph(this, "Bidirectional Graph", false);

		if(_uiBidirGraph->init(_parameters, false))
		{
			_uiBidirGraph->setGraphName(graphNameComboBox->currentText().latin1());
			_uiBidirGraph->show();
		}
	}
}

void HidroMainWindow::algorithmComboBox_activated( const QString & value )
{
	bglTextEdit->clear();

	if(value.isEmpty())
	{
		return;
	}

	std::string boostAalgorithm = value.latin1();

	HidroBoostParams pBoost;

	if(boostAalgorithm == "Connected Components")
	{
		pBoost._boostAlgorithmName = "Connected Components";
	}
	else if(boostAalgorithm == "Strongly Connected Components")
	{
		pBoost._boostAlgorithmName = "Strongly Connected Components";
	}
	else if(boostAalgorithm == "Breadth First Search")
	{
		pBoost._boostAlgorithmName = "Breadth First Search";
	}
	else if(boostAalgorithm == "Dijkstra Shortest Path")
	{
		pBoost._boostAlgorithmName = "Dijkstra Shortest Path";
	}
	else if(boostAalgorithm == "Bidirectional Graph")
	{
		pBoost._boostAlgorithmName = "Bidirectional Graph";
	}
	else if(boostAalgorithm == "Incidence Graph")
	{
		pBoost._boostAlgorithmName = "Incidence Graph";
	}

	HidroBoostAlgorithms* aBoost = HidroBoostAlgorithmFactory::make(pBoost);

	if(aBoost)
	{
		std::string algorithmDescritpion = aBoost->getBGLalgorithmDescription();

		bglTextEdit->setText(algorithmDescritpion.c_str());
	}
}

void HidroMainWindow::okUpScalePushButton_clicked()
{
//verify interface parameters
	if(graphNameUpScaleComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Graph name not defined."));
		return;
	}

	if(upScaleInputLayerComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

	if(cotatLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Threshold COTAT value not defined."));
		return;
	}

	bool okCotat = false;

	int cotatValue = cotatLineEdit->text().toInt(&okCotat);

	if(!okCotat)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Threshold COTAT invalid value."));
		return;
	}

	if(mufpLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Threshold MUFP value not defined."));
		return;
	}

	bool okMufp = false;

	int mufpValue = mufpLineEdit->text().toInt(&okMufp);

	if(!okMufp)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Threshold MUFP invalid value."));
		return;
	}

	if(outputGraphNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output graph name not defined."));
		return;
	}

//execute upscaling
	HidroUpscale upScale(_parameters->getCurrentDatabasePtr());

	if(!upScale.generateUpscale(graphNameUpScaleComboBox->currentText().latin1(), 
							upScaleInputLayerComboBox->currentText().latin1(),
							cotatValue,
							mufpValue))
	{
		QMessageBox::warning(this, tr("Warning"), upScale.errorMessage().c_str());
		return;
	}


//persister graph
	HidroGraph* graph = upScale.getOutputGraph();

	std::string graphName = outputGraphNameLineEdit->text();

	graph->setGraphName(graphName);

	if(graph && graph->isGraphValid())
	{

		//get input layer grid
		TeLayer* inputLayer = NULL;

		HidroUtils utils(_parameters->getCurrentDatabasePtr());

		inputLayer = utils.getLayerByName(upScaleInputLayerComboBox->currentText().latin1());

		TeProjection* proj = TeProjectionFactory::make(inputLayer->projection()->params());


		HidroPersister persister(_parameters->getCurrentDatabasePtr());
		if(!persister.saveGraphInDatabase(*graph, proj, inputLayer->raster()->params().resx_))
		{
			QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
			return;
		}

		//preview terralib objects
		std::string mess = "Do you really want to create graph view? \n This operation can take a few minutes.";
		int res = QMessageBox::question(this, tr("Create View"), tr(mess.c_str()), tr("&Yes"), tr("&No"));

		if(res == 0)
		{
			if(!persister.createGraphView(graph, proj))
			{
				QMessageBox::warning(this, tr("Warning"), persister.getErrorMessage().c_str());
				return;
			}

			_parameters->updateTVInterface();
		}

		listGraphs();

		QMessageBox::information(this, tr("Information"), tr("Graph Generated."));
	}
}

void HidroMainWindow::generateMaskPushButton_clicked()
{
//check input parameters
	if(maskGraphNameComboBox->currentText().isEmpty())
	{
		return;
	}
	std::string graphName = maskGraphNameComboBox->currentText().latin1();

	if(maskInputLayerComboBox->currentText().isEmpty())
	{
		return;
	}
	std::string inputLayer = maskInputLayerComboBox->currentText().latin1();

	if(maskOutputLayerNameLineEdit->text().isEmpty())
	{
		return;
	}
	std::string outputLayerName = maskOutputLayerNameLineEdit->text().latin1();

//load graph
	HidroPersister persister(_parameters->getCurrentDatabasePtr());

	HidroGraph inputGraph;

	if(!persister.loadGraphFromDatabase(graphName, inputGraph))
	{
		return;
	}

//load layer
	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	TeLayer* layer = utils.getLayerByName(inputLayer);

	if(!layer)
	{
		return;
	}

//generate mask
	HidroConvexHull cHull(_parameters->getCurrentDatabasePtr());

	TeLayer* layerMask = cHull.generateMask(&inputGraph, layer->raster(), outputLayerName);

	if(!layerMask)
	{
		return;
	}

//generate pols
	if(bool generatePols = generatePolsCheckBox->isChecked())
	{
		TeLayer* layerPols = cHull.generatePols(layerMask->raster(), outputLayerName);

		if(!layerPols)
		{
			return;
		}
	}

	_parameters->updateTVInterface();
}

void HidroMainWindow::editMaskPushButton_clicked()
{
	if(_maskWindow)
	{
		delete _maskWindow;
	}

	_maskWindow = new HidroMaskWindow(this, "Edit Mask", false);

	_maskWindow->init(_parameters);

	_maskWindow->show();
}


void HidroMainWindow::inputTerraHidroRadioButton_clicked()
{
	inputUPLEFTLineEdit->setText( "32" );   
	inputUPTLineEdit->setText( "64" );
	inputUPRIGHTLineEdit->setText( "128" );
	inputLEFTLineEdit->setText( "16" );
	inputPITLineEdit->setText( "0" );
	inputRIGHTLineEdit->setText( "1" );
	inputDOWNLEFTLineEdit->setText( "8" );
	inputDOWNTLineEdit->setText( "4" );
	inputDOWNRIGHTLineEdit->setText( "2" );
	inputDUMMYLineEdit->setText( "255" );

   inputUseDummyCheckBox->setChecked( true );
}

void HidroMainWindow::inputHandRadioButton_clicked()
{
	inputUPLEFTLineEdit->setText( "8" );
	inputUPTLineEdit->setText( "5" );
	inputUPRIGHTLineEdit->setText( "3" );
	inputLEFTLineEdit->setText( "7" );
	inputPITLineEdit->setText( "0" );
	inputRIGHTLineEdit->setText( "2" );
	inputDOWNLEFTLineEdit->setText( "6" );
	inputDOWNTLineEdit->setText( "4" );
	inputDOWNRIGHTLineEdit->setText( "1" );
	inputDUMMYLineEdit->setText( "0" );

   inputUseDummyCheckBox->setChecked( false );
}

void HidroMainWindow::inputMGBRadioButton_clicked()
{
	inputUPLEFTLineEdit->setText( "64" );
	inputUPTLineEdit->setText( "128" );
	inputUPRIGHTLineEdit->setText( "1" );
	inputLEFTLineEdit->setText( "32" );
	inputPITLineEdit->setText( "0" );
	inputRIGHTLineEdit->setText( "2" );
	inputDOWNLEFTLineEdit->setText( "16" );
	inputDOWNTLineEdit->setText( "8" );
	inputDOWNRIGHTLineEdit->setText( "4" );
	inputDUMMYLineEdit->setText( "-9999" );

   inputUseDummyCheckBox->setChecked( true );
}

void HidroMainWindow::inputCustomRadioButton_clicked()
{
	inputUPLEFTLineEdit->setText( "" );
	inputUPTLineEdit->setText( "" );
	inputUPRIGHTLineEdit->setText( "" );
	inputLEFTLineEdit->setText( "" );
	inputPITLineEdit->setText( "" );
	inputRIGHTLineEdit->setText( "" );
	inputDOWNLEFTLineEdit->setText( "" );
	inputDOWNTLineEdit->setText( "" );
	inputDOWNRIGHTLineEdit->setText( "" );
	inputDUMMYLineEdit->setText( "" );

   inputUseDummyCheckBox->setChecked( false );
}

void HidroMainWindow::outputTerraHidroRadioButton_clicked()
{	
	outputUPLEFTLineEdit->setText( "32" );
	outputUPTLineEdit->setText( "64" );
	outputUPRIGHTLineEdit->setText( "128" );
	outputLEFTLineEdit->setText( "16" );
	outputPITLineEdit->setText( "0" );
	outputRIGHTLineEdit->setText( "1" );
	outputDOWNLEFTLineEdit->setText( "8" );
	outputDOWNTLineEdit->setText( "4" );
	outputDOWNRIGHTLineEdit->setText( "2" );
	outputDUMMYLineEdit->setText( "255" );

   outputUseDummyCheckBox->setChecked( true );
}

void HidroMainWindow::outputHandRadioButton_clicked()
{
	outputUPLEFTLineEdit->setText( "8" );
	outputUPTLineEdit->setText( "5" );
	outputUPRIGHTLineEdit->setText( "3" );
	outputLEFTLineEdit->setText( "7" );
	outputPITLineEdit->setText( "0" );
	outputRIGHTLineEdit->setText( "2" );
	outputDOWNLEFTLineEdit->setText( "6" );
	outputDOWNTLineEdit->setText( "4" );
	outputDOWNRIGHTLineEdit->setText( "1" );
	outputDUMMYLineEdit->setText( "0" );

   outputUseDummyCheckBox->setChecked( false );
}

void HidroMainWindow::outputMGBRadioButton_clicked()
{
	outputUPLEFTLineEdit->setText( "64" );
	outputUPTLineEdit->setText( "128" );
	outputUPRIGHTLineEdit->setText( "1" );
	outputLEFTLineEdit->setText( "32" );
	outputPITLineEdit->setText( "0" );
	outputRIGHTLineEdit->setText( "2" );
	outputDOWNLEFTLineEdit->setText( "16" );
	outputDOWNTLineEdit->setText( "8" );
	outputDOWNRIGHTLineEdit->setText( "4" );
	outputDUMMYLineEdit->setText( "-9999" );

   outputUseDummyCheckBox->setChecked( true );
}

void HidroMainWindow::outputCustomRadioButton_clicked()
{
	outputUPLEFTLineEdit->setText( "" );
	outputUPTLineEdit->setText( "" );
	outputUPRIGHTLineEdit->setText( "" );
	outputLEFTLineEdit->setText( "" );
	outputPITLineEdit->setText( "" );
	outputRIGHTLineEdit->setText( "" );
	outputDOWNLEFTLineEdit->setText( "" );
	outputDOWNTLineEdit->setText( "" );
	outputDOWNRIGHTLineEdit->setText( "" );
	outputDUMMYLineEdit->setText( "" );

   outputUseDummyCheckBox->setChecked( false );
}

void HidroMainWindow::convertPushButton_clicked()
{
  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }

  HidroUtils utils(_parameters->getCurrentDatabasePtr());
  std::string errorMessage = "";

   int inputLDDmap[10];
	inputLDDmap[0] = inputUPLEFTLineEdit->text().toInt();
	inputLDDmap[1] = inputUPTLineEdit->text().toInt();
	inputLDDmap[2] = inputUPRIGHTLineEdit->text().toInt();
	inputLDDmap[3] = inputLEFTLineEdit->text().toInt();
	inputLDDmap[4] = inputPITLineEdit->text().toInt();
	inputLDDmap[5] = inputRIGHTLineEdit->text().toInt();
	inputLDDmap[6] = inputDOWNLEFTLineEdit->text().toInt();
	inputLDDmap[7] = inputDOWNTLineEdit->text().toInt();
	inputLDDmap[8] = inputDOWNRIGHTLineEdit->text().toInt();
	inputLDDmap[9] = inputDUMMYLineEdit	->text().toInt();

	int outputLDDmap[10];
	outputLDDmap[0] = outputUPLEFTLineEdit->text().toInt();
	outputLDDmap[1] = outputUPTLineEdit->text().toInt();
	outputLDDmap[2] = outputUPRIGHTLineEdit->text().toInt();
	outputLDDmap[3] = outputLEFTLineEdit->text().toInt();
	outputLDDmap[4] = outputPITLineEdit->text().toInt();
	outputLDDmap[5] = outputRIGHTLineEdit->text().toInt();
	outputLDDmap[6] = outputDOWNLEFTLineEdit->text().toInt();
	outputLDDmap[7] = outputDOWNTLineEdit->text().toInt();
	outputLDDmap[8] = outputDOWNRIGHTLineEdit->text().toInt();
	outputLDDmap[9] = outputDUMMYLineEdit	->text().toInt();

	std::string lddInputName = converterInputLDDComboBox->currentText().ascii();
	std::string lddOutputName = converterOutputLDDLineEdit->text().ascii();

   bool useDummy = outputUseDummyCheckBox->isChecked();

   int dataType = 1;

   if( outputTerraHidroRadioButton->isChecked() )
     dataType = 1;
   if( outputHandRadioButton->isChecked() )
     dataType = 2 ;
   if( outputMGBRadioButton->isChecked() )
     dataType = 3;
   if( outputCustomRadioButton->isChecked() )
     dataType = 4;

   convertLDD( _parameters->getCurrentDatabasePtr(), lddInputName, inputLDDmap, lddOutputName, outputLDDmap, dataType, useDummy );

   // create theme  
  TeLayer* outputDEMLayer = NULL;
	outputDEMLayer = utils.getLayerByName( lddOutputName );
  if( view )
  {
    utils.rasterTheme( view, outputDEMLayer );
  }

   _parameters->updateTVInterface();
	QMessageBox::information(this, tr("Information"), tr("LDD Converted."));
}


void HidroMainWindow::accumulatePushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // LDD
  if(accumulatorLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  
  // Output name (hand layer  name)
	if(accumulatoedNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Hand layer name not defined."));
		return;
	}
	if(!checkLayerName(accumulatoedNameLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Hand layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());  

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( accumulatorLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();  

  // output raster
  TeRaster* accumulatedRaster = NULL;

  bool kmArea = areaRadioButton->isChecked();

  // Accumulator
  HidroAccumulator hidroAccumulator( lddRaster, accumulatedRaster, kmArea );

  // execut the algorithm
  if( !hidroAccumulator.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroAccumulator.getErrorMessage().c_str()));
    
    if( accumulatedRaster )
    {
      delete accumulatedRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Accumulated Raster into database.");
  utils.saveOutputRaster( accumulatoedNameLineEdit->text().latin1(), accumulatedRaster );

  // create theme  
  TeLayer* accumulatedLayer = NULL;
	accumulatedLayer = utils.getLayerByName( accumulatoedNameLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, accumulatedLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastACU", accumulatoedNameLineEdit->text() );
  settings.writeEntry( "lastLDD", accumulatorLDDComboBox->currentText() );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Accumulated\n");
  message.append("Start Time: ");
  message.append( hidroAccumulator.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroAccumulator.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroAccumulator.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::extractDrainagePushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }

  // Accumulated
  if(accumulatedComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Accumulated layer not defined."));
		return;
	}

  // Threshold
  if(drainageThresholdLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Threshold not defined."));
		return;
	}

	// Output name (drainage layer  name)
	if(drainageNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Drainage layer name not defined."));
		return;
	}
	if(!checkLayerName(drainageNameLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Drainage layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // Accumulated
  TeLayer* accumulatedLayer = NULL;
	accumulatedLayer = utils.getLayerByName( accumulatedComboBox->currentText().latin1() );
  TeRaster* accumulatedRaster = NULL;
  accumulatedRaster = accumulatedLayer->raster();

  // get threshold
  float threshold = drainageThresholdLineEdit->text().toFloat();

  // output raster
  TeRaster* drainageRaster = NULL;

  // drainage
  HidroDrainage hidroDrainage( accumulatedRaster, drainageRaster, threshold );

  // execut the algorithm
  if( !hidroDrainage.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroDrainage.getErrorMessage().c_str()));
    
    if( drainageRaster )
    {
      delete drainageRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Drainage Raster into database.");
  utils.saveOutputRaster( drainageNameLineEdit->text().latin1(), drainageRaster );

  // create theme  
  TeLayer* drainageLayer = NULL;
	drainageLayer = utils.getLayerByName( drainageNameLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendOneColor( view, drainageLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastDRE", drainageNameLineEdit->text() );
  settings.writeEntry( "lastACU", accumulatedComboBox->currentText() );  

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Drainage\n");
  message.append("Start Time: ");
  message.append( hidroDrainage.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroDrainage.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroDrainage.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::suggestsThresholdPushButton_clicked()
{
  if(accumulatedComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

  TeLayer* inputLayer = NULL;

	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	inputLayer = utils.getLayerByName(accumulatedComboBox->currentText().latin1());

	if(!inputLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}

  std::string lddName = drainageNameLineEdit->text().latin1();
  
  if( !inputLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

    return;
  }

  int threshold = (int)(inputLayer->raster()->params().vmax_[0] * 0.01);
  drainageThresholdLineEdit->setText( QString::number( threshold ) );
}



void HidroMainWindow::segmentalPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // LDD
  if(segmenterLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  // Drainage
  if(segmenterDrainageComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Drainage layer not defined."));
		return;
	}

  // Output name (segmented layer  name)
	if(segmentedLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Segmented layer name not defined."));
		return;
	}
	if(!checkLayerName(segmentedLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Segmented layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());  

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( segmenterLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  // Drainage
  TeLayer* drainageLayer = NULL;
	drainageLayer = utils.getLayerByName( segmenterDrainageComboBox->currentText().latin1() );
  TeRaster* drainageRaster = NULL;
  drainageRaster = drainageLayer->raster();

  // output raster
  TeRaster* segmentedRaster = NULL;

  // Segmenter
  HidroSegmenter hidroSegmenter( lddRaster, drainageRaster, segmentedRaster );

  // execut the algorithm
  if( !hidroSegmenter.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroSegmenter.getErrorMessage().c_str()));
    
    if( segmentedRaster )
    {
      delete segmentedRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Segmented Raster into database.");
  utils.saveOutputRaster( segmentedLineEdit->text().latin1(), segmentedRaster );

  // create theme  
  TeLayer* segmentedLayer = NULL;
	segmentedLayer = utils.getLayerByName( segmentedLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendRandomColors( view, segmentedLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastSEG", segmentedLineEdit->text() );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Segmenter\n");
  message.append("Start Time: ");
  message.append( hidroSegmenter.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroSegmenter.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroSegmenter.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::extractMicroBasinPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // LDD
  if(microBasinLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  // Segmented
  if(microBasinSegmentedComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Segmented layer not defined."));
		return;
	}

  // Output name (miniBasins layer  name)
	if(microBasinLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("MiniBasins layer name not defined."));
		return;
	}
	if(!checkLayerName(microBasinLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("MiniBasins layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());  

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( microBasinLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  // Segmented
  TeLayer* segmentedLayer = NULL;
	segmentedLayer = utils.getLayerByName( microBasinSegmentedComboBox->currentText().latin1() );
  TeRaster* segmentedRaster = NULL;
  segmentedRaster = segmentedLayer->raster();

  // output raster
  TeRaster* miniBasinsRaster = NULL;

  // Mini Basins
  HidroMiniBasins hidroMiniBasins( lddRaster, segmentedRaster, miniBasinsRaster );

  // execut the algorithm
  if( !hidroMiniBasins.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroMiniBasins.getErrorMessage().c_str()));
    
    if( miniBasinsRaster )
    {
      delete miniBasinsRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving MiniBasins Raster into database.");
  utils.saveOutputRaster( microBasinLineEdit->text().latin1(), miniBasinsRaster );

  // create theme  
  TeLayer* miniBasinsLayer = NULL;
	miniBasinsLayer = utils.getLayerByName( microBasinLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendRandomColors( view, miniBasinsLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastMINI", microBasinLineEdit->text() );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Mini Basins\n");
  message.append("Start Time: ");
  message.append( hidroMiniBasins.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroMiniBasins.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroMiniBasins.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::delimitPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }
  
  // LDD
	if(delimiterLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  
  // ACU
	if(delimiterAcuComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Accumulated layer not defined."));
		return;
	}

  //Points
  if(batchPointsComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Batch Points Layer is not defined."));
		return;
	}  
  
  // OutputName
	if(delimitedNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("SubWatershed layer name not defined."));
		return;
	}
	if(!checkLayerName(delimitedNameLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("SubWatershed layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( delimiterLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  // ACU
  TeLayer* acuLayer = NULL;
	acuLayer = utils.getLayerByName( delimiterAcuComboBox->currentText().latin1() );
  TeRaster* acuRaster = NULL;
  acuRaster = acuLayer->raster();
  
  // pega os BatchPoints
  TeLayer* batchPointLayer = NULL;
	batchPointLayer = utils.getLayerByName(batchPointsComboBox->currentText().latin1());
	if(!batchPointLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting Batch Points layer from database."));
		return;
	}
  TePointSet pointSet;
  if( !batchPointLayer->getPoints( pointSet ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("The layer dont have any points!"));    
		return;
  }

  // output raster
  TeRaster* subWatershedRaster = NULL;

  bool minimizeBoundingBox = delimiterBoundingBoxCheckBox-> isChecked();

  // SubWatershed
  HidroSubWatershed hidroSubWatershed( lddRaster, acuRaster, pointSet, subWatershedRaster, minimizeBoundingBox );

  // execut the algorithm
  if( !hidroSubWatershed.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroSubWatershed.getErrorMessage().c_str()));
    
    if( subWatershedRaster )
    {
      delete subWatershedRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving SubWatershed Raster into database.");
  utils.saveOutputRaster( delimitedNameLineEdit->text().latin1(), subWatershedRaster );

  // create theme  
  TeLayer* subWatershedLayer = NULL;
	subWatershedLayer = utils.getLayerByName( delimitedNameLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendRandomColors( view, subWatershedLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastSUB", delimitedNameLineEdit->text() );
  settings.writeEntry( "lastLDD", delimiterLDDComboBox->currentText() );
  settings.writeEntry( "lastACU", delimiterAcuComboBox->currentText() );
  settings.writeEntry( "lastPoints", batchPointsComboBox->currentText() );

  QString minimize;
  if( minimizeBoundingBox )
  {
    minimize = "1";
  }
  else
  {
    minimize = "0";
  }

  settings.writeEntry( "lastMinimize", minimize );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("SubWatershed\n");
  message.append("Start Time: ");
  message.append( hidroSubWatershed.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroSubWatershed.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroSubWatershed.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::cutPushButton_clicked()
{
  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select the output view in TerraView interface."));
    return;
  }

  //check input parameters from interface
	if(cutLimiterComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}
  
  //check input parameters from interface
	if(cutInputComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

	if(outputNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output Layer name not defined."));
		return;
	}

	if(!checkLayerName(outputNameLineEdit->text().latin1()))
	{
		return;
	}

  // pega os dados do banco
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // pega o limitador
  TeLayer* limiterLayer = NULL;
	limiterLayer = utils.getLayerByName(cutLimiterComboBox->currentText().latin1());
	if(!limiterLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}
  if( !limiterLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

    return;
  }
  TeRaster* limiterRaster = limiterLayer->raster();  

  // pega o inputData
  TeLayer* inputLayer = NULL;
	inputLayer = utils.getLayerByName(cutInputComboBox->currentText().latin1());
	if(!inputLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}
  if( !inputLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

    return;
  }
  TeRaster* inputRaster = inputLayer->raster();

  TeRaster* outputRaster = extractByRasterMask( limiterRaster, inputRaster );  

  std::string outputName = outputNameLineEdit->text().latin1();

  utils.saveOutputRaster( outputName, outputRaster );

  // Create Theme  
  TeLayer* outputLayer = NULL;
	outputLayer = utils.getLayerByName( outputName );  
  if( view )
  {
    utils.rasterTheme( view, outputLayer );    
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastSUB", cutLimiterComboBox->currentText() );  


//list all layers from current database
	listLayers();

	_parameters->updateTVInterface();

	QMessageBox::information(this, tr("Information"), tr("Output data created."));
}


void HidroMainWindow::createMaskPushButton_clicked()
{
  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }

  //check input parameters
	if(maskComboBox->currentText().isEmpty())
	{
		return;
	}	

	if(maskNameLineEdit->text().isEmpty())
	{
		return;
	}
	std::string outputLayerName = maskNameLineEdit->text().latin1();

//load layer
	HidroUtils utils(_parameters->getCurrentDatabasePtr());

  TeLayer* layer = utils.getLayerByName( maskComboBox->currentText().latin1() );

	if(!layer)
	{
		return;
	}

//generate mask
	HidroConvexHull cHull(_parameters->getCurrentDatabasePtr());

	//TeLayer* layerPols = cHull.generateLines(layer->raster(), outputLayerName);
  TeLayer* layerPols = cHull.generatePols(layer->raster(), outputLayerName);

	if(!layerPols)
	{
		return;
	}

  // create theme
  TeLayer* vectorLayer = NULL;
	vectorLayer = utils.getLayerByName( maskNameLineEdit->text().latin1() );
  if( view )
  {
    HidroPersister persister(_parameters->getCurrentDatabasePtr());
    persister.createTheme( maskNameLineEdit->text().latin1(), "", view, vectorLayer );    
  }

	_parameters->updateTVInterface();

  QMessageBox::information(this, tr("Information"), tr("Vector data Created."));
}


void HidroMainWindow::vectorizePushButton_clicked()
{
  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }

  //check input parameters from interface
	if(lddComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

  if(segmentedComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Layer not defined."));
		return;
	}

	if(vectorLayerLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Output Layer name not defined."));
		return;
	}

	if(!checkLayerName(vectorLayerLineEdit->text().latin1()))
	{
		return;
	}

  // pega os dados do banco
  HidroUtils utils(_parameters->getCurrentDatabasePtr());
  TeDatabase* db = _parameters->getCurrentDatabasePtr();

  // pega o LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName(lddComboBox->currentText().latin1());
	if(!lddLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}
  if( !lddLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

    return;
  }
  TeRaster* lddRaster = lddLayer->raster();
  // converter em HidroMatrix
  HidroMatrix<unsigned char> lddMatrix( lddRaster->params().nlines_, lddRaster->params().ncols_,
    lddRaster->params() );
  // copia valores do tif para matriz LDD
  double lddVal;
  for( unsigned int line=0; line<lddMatrix.GetLines(); line++ )
  {
    for( unsigned int column=0; column<lddMatrix.GetColumns(); column++ )
    {
      lddRaster->getElement( column, line, lddVal );
      lddMatrix[line][column] = (unsigned char)lddVal;
    }
  }

  // pega a Drenagem
  TeLayer* segmentedLayer = NULL;
	segmentedLayer = utils.getLayerByName(segmentedComboBox->currentText().latin1());
	if(!segmentedLayer)
	{
		QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}
  if( !segmentedLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));

    return;
  }
  TeRaster* segmentedRaster = segmentedLayer->raster();
  // converter em HidroMatrix
  HidroMatrix<int> segmentedMatrix( segmentedRaster->params().nlines_, segmentedRaster->params().ncols_,
    segmentedRaster->params() );
  // copia valores do tif para matriz LDD
  double drainageVal;
  for( unsigned int line=0; line<segmentedMatrix.GetLines(); line++ )
  {
    for( unsigned int column=0; column<segmentedMatrix.GetColumns(); column++ )
    {
      segmentedRaster->getElement( column, line, drainageVal );
      segmentedMatrix[line][column] = (int)drainageVal;
    }
  } 

  // Dados de sada

  std::string segmentedName = vectorLayerLineEdit->text().latin1();

  Segmenter2Vector segmenter( segmentedName, db, segmentedRaster, lddMatrix, segmentedMatrix );
  
  if( !segmenter.vectorizing() )
	{
    std::string errorMessage = segmenter.getErrorMessage();
    QMessageBox::warning(this, tr("Warning"), tr(errorMessage.c_str()));
		return;
	}  

  std::string filename = segmentedName + "_segmenter_statistics.txt";
  segmenter.print( filename );
  
  lddMatrix.freeMemory();
  segmentedMatrix.freeMemory();

  // create theme
  TeLayer* vectorLayer = NULL;
	vectorLayer = utils.getLayerByName( vectorLayerLineEdit->text().latin1() );
  if( view )
  {
    HidroPersister persister(_parameters->getCurrentDatabasePtr());
    persister.createTheme( vectorLayerLineEdit->text().latin1(), "", view, vectorLayer );    
  }
  

//list all layers from current database
	listLayers();

	_parameters->updateTVInterface();

	QMessageBox::information(this, tr("Information"), tr("Segmented Created."));
}


void HidroMainWindow::okPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }
  
  // DEM
  if(demComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input DEM layer not defined."));
		return;
	}
  // LDD
  if(lddComboBox_2->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  // Drainage
  if(drainageComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Drainage layer not defined."));
		return;
	}

  // Output name (hand layer  name)
	if(handLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Hand layer name not defined."));
		return;
	}
	if(!checkLayerName(handLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Hand layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // DEM
  TeLayer* demLayer = NULL;
	demLayer = utils.getLayerByName( demComboBox->currentText().latin1() );
  TeRaster* demRaster = NULL;
  demRaster = demLayer->raster();

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( lddComboBox_2->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  // Drainage
  TeLayer* drainageLayer = NULL;
	drainageLayer = utils.getLayerByName( drainageComboBox->currentText().latin1() );
  TeRaster* drainageRaster = NULL;
  drainageRaster = drainageLayer->raster();

  // output raster
  TeRaster* handRaster = NULL;

  // Hand
  HidroHand hidroHand( demRaster, lddRaster, drainageRaster, handRaster );

  // execut the algorithm
  if( !hidroHand.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroHand.getErrorMessage().c_str()));
    
    if( handRaster )
    {
      delete handRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Hand Raster into database.");
  utils.saveOutputRaster( handLineEdit->text().latin1(), handRaster );

  // create theme  
  TeLayer* handLayer = NULL;
	handLayer = utils.getLayerByName( handLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, handLayer );
  }

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Hand\n");
  message.append("Start Time: ");
  message.append( hidroHand.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroHand.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroHand.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );  
}

void HidroMainWindow::accumulatorLDDComboBox_activated( const QString & )
{
  // LDD
  if(accumulatorLDDComboBox->currentText().isEmpty())
	{
    areaRadioButton->setDisabled( true );
    areaRadioButton->setChecked( false );
    pixelRadioButton->setChecked( true );
		return;
	}
	
	// open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());  

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( accumulatorLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  string t = lddRaster->params().projection()->name();

  if( lddRaster->params().projection()->name().compare( "LatLong" ) == 0 )
  {
    areaRadioButton->setDisabled( false );
  }
  else
  {
    areaRadioButton->setDisabled( true );
    areaRadioButton->setChecked( false );
    pixelRadioButton->setChecked( true );
  }  
}


void HidroMainWindow::slopePushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }
  
  // DEM
  if(slopeDEMComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input DEM layer not defined."));
		return;
	}  

  // Output name (slope layer  name)
	if(slopeNameLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Hand layer name not defined."));
		return;
	}
	if(!checkLayerName(slopeNameLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Hand layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

  // DEM
  TeLayer* demLayer = NULL;
	demLayer = utils.getLayerByName( slopeDEMComboBox->currentText().latin1() );
  TeRaster* demRaster = NULL;
  demRaster = demLayer->raster();

  // output raster
  TeRaster* slopeRaster = NULL;

  // Hand
  HidroSlope hidroSlope( demRaster, slopeRaster );

  // execut the algorithm
  if( !hidroSlope.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroSlope.getErrorMessage().c_str()));
    
    if( slopeRaster )
    {
      delete slopeRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Slope Raster into database.");
  utils.saveOutputRaster( slopeNameLineEdit->text().latin1(), slopeRaster );

  // create theme  
  TeLayer* slopeLayer = NULL;
	slopeLayer = utils.getLayerByName( slopeNameLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, slopeLayer );
  }

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Slope\n");
  message.append("Start Time: ");
  message.append( hidroSlope.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroSlope.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroSlope.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );  
}


void HidroMainWindow::upscalePushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // LDD
  if(upLDDLineEdit->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  // Accumulated
  if(upAccumulatedLineEdit->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Accumulated layer not defined."));
		return;
	}

  // Factor
  if(upFactorLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input facotr value not defined."));
		return;
	}  

  // Output name (upscale layer  name)
	if(upScaledLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("UpscaleRaster layer name not defined."));
		return;
	}
	if(!checkLayerName(upScaledLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("UpscaleRaster layer name already exist in this database."));
		return;
	}

  // Output name (outlet pixels layer  name)
	if(upOutletPixelsLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr(" Outlet Pixels layer name not defined."));
		return;
	}
	if(!checkLayerName(upOutletPixelsLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Outlet Pixels layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr()); 

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( upLDDLineEdit->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  // Accumulated
  TeLayer* accumulatedLayer = NULL;
	accumulatedLayer = utils.getLayerByName( upAccumulatedLineEdit->currentText().latin1() );
  TeRaster* accumulatedRaster = NULL;
  accumulatedRaster = accumulatedLayer->raster();

  // Parameters
  int factor = upFactorLineEdit->text().toInt();
  int mufp = upMUPFLineEdit->text().toInt();
  int at = upATLineEdit->text().toInt();

  // output raster
  TeRaster* scaledRaster = NULL;  
  TeRaster* outletPixelsRaster = NULL;

  // UpscaleRaster
  HidroUpscaleRaster hidroUpscaleRaster( lddRaster,
    accumulatedRaster,
    scaledRaster,
    outletPixelsRaster,
    factor,
    mufp,
    at );

  // execut the algorithm
  if( !hidroUpscaleRaster.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroUpscaleRaster.getErrorMessage().c_str()));
    
    if( scaledRaster )
    {
      delete scaledRaster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Scaled Raster into database.");
  utils.saveOutputRaster( upScaledLineEdit->text().latin1(), scaledRaster );

  // import result to database
  TeProgress::instance()->setMessage("Saving Outlet Pixels Raster into database.");
  utils.saveOutputRaster( upOutletPixelsLineEdit->text().latin1(), outletPixelsRaster );

  // create theme  
  TeLayer* upscaleLayer = NULL;
	upscaleLayer = utils.getLayerByName( upScaledLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, upscaleLayer );
  }

  // create theme  
  TeLayer* outletLayer = NULL;
	outletLayer = utils.getLayerByName( upOutletPixelsLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendOneColor( view, outletLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastUpLDD", upScaledLineEdit->text() );
  //settings.writeEntry( "lastLDD", upScaledLineEdit->text() );
  settings.writeEntry( "lastOUT", upOutletPixelsLineEdit->text() );
  settings.writeEntry( "lastFactor", upFactorLineEdit->text() );

  // list all layers from current database
	listLayers();

  // update the TerraView interface
	_parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("UpscaleRaster\n");
  message.append("Start Time: ");
  message.append( hidroUpscaleRaster.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroUpscaleRaster.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroUpscaleRaster.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::upRecalculatePushButton_clicked()
{
  float factor = upFactorLineEdit->text().toFloat();

  int mufp = factor / 5;
  upMUPFLineEdit->setText( QString::number( mufp ) );

  int at = factor * factor;
  upATLineEdit->setText( QString::number( at ) );

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr()); 

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( upLDDLineEdit->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();

  float resx = lddRaster->params().resx_ * factor;
  float resy = lddRaster->params().resy_ * factor;

  upResXLineEdit->setText( QString::number( resx ) );
  upResYLineEdit->setText( QString::number( resy ) );  

  int lines = (int)(lddRaster->params().nlines_ / factor + 0.5);
  float columns = (int)(lddRaster->params().ncols_ / factor + 0.5);

  upLinesLineEdit->setText( QString::number( lines ) );
  upColumnsLineEdit->setText( QString::number( columns ) );
}


void HidroMainWindow::lsPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // LDD
  if(lsLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input LDD layer not defined."));
		return;
	}
  
  // DEM
  if(lsDEMComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input DEM layer not defined."));
		return;
	}  
  
  // Accumulated
  if(lsACUComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Accumulated layer not defined."));
		return;
	}
  
  // Outlet Pixels
  if(lsOutletComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Accumulated layer not defined."));
		return;
	}

  // Factor
  if(lsFactorLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input facotr value not defined."));
		return;
	} 

  // Output name (outlet pixels layer  name)
	if(lsLenghtLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr(" Lenght layer name not defined."));
		return;
	}
	if(!checkLayerName(lsLenghtLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Outlet Pixels layer name already exist in this database."));
		return;
	}
  
  // Output name (outlet pixels layer  name)
	if(lsSlopeLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr(" Outlet Pixels layer name not defined."));
		return;
	}
	if(!checkLayerName(lsSlopeLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Outlet Pixels layer name already exist in this database."));
		return;
	}

  // Output name (lsscale layer  name)
	if(lsOutlet2LineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Outlet Pixels 2 layer name not defined."));
		return;
	}
	if(!checkLayerName(lsOutlet2LineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Outlet Pixels 2 layer name already exist in this database."));
		return;
	}  

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr()); 

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( lsLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();
  
  // DEM
  TeLayer* demLayer = NULL;
	demLayer = utils.getLayerByName( lsDEMComboBox->currentText().latin1() );
  TeRaster* demRaster = NULL;
  demRaster = demLayer->raster();

  // Accumulated
  TeLayer* accumulatedLayer = NULL;
	accumulatedLayer = utils.getLayerByName( lsACUComboBox->currentText().latin1() );
  TeRaster* accumulatedRaster = NULL;
  accumulatedRaster = accumulatedLayer->raster();
  
  // Outlet Pixels
  TeLayer* outletPixelsLayer = NULL;
	outletPixelsLayer = utils.getLayerByName( lsOutletComboBox->currentText().latin1() );
  TeRaster* outletPixelsRaster = NULL;
  outletPixelsRaster = outletPixelsLayer->raster();

  // Parameters
  int factor = lsFactorLineEdit->text().toInt();

  bool useMinimumSlope = lsMinumumSlopecheckBox->isChecked();
  double minimumSlope = lsMinimumSlopeLineEdit->text().toDouble();
  //double headSlope = lsHeadSlopeLineEdit->text().toDouble();
  double headSlope = 0.0;
  
  // output raster
  TeRaster* lenghtRaster = NULL;  
  TeRaster* slopeRaster = NULL;  
  TeRaster* outletPixels2Raster = NULL;

  // Length and Slope
  HidroLengthSlope hidroLengthSlope( lddRaster,
    demRaster,
    accumulatedRaster,    
    outletPixelsRaster,
    lenghtRaster,
    slopeRaster,
    outletPixels2Raster,
    factor,
    useMinimumSlope,
    minimumSlope,
    headSlope );

  // execut the algorithm
  if( !hidroLengthSlope.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroLengthSlope.getErrorMessage().c_str()));
    
    if( slopeRaster )
    {
      delete slopeRaster;
    }

    if( lenghtRaster )
    {
      delete lenghtRaster;
    }
    
    if( outletPixels2Raster )
    {
      delete outletPixels2Raster;
    }
    
    return;
  }
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Lenght Raster into database.");
  utils.saveOutputRaster( lsLenghtLineEdit->text().latin1(), lenghtRaster );
  
  // import result to database
  TeProgress::instance()->setMessage("Saving Slope Raster into database.");
  utils.saveOutputRaster( lsSlopeLineEdit->text().latin1(), slopeRaster );

  // import result to database
  TeProgress::instance()->setMessage("Saving Outlet Pixels 2 Raster into database.");
  utils.saveOutputRaster( lsOutlet2LineEdit->text().latin1(), outletPixels2Raster );

  // create theme  
  TeLayer* lenghtLayer = NULL;
	lenghtLayer = utils.getLayerByName( lsLenghtLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, lenghtLayer );
  }
  
  // create theme  
  TeLayer* slopeLayer = NULL;
	slopeLayer = utils.getLayerByName( lsSlopeLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, slopeLayer );
  }

  // create theme  
  TeLayer* outletLayer = NULL;
	outletLayer = utils.getLayerByName( lsOutlet2LineEdit->text().latin1() );
  if( view )
  {
    utils.rasterLegendRandomColors( view, outletLayer );
  }

  // update settings
  QSettings settings;  
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  settings.writeEntry( "lastLDD", lsLDDComboBox->currentText() );
  settings.writeEntry( "lastACU", lsACUComboBox->currentText() );
  settings.writeEntry( "lastOUT", lsOutletComboBox->currentText() );  
  settings.writeEntry( "lastDEM", lsDEMComboBox->currentText() );
  settings.writeEntry( "lastFactor", lsFactorLineEdit->text() );
  settings.writeEntry( "lastLength", lsLenghtLineEdit->text() );
  settings.writeEntry( "lastSlope", lsSlopeLineEdit->text() );
  settings.writeEntry( "lastPaths", lsOutlet2LineEdit->text() );

  // list all layers from current database
	listLayers();  

  // lsdate the TerraView interface
  _parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("UpscaleRaster\n");
  message.append("Start Time: ");
  message.append( hidroLengthSlope.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroLengthSlope.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroLengthSlope.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::g2rGraphNameComboBox_activated( const QString &value )
{
  g2rVertexLayerLineEdit->setText("");
	g2rEdgeLayerLineEdit->setText("");

	if(value.isEmpty())
	{
//disabable edition
		//editTabWidget->setDisabled(true);
		return;
	}

	HidroMetadata metadata(_parameters->getCurrentDatabasePtr());

	TeLayer* vertexLayer	= metadata.getGraphLayerPoints(value.latin1());
	TeLayer* edgeLayer		= metadata.getGraphLayerLines(value.latin1());

	if(!vertexLayer || !edgeLayer)
	{
//graph layers not found
		//editTabWidget->setDisabled(true);

		QMessageBox::warning(this, tr("Warning"), tr("Layers from graph not found."));
		return;
	}

//inform in interface the graph layer names
	g2rVertexLayerLineEdit->setText(vertexLayer->name().c_str());
	g2rEdgeLayerLineEdit->setText(edgeLayer->name().c_str());

//get themes from graph
	//listThemes();
}

/*void HidroMainWindow::listThemes()
{
  HidroUtils utils(_parameters->getCurrentDatabasePtr());

	std::vector<std::string> vertexThemes = utils.listLayerThemes(g2rVertexLayerLineEdit->text().latin1());
	g2rVertexThemeComboBox->clear();
	g2rVertexThemeComboBox->insertItem("");

	for(unsigned int i =0; i < vertexThemes.size(); ++i)
	{
		g2rVertexThemeComboBox->insertItem(vertexThemes[i].c_str());
	}
	
	std::vector<std::string> edgesThemes = utils.listLayerThemes(g2rEdgeLayerLineEdit->text().latin1());
	g2rEdgeThemeComboBox->clear();
	g2rEdgeThemeComboBox->insertItem("");

	for(unsigned int i =0; i < edgesThemes.size(); ++i)
	{
		g2rEdgeThemeComboBox->insertItem(edgesThemes[i].c_str());
	}
}*/


void HidroMainWindow::g2rOKPushButton_clicked()
{
  // check inputs

  // Get View
  TeView* view = 0;
  view = _parameters->getCurrentViewPtr();
  if( !view )
  {
    QMessageBox::warning(this, tr("Warning"), tr("Please select one view in TerraView interface."));
    return;
  }  
  
  // Graph
  if(g2rGraphNameComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Graph layer not defined."));
		return;
	}

  // LDD
  if(g2rLDDComboBox->currentText().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Input Upscaled LDD layer not defined."));
		return;
	}
  
  // Output LDD (lsscale layer  name)
	if(g2rLDDLineEdit->text().isEmpty())
	{
		QMessageBox::warning(this, tr("Warning"), tr("Outlet LDD layer name not defined."));
		return;
	}
	if(!checkLayerName(g2rLDDLineEdit->text().latin1()))
	{
    QMessageBox::warning(this, tr("Warning"), tr("Outlet LDD layer name already exist in this database."));
		return;
	}

  // open input raster's
  HidroUtils utils(_parameters->getCurrentDatabasePtr()); 

  // LDD
  TeLayer* lddLayer = NULL;
	lddLayer = utils.getLayerByName( g2rLDDComboBox->currentText().latin1() );
  TeRaster* lddRaster = NULL;
  lddRaster = lddLayer->raster();
  
  // output raster
  TeRaster* outLDDRaster = NULL;  

  // get graph
  HidroMetadata metadata( _parameters->getCurrentDatabasePtr() );

  // get points
  TeLayer* vertexLayer = metadata.getGraphLayerPoints( g2rGraphNameComboBox->currentText().latin1() );

  // get edges
  TeLayer* edgeLayer = metadata.getGraphLayerLines( g2rGraphNameComboBox->currentText().latin1() );  

  // Length and Slope
  HidroGraph2Raster hidroGraph2Raster( lddRaster,
    vertexLayer,
    edgeLayer,    
    outLDDRaster );

  // execut the algorithm
  if( !hidroGraph2Raster.execute() )
  {
    QMessageBox::warning(this, tr("Warning"), tr(hidroGraph2Raster.getErrorMessage().c_str()));
    
    if( outLDDRaster )
    {
      delete outLDDRaster;
    }

    if( lddRaster )
    {
      delete lddRaster;
    }
    
    return;
  }

  // open input raster's
  //HidroUtils utils(_parameters->getCurrentDatabasePtr()); 
  
  // import result to database
  TeProgress::instance()->setMessage("Saving LDD Raster into database.");
  utils.saveOutputRaster( g2rLDDLineEdit->text().latin1(), outLDDRaster );
  
  // create theme  
  TeLayer* outLDDLayer = NULL;
	outLDDLayer = utils.getLayerByName( g2rLDDLineEdit->text().latin1() );
  if( view )
  {
    utils.rasterTheme( view, outLDDLayer );
  }
  
  // list all layers from current database
	listLayers();

  // lsdate the TerraView interface
  _parameters->updateTVInterface();

  // close the window
  //accept(); 

  // informe the successful process
  QString message;
  message.append("Graph 2 Raster\n");
  message.append("Start Time: ");
  message.append( hidroGraph2Raster.getTimeStart().c_str() );
  message.append("\nEnd Time: ");
  message.append( hidroGraph2Raster.getTimeEnd().c_str() );
  message.append("\n\nTotal Time: ");
  message.append( hidroGraph2Raster.getTimeTotal().c_str() );

	QMessageBox::information(this, tr("Information"), tr(message) );
}


void HidroMainWindow::lsMinumumSlopecheckBox_stateChanged( int state )
{
  if( !state )
  {
    lsMinimumSlopeLineEdit->setDisabled( true );
    //lsHeadSlopeLineEdit->setDisabled( true );    
    //lsHeadSlopeLineEdit->setText("-0.99");
  }
  else
  {
    lsMinimumSlopeLineEdit->setDisabled( false );
    //lsHeadSlopeLineEdit->setDisabled( false );
    //lsHeadSlopeLineEdit->setText("0.0");
  }
}


void HidroMainWindow::lddOutputDirectoryToolButton_clicked()
{
  QSettings settings;
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );

  QString outPath = settings.readEntry( "outputDirectory" );  

  QString outFileName = QFileDialog::getExistingDirectory( outPath );
	
  if (!outFileName.isEmpty())	
  {
	  lddDirectoryOutputLineEdit->setText(outFileName);    
    settings.writeEntry( "outputDirectory", outFileName );
  }
}

void HidroMainWindow::lastLayers()
{
  // qsettings defaults
  QSettings settings;

  // LDD
  settings.setPath( "TerraHidro.br", "LDD", QSettings::User );
  QString outputPath = settings.readEntry( "outputDirectory" );

  if( outputPath.isEmpty() )
  {
    outputPath = QDir::homeDirPath();    
  }
  lddDirectoryOutputLineEdit->setText( outputPath );
  
  QString lastLDD = settings.readEntry("lastLDD");
  QString lastACU = settings.readEntry("lastACU");
  QString lastDRE = settings.readEntry("lastDRE");
  QString lastPoints = settings.readEntry("lastPoints");
  QString lastSUB = settings.readEntry("lastSUB");
  QString lastSEG = settings.readEntry("lastSEG");
  QString lastMINI = settings.readEntry("lastMINI");
  QString lastOUT = settings.readEntry("lastOUT");
  QString lastFactor = settings.readEntry("lastFactor");
  QString lastDEM = settings.readEntry("lastDEM");
  bool lastMinimize = settings.readEntry("lastMinimize").toInt();
  QString removeOutlierThreshold = settings.readEntry("removeOutlierThreshold");
  bool removeOutlier = settings.readEntry("removeOutlier").toInt();

  if( removeOutlierThreshold.isEmpty() )
  {
    removeOutlierThreshold = "90";
    removeOutlier = true;
  }

  int index;

  // acculator
  index = comboboxFind( accumulatorLDDComboBox, lastLDD );
  accumulatorLDDComboBox->setCurrentItem( index );
  accumulatorLDDComboBox_activated( accumulatorLDDComboBox->currentText() );

  // drainage
  index = comboboxFind( accumulatedComboBox, lastACU );
  accumulatedComboBox->setCurrentItem( index );
  accumulatedComboBox_activated( accumulatedComboBox->currentText() );

  // subwatersheds
  index = comboboxFind( delimiterLDDComboBox, lastLDD );
  delimiterLDDComboBox->setCurrentItem( index );

  index = comboboxFind( delimiterAcuComboBox, lastACU );
  delimiterAcuComboBox->setCurrentItem( index );

  index = comboboxFind( batchPointsComboBox, lastPoints );
  batchPointsComboBox->setCurrentItem( index );

  delimiterBoundingBoxCheckBox->setChecked( lastMinimize );

   // Cut
  index = comboboxFind( cutLimiterComboBox, lastSUB );
  cutLimiterComboBox->setCurrentItem( index );

  // Segmenter
  index = comboboxFind( segmenterLDDComboBox, lastLDD );
  segmenterLDDComboBox->setCurrentItem( index );

  index = comboboxFind( segmenterDrainageComboBox, lastDRE );
  segmenterDrainageComboBox->setCurrentItem( index );

  // MiniBasins
  index = comboboxFind( microBasinLDDComboBox, lastLDD );
  microBasinLDDComboBox->setCurrentItem( index );

  index = comboboxFind( microBasinSegmentedComboBox, lastSEG );
  microBasinSegmentedComboBox->setCurrentItem( index );

  // Subwatershed 2 Vector
  index = comboboxFind( maskComboBox, lastMINI );
  maskComboBox->setCurrentItem( index );

  if( index == 0 )
  {
    index = comboboxFind( maskComboBox, lastSUB );
     maskComboBox->setCurrentItem( index );
  }

  // Segmented 2 Vector
  index = comboboxFind( lddComboBox, lastLDD );
  lddComboBox->setCurrentItem( index );

  index = comboboxFind( segmentedComboBox, lastSEG );
  segmentedComboBox->setCurrentItem( index );

  // converter
  index = comboboxFind( converterInputLDDComboBox, lastLDD );
  converterInputLDDComboBox->setCurrentItem( index );

  // hand
  index = comboboxFind( lddComboBox_2, lastLDD );
  lddComboBox_2->setCurrentItem( index );

  index = comboboxFind( drainageComboBox, lastDRE );
  drainageComboBox->setCurrentItem( index );

  // upscale raster
  index = comboboxFind( upLDDLineEdit, lastLDD );
  upLDDLineEdit->setCurrentItem( index );

  index = comboboxFind( upAccumulatedLineEdit, lastACU );
  upAccumulatedLineEdit->setCurrentItem( index );

  // Length and Slope
  index = comboboxFind( lsLDDComboBox, lastLDD );
  lsLDDComboBox->setCurrentItem( index );

  index = comboboxFind( lsACUComboBox, lastACU );
  lsACUComboBox->setCurrentItem( index );

  index = comboboxFind( lsOutletComboBox, lastOUT );
  lsOutletComboBox->setCurrentItem( index );

  index = comboboxFind( lsDEMComboBox, lastDEM );
  lsDEMComboBox->setCurrentItem( index );

  lsFactorLineEdit->setText( lastFactor );

  // RemoveOutlier
  removeOutliersCheckBox->setChecked( removeOutlier );
  lowOutlierThresholdLineEdit->setText( removeOutlierThreshold );
}


void HidroMainWindow::flowInputLayerComboBox_activated( const QString &lddName )
{
  QString fixedName = lddName + "_Modified";
  flowFixedDEMLineEdit->setText( fixedName );
}


void HidroMainWindow::accumulatedComboBox_activated( const QString & )
{
  if(accumulatedComboBox->currentText().isEmpty())
	{		
		return;
	}

  TeLayer* inputLayer = NULL;

	HidroUtils utils(_parameters->getCurrentDatabasePtr());

	inputLayer = utils.getLayerByName(accumulatedComboBox->currentText().latin1());

	if(!inputLayer)
	{
		//QMessageBox::warning(this, tr("Warning"), tr("Error getting layer from database."));
		return;
	}

  std::string lddName = drainageNameLineEdit->text().latin1();
  
  if( !inputLayer->hasGeometry( TeRASTER ) )
  {
    //QMessageBox::warning(this, tr("Warning"), tr("inputLayer nao tem teraster."));
    return;
  }

  int threshold = (int)(inputLayer->raster()->params().vmax_[0] * 0.01);
  drainageThresholdLineEdit->setText( QString::number( threshold ) );
}


void HidroMainWindow::batchPointsToolButton_clicked()
{
  close();

  /*if(_batchPointsWindow)
	{
		delete _batchPointsWindow;
	}*/
  HidroBatchPointsWindow* batchPointsWindow = new HidroBatchPointsWindow((QWidget*)_parameters->qtmain_widget_ptr_, "Batch Points Window", false);  
	batchPointsWindow->init2(_parameters, true);
	batchPointsWindow->show();

  /*if( batchPointsWindow )
  {
    delete batchPointsWindow;
    batchPointsWindow = NULL;
  }*/
  
  //show();
}


void HidroMainWindow::removeOutliersCheckBox_stateChanged( int )
{
  if( removeOutliersCheckBox->isChecked())
  {
    textLabel1_28->setEnabled( true );
    lowOutlierThresholdLineEdit->setEnabled( true);
  }
  else
  {
    textLabel1_28->setEnabled( false );
    lowOutlierThresholdLineEdit->setEnabled( false );
  }
}
