#ifndef HIDRO_FIX_PFS_H_
#define HIDRO_FIX_PFS_H_

// Hidro template classes
#include <TePDIMatrix.hpp>
#include <HidroSmartQueue.h>

// foreigner classes
class TeDatabase;
class TeRasterParams;
class HidroFlowGrid;

class BreachingNode
{
public:
  unsigned int x_;
  unsigned int y_;
  double altimetria_;
};

/**
 * @class HidroFixPFS
 * @brief This class fix the pits of an LDD using the PFS (Priority Fisrst Serarch) algorithm (Sedgewick, 1988).
 * @author Alexandre Copertino Jardim <alexcj@dpi.inpe.br>
 * @ingroup hidroFlow The Group for generate flow functions.
**/
class HidroFixPFS
{
public:  
  /**
   * @brief Main Constructor.
   * @param database Database pointer.
   * @param rasterParams DEM parameters.
   * @param demMatrix DEM.
   * @param lddMatrix LDD.
   * @param lddDirectionsMap LDD convention.
  **/
  HidroFixPFS( HidroFlowGrid* flowGrid, TePDIMatrix<double>& demMatrix,
    TePDIMatrix<unsigned char>& lddMatrix,    
    std::string demName,
    std::string lddName, 
    bool cavePlaneAreas,
    bool singlePits,
    bool partialResult,
    bool removeOutliers,
    double lowOutlierThreshold,
    std::string outputDirectory );

  ~HidroFixPFS();

  /**
   * @brief Run the PFS (Priority Fisrst Serarch) algorithm to fix the pits (Sedgewick, 1988).
   * @return True for normal execution.
  **/
  bool pfs();  

  void fillPlaneAreas();  

  void solveSimplePits();  

  void removingDEMoutliers();

  std::string errorMessage_;

protected:
  /**
   * @brief Initialize the pit queue.
   * @return True for normal execution.
  **/
  bool initPitQueue();  

  /**
   * @brief Compute the mean altimetria of a window of 3x3, 5x5, 7x7, etc. centred in x, y;
   * @param x Window center column.
   * @param y Window center line.
   * @param windowSize Window size (3x3, 5x5, 7x7, ...).
   * @param includeCenter Include the center cell in the calculus.
   * @return The mean.
  **/
  double meanNeigbhors( unsigned int x, unsigned int y, int windowSize, bool includeCenter );

  /**
   *@brief Return the altimetria of the smallest neighbor of a cell.
   * @param line cell line number.
   * @param column cell column number.
   *@return the altimetria of the smallest neighbor
  **/
  float smallerNeighbor( unsigned int line, unsigned int column );

  /**
   * @brief Verify how many neighbors are pits.
   * @param x Column of the cell.
   * @param y Line of the cell.
   * @return Number of neighbors that are pits.
  **/
  int pitsNeighbors( unsigned int x, unsigned int y );

  /**
   * @brief Recompute the LDD for Neighborhood.
   * @param x Column of the cell.
   * @param y Line of the cell.
  **/
  void recomputeNeighborhoodLDD( unsigned int x, unsigned int y );

  /**
   * @brief Print final results in txt arquive.   
  **/
  void print();

  /**
   * @brief Check if is boundary region of fix method ( -4 lines and columns of DEM ).
   * @param x Column of the cell.
   * @param y Line of the cell.
   * @return True if is boundary region.
  **/
  bool boundaryFixRegion( unsigned int x, unsigned int y );

  bool boundaryRegion( unsigned int x, unsigned int y );

  /**
   * @brief Verify if it is a plane area (all the neighboors has the same altimetria).
   * @param x Column of the cell.
   * @param y Line of the cell.
   * @return True if is plane are.
  **/
  bool isPlaneBorder( unsigned int x, unsigned int y );

  // -- Time
  std::string starTime_;
  std::string endTime_;
  std::string partialTime_;

private:
  // -- Members.
  HidroFlowGrid* flowGrid_;
  TePDIMatrix<double>& demMatrix_;
  TePDIMatrix<unsigned char>& lddMatrix_;  

  // -- Fix parameters
  unsigned int pitMaxTreatment_;
  double boost_;

  // -- Raster limits.
  unsigned int rasterLines_;
  unsigned int rasterColumns_;
  bool hasDummy_;
  double demDummy_;


  // -- Caving Plane Areas.
  bool cavePlaneAreas_;
  double caving_increment_;
  double double_problem_increment_;

  bool singlePits_;
  bool partialResult_;
  std::string outputDirectory_;

  // -- Main Pit Queue.
  HidroSmartQueue pitQueue_;

  // low Outlier
  bool removeOutliers_;
  double lowOutlierThreshold_;
  
  // -- Queue controllers and final information.
  unsigned int initialSize_;  
  unsigned int solvedPitsNumber_;
  unsigned int addPitsNumber_;
  unsigned int unsolvedPitsNumber_;
  unsigned int simplePits_;

  // -- DEM and LDD output name
  std::string demName_;
  std::string lddName_;
  std::string queueStatisticsFileName_;
  std::string pitCorrectionStatisticsFileName_;

  // -- Estatisticas
  unsigned int double_problem_;
  unsigned int pits_in_path_;

  double path_length_mean_;
  double visited_number_mean_;
  unsigned int pits_solved_;
  unsigned int pits_solved_order_;
  unsigned int big_path_;
  unsigned int big_path_number_;  
  unsigned int unsolved_double_problem_;
  unsigned int max_path_leght_; // Limited by coumputer memory
  double border_path_gradient_;
  double thresholdAltimetria_;
};

#endif // HIDRO_FIX_PFS_H_