/******************************************************************************
* Hidro development team
* 
* Terra View Hidro Plugin
* 
* @(#) HidroFlowUtils.h
*
*******************************************************************************
*
* $Rev: 8883 $:
*
* $Author: alexcj $:
*
* $Date: 2010-08-30 17:16:16 -0300 (seg, 30 ago 2010) $:
*
******************************************************************************/

/*!
	\brief This is the Hidro Flow Group.
	@defgroup hidroFlow The Group for Plugin Flow Classes.
*/

#ifndef HIDRO_FLOW_UTILS_H_
#define HIDRO_FLOW_UTILS_H_

/** 
  * \file HidroFlowUtils.h
  *
  *
  * \brief This file has commung functions used by extract flow algorithms
  *
  * This file gives support to flow methods implemented in hidro plugin
  * Flow Methods:
  *                 DEM 8 Connected Single Flow
  *                 DEM 8 Connected Multi Flow
  *
  * \sa HidroFlowDEM8Connected.h
  *
  * \author Eric Silva Abreu <eric.abreu@funcate.org.br>
  * \version 1.0
  *
/*!
  @ingroup hidroFlow
 */


/*
** ----------------------------------------------------------------------------
** Includes:
*/

#include <string>
#include <vector>

class TeRaster;
class TeView;
class TeDatabase;
class TeLayer;
class TeTable;
class TePointSet;
#include <HidroMatrix.hpp>
#include <TePDIMatrix.hpp>
#include <HidroTime.h>

#define LDD_DUMMY 255

#define M_PI 3.14159265358979323846
#define M_SQRT2 1.41421356237309504880
// I don't know why I cant use math.h <alexandre - alexcj@dpi.inpe.br>

class QComboBox;
class QString;

class Cell
{
public:
  Cell();

  Cell( unsigned int line, unsigned int column, float accumulated, int subwatershed );

  // Sobrecarga do operador > (greater than) para utilizar a priority_queue
  bool operator>( const Cell &rightSide ) const; 

  unsigned int line_;
  unsigned int column_;
  float accumulated_;
  int subwatershed_;
};


int comboboxFind( QComboBox* combobox, QString text );

template <class T>
bool copyRaster2PDIMatrix( TeRaster* raster, TePDIMatrix<T>& matrix, unsigned char maxMemPercentUsage = 50  )
{
  unsigned int nlines = raster->params().nlines_;
  unsigned int ncolumns = raster->params().ncols_;

  matrix.Reset( nlines, ncolumns, HidroMatrix<T>::AutoMemPol, matrix.getMaxTmpFileSize(), maxMemPercentUsage );

  TeProgress::instance()->setTotalSteps( nlines );

  double val;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      raster->getElement( column, line, val );
      matrix[line][column] = (T) val;
    }
    TeProgress::instance()->setProgress( line );

    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return false;
    }
  }

  return true;
}

template <class T>
bool copyPDIMatrix2Raster( TePDIMatrix<T>& matrix, TeRaster* raster )
{
  unsigned int nlines = matrix.GetLines();
  unsigned int ncolumns = matrix.GetColumns();  

  TeProgress::instance()->setTotalSteps( nlines );

  double val;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      val = (double) matrix[line][column];
      raster->setElement( column, line, val );
    }
    TeProgress::instance()->setProgress( line );

    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return false;
    }
  }

  return true;
}

namespace HidroFlowUtils 
{
  /* \brief Recalculate the min and max values of a raster.
   */
  void recalculateRasterMinMax( TeRaster* raster );
}

/** \brief Find the minimun slope 
    \params altimetriaNeighbors[8]		Grid values
    \params lddDirectionsMap[8]	      	Output raster values
    \params maxSlopeDirection			    Output raster lddDirectionsMap. Auxiliar variable used if exist only one direction.
    \return			                  		Lowest lddDirectionsMap in the grid
    */
double minValue(double altimetriaNeighbors[8], unsigned char lddDirectionsMap[8], unsigned char& minSlopeDirection);

double maxValue(double altimetriaNeighbors[8], unsigned char lddDirectionsMap[8], unsigned char& maxSlopeDirection);
	 
/** \brief Finds the number of existing directions in the grid
    \params altimetriaNeighbors[8]		Grid values
    \params vmin			Lowest lddDirectionsMap
    \return					Number of directions in the grid
    */
int ndirectionCompute(double altimetriaNeighbors[8], double vmin);

/** \brief Compute the altimetria mean in a 3x3 region of center cell
	\param raster 			Input raster
	\param gridcol			Column of the center cell
	\param gridlin			Line of the center cell
	\return					    Returns the altimetria mean in a 3x3 region of center cell
	*/
double altimetriaMeanCompute ( HidroMatrix<double>& raster, int gridlin, int gridcol );//(TeRaster* raster, int gridcol, int gridlin, int grid, int i);

double mediana(std::vector<double> vals, int grid);

/** \brief Function used to generate a layer with points of pits.
  */
bool generatePtsPits(TeDatabase* db, TeView* view, TeRaster* outputRaster, std::string layerName, std::string lddName);

/** \brief Creates attributes table.
  */
bool createTable(TeDatabase* db, TeTable& pointTable, std::string layerName);

/** \brief Insert values in the table.
  */
bool valuesTable(TeLayer* layer, TeTable& pointTable, TePointSet ptset);

/** \brief Saves layer in database.
  */
bool saveLayer(TeDatabase* db, TeLayer* layer, TeTable& pointTable);

/** \brief Compute slope angle values for neighborhood
	\param inputRaster	      	Input parameter, terralib raster attribute, this raster could be any regular grid
	\param col	    		      	Current column from inputRaster
	\param lin			          	Current line from inputRaster
  \param slopeAngle           slope angle values for neighborhood
  \return				            	True if center cell is not dummy and False if center cell is dummy (can't compute the slope).
	*/
bool slopeCompute( HidroMatrix<double>& inputRaster, int col, int lin, double slopeAngle[8] );

/** \brief Check if its a border cell.
  \param demMatrix DEM
  \param col Cell X coordinate.
  \param lin Cell Y coordinate.
 */
bool isBorder( HidroMatrix<double>& demMatrix, unsigned int col, unsigned int lin );

/**
  \brief Convert LDD.
  \param lddInputName Input LDD name (layer).
  \param inputLDDmap Input LDD code.
  \param lddOutputName Output LDD name (layer).
  \param outputLDDmap Output LDD code.
 */
bool convertLDD( TeDatabase* db, std::string lddInputName, int inputLDDmap[10], std::string lddOutputName, int outputLDDmap[10], int dataType,bool useDummy );
/*
** ---------------------------------------------------------------
** End:
*/

#endif //#define HIDRO_FLOW_UTILS_H_