#ifndef HIDRO_MATRIX_H_
#define HIDRO_MATRIX_H_

#include <TePDIMatrix.hpp>

#include <TeDatabase.h>
#include <TeRaster.h>
#include <TeRasterParams.h>
#include <TeImportRaster.h>

#include <qmessagebox.h>

#define LDD_DUMMY 255

#define HIDRO_MATRIX_MAX_PERCENT_USAGE 10

/**
  * @class HidroMatrix
  * @brief This is the template class to deal with a generic matrix extends the TePDIMatrix.
  * @author Alexandre Copertino Jardim <alexcj@dpi.inpe.br>
  * @ingroup hidroFlow The Group for generate flow functions.
  */
template< class T >
class HidroMatrix : public TePDIMatrix<T>
{
public:
  /**
    * @brief Default Constructor.
    * @note The default mamory policy is RAMMemPol.
    */
  HidroMatrix();    

  /**
    * @brief Default Destructor
    */
  virtual ~HidroMatrix();

  /**
    * @brief Alternative Constructor.
    *
    * @param lines Number of lines.
    * @param columns Number of columns.
    * @param raster TeRaster that will be converted in HidroMatrix.
    * @param dataType TeUNSIGNEDCHAR for LDD TeFLOAT for DEM.
    * @note The default mamory policy is RAMMemPol.
    */
  HidroMatrix( unsigned int lines, unsigned int columns, TeRasterParams& rasterParams );

  HidroMatrix( unsigned int lines, unsigned int columns );

  HidroMatrix( TeRaster* raster );

  void reset( unsigned int lines, unsigned int columns, TeRasterParams& rasterParams );

  void setDummy( T dummy );      

  T getDummy();

  bool hasDummy();

  bool getElement( unsigned int col, unsigned int line, T &val );

  void setElement( unsigned int col, unsigned int line, T val );

  //! Close and Save the matrix in database as TeRaster.
  bool save( TeDatabase* db, std::string layerName, TeDataType dataType, bool useMultiResolution = false );

  bool saveTiff( std::string layerName, TeDataType dataType );

  void freeMemory();

  TeRasterParams* rasterParams_;

protected:
  T dummy_;
  bool hasDummy_;        
};

template< class T >
HidroMatrix< T >::HidroMatrix() : TePDIMatrix<T>()
{
  hasDummy_ = false;
}

// Main Constructor used in TeHidro
template< class T >
HidroMatrix< T >::HidroMatrix( unsigned int lines, unsigned int columns, TeRasterParams& rasterParams )
{
  Reset( lines, columns, TePDIMatrix<T>::AutoMemPol, TePDIMatrix<T>::maxTmpFileSize_, HIDRO_MATRIX_MAX_PERCENT_USAGE );

  hasDummy_ = false;

  // Copy the raster parameters because the raster will be deleted
  rasterParams_ = new TeRasterParams(rasterParams);
}


template< class T >
HidroMatrix< T >::HidroMatrix( unsigned int lines, unsigned int columns )
{
  Reset( lines, columns, TePDIMatrix<T>::AutoMemPol, TePDIMatrix<T>::maxTmpFileSize_, HIDRO_MATRIX_MAX_PERCENT_USAGE );
  rasterParams_ = NULL;
  hasDummy_ = false;
}

template< class T >
HidroMatrix< T >::HidroMatrix( TeRaster* raster ) : rasterParams_(0)
{
  // Copy the raster parameters because the raster can be deleted
  // Its necessary to save the raster  
  rasterParams_ = new TeRasterParams( raster->params() );

  Reset( rasterParams_->nlines_, rasterParams_->ncols_, TePDIMatrix<T>::AutoMemPol, TePDIMatrix<T>::maxTmpFileSize_, HIDRO_MATRIX_MAX_PERCENT_USAGE );  

  // Set the matrix Dummy
  if( raster->params().useDummy_ )
  {
    this->setDummy( (T)raster->params().dummy_[0] );
  }
  else
  {
    hasDummy_ = false;
  }

  // Copy the raster values
  double rasterVal;  
  for( unsigned int line=0; line<TePDIMatrix<T>::totalLines_; line++ )
  {
    for( unsigned int column=0; column<TePDIMatrix<T>::totalColumns_; column++ )
    {
      raster->getElement( column, line, rasterVal );
      TePDIMatrix<T>::getScanLine( line )[ column ] = (T)rasterVal;      
    }    
  }
}

template< class T >
void  HidroMatrix< T >::reset( unsigned int lines, unsigned int columns, TeRasterParams& rasterParams )
{
  Reset( lines, columns, TePDIMatrix<T>::AutoMemPol, TePDIMatrix<T>::maxTmpFileSize_, HIDRO_MATRIX_MAX_PERCENT_USAGE );

  if( rasterParams.useDummy_ )
  {
    setDummy( (T)rasterParams.dummy_[0] );
  }
  else
  {
    hasDummy_ = false;
  }

  // Copy the raster parameters because the raster will be deleted
  rasterParams_ = new TeRasterParams(rasterParams);
}


template< class T >
  HidroMatrix< T >::~HidroMatrix()
{
  if(rasterParams_)
    delete rasterParams_;
}


template< class T >
void HidroMatrix< T >::setDummy( T dummy )
{
  dummy_ = dummy;
  hasDummy_ = true;
}


template< class T >
T HidroMatrix< T >::getDummy()
{
  return dummy_;
}


template< class T >
bool HidroMatrix< T >::hasDummy()
{
  return hasDummy_;
}


template< class T >
bool HidroMatrix< T >::getElement( unsigned int column, unsigned int line, T &val )
{
  val = TePDIMatrix<T>::getScanLine( line )[ column ];

  if( val == dummy_ )
    return false;

  return true;
}


template< class T >
void HidroMatrix< T >::setElement( unsigned int column, unsigned int line, T val )
{
  TePDIMatrix<T>::getScanLine( line )[ column ] = val;
}


template< class T >
void HidroMatrix< T >::freeMemory()
{
  TePDIMatrix<T>::clear();
}


template< class T >
bool HidroMatrix< T >::save( TeDatabase* db, std::string layerName, TeDataType dataType, bool useMultiResolution )
{
  // check if layer name already exits in the database
  int i = 0;
	while( db->layerExist(layerName) )
	{
		i++;
		layerName = layerName + "_" + Te2String(i);
	}

  // raster params
  TeRasterParams params( *rasterParams_ );
  params.decoderIdentifier_ = "SMARTMEM";
  params.mode_ = 'w';
  
  if( hasDummy_ )
  {
    params.useDummy_ = true;
    params.setDummy( dummy_ );    
  }
  else
  {
    params.useDummy_ = false;
  }

  params.setDataType( dataType );

  TeRaster* raster = new TeRaster( params );
  
  // verify if output raster created is valid
  if( !raster->init() )
  {
    QMessageBox::warning(0, "Warning", "Error create TeRaster.");
  	delete raster;
	  return false;
  }

  // transform in TeRaster
  unsigned int rasterColumns = rasterParams_->ncols_; // number of columns from input raster
  unsigned int rasterLines = rasterParams_->nlines_; // number of rows from input raster

  T rasterValue;  
  T vmax;
  T vmin;
  if( dataType == TeUNSIGNEDCHAR )
  {
    vmax = 0;
    vmin = 255;
  }
  else if( dataType == TeINTEGER )
  {
    vmax = -2147483648;
    vmin =  2147483647;
  }
  else
  {
    vmax = -TeMAXFLOAT;
    vmin =  TeMAXFLOAT;
  }
  
  for( unsigned int lin = 0; lin < rasterLines; ++lin )
	{
		for( unsigned int col = 0; col < rasterColumns; ++col )
    {			
      if( getElement(col, lin, rasterValue) )
      {        
        if(rasterValue > vmax) vmax = rasterValue;
        if(rasterValue < vmin) vmin = rasterValue;
        raster->setElement( col, lin, rasterValue );
      }
      else
      {
        raster->setElement( col, lin, dummy_ );
      }
    }
  }
  raster->params().vmax_[0] = vmax;
  raster->params().vmin_[0] = vmin;

  TeLayer* layer = new TeLayer(layerName, db, params.projection() );
	if (layer->id() <= 0)
  {
		delete layer;
		delete raster;
		return false;
	}

  unsigned int blockW = rasterColumns;
	unsigned int blockH = rasterLines;
  if( raster->params().ncols_ > 512 )
	{
		blockW = 512;	
	}
	if( raster->params().nlines_ > 512 )
  {
    blockH = 512;
  }
  
	if (!TeImportRaster(layer, raster, blockW, blockH, TeRasterParams::TeNoCompression, 
		"01", raster->params().dummy_[0], raster->params().useDummy_,  TeRasterParams::TeNoExpansible))
  {
		delete layer;
		delete raster;
		return false;
	}

  // cria multiresoluo para dados grandes
  /*if( useMultiResolution )
  if( rasterColumns*rasterLines > 1000*1000 )
  {    
    vector<int> extraRes;
    unsigned int i;
    unsigned int nivel = 1;
    for (i = 500*500; i < rasterColumns*rasterLines; i=i*4 ) 
    {
      extraRes.push_back(nivel);
      nivel++;
    }

		TeProgress::instance()->reset();	
		
    // build extra resolutions
		for (i=0; i<extraRes.size();i++)
		{
			if(TeProgress::instance())
			{
				QString caption = "Creating multiresolution";
				TeProgress::instance()->setCaption(caption.latin1());
				QString msg = "Level " + QString("%1").arg(extraRes[i]);
				msg += ". Please, wait!";
				TeProgress::instance()->setMessage(msg.latin1());
			}
			int resFac = (int)pow(2.0,extraRes[i]);      
			if (!TeBuildLowerResolution(layer,raster,resFac,"01"))
			{
				QMessageBox::warning(0, "Warning",
					"Fail to build the resolution!");
				break;
			}
		}
		if (TeProgress::instance())
			TeProgress::instance()->reset();
  } /**/ 

  raster->clear();
  delete raster;
  raster = 0;

  return true;
}

template< class T >
bool HidroMatrix< T >::saveTiff( std::string layerName, TeDataType dataType )
{
  TeRasterParams* pTiff = new TeRasterParams();
  pTiff->fileName_ = layerName;
	pTiff->nBands(1);
	pTiff->setDataType(dataType);
  pTiff->boxLinesColumns( rasterParams_->box().x1(), rasterParams_->box().y1(), rasterParams_->box().x2(), rasterParams_->box().y2(), rasterParams_->nlines_, rasterParams_->ncols_); 
  pTiff->mode_ = 'c';

  if( hasDummy_ )
  {
    pTiff->useDummy_ = true;
    pTiff->setDummy( dummy_ );    
  }
  else
  {
    pTiff->useDummy_ = false;
  }

  pTiff->setDataType( dataType );  

  TeRaster* raster = new TeRaster( *pTiff );
  
  // verify if output raster created is valid
  if( !raster->init() )
  {
    //QMessageBox::warning(0, "Warning", "Error create TeRaster.");
    cout << raster->errorMessage()<< endl;
  	delete raster;
	  return false;
  }

  // transform in TeRaster
  unsigned int rasterColumns = rasterParams_->ncols_; // number of columns from input raster
  unsigned int rasterLines = rasterParams_->nlines_; // number of rows from input raster

  T rasterValue;
  T vmax = (T)-TeMAXFLOAT;
  T vmin =  (T)TeMAXFLOAT;
  
  for( unsigned int lin = 0; lin < rasterLines; ++lin )
	{
		for( unsigned int col = 0; col < rasterColumns; ++col )
    {			
      if( getElement(col, lin, rasterValue) )
      {        
        if(rasterValue > vmax) vmax = rasterValue;
        if(rasterValue < vmin) vmin = rasterValue;
        raster->setElement( col, lin, rasterValue );
      }
      else
      {
        raster->setElement( col, lin, dummy_ );
      }
    }
  }
  raster->params().vmax_[0] = vmax;
  raster->params().vmin_[0] = vmin;    

  unsigned int blockW = rasterColumns;
	unsigned int blockH = rasterLines;
  if( raster->params().ncols_ > 512 )
	{
		blockW = 512;	
	}
	if( raster->params().nlines_ > 512 )
  {
    blockH = 512;
  }  
  
  delete raster;
  raster = 0;
  delete pTiff;
  return true;
}


#endif // HIDRO_MATRIX_H_
