// Formulas
// http://www.soi.city.ac.uk/~jwo/phd/

#include <HidroSlope.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

HidroSlope::HidroSlope( TeRaster* demRaster, TeRaster* &slopeRaster ) :
HidroFlowAlgorithm( demRaster ),
demRaster_(demRaster),
slopeRaster_(slopeRaster)
{
}

HidroSlope::~HidroSlope()
{
  // Free Memory
  demMatrix_.clear();
  slopeMatrix_.clear();
}

bool
HidroSlope::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();  

  // load data

  // load DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load DEM data.");    
  if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output Hand raster
  
  // slopeRaster params
  TeRasterParams slopeRasterParams = demRaster_->params();
  
  // Set dummy
  slopeRasterParams.setDummy( -9999 );
  slopeRasterParams.useDummy_ = true;  

  // Change mode
  slopeRasterParams.mode_ = 'w';
  slopeRasterParams.decoderIdentifier_ = "SMARTMEM";
  slopeRasterParams.setDataType( TeFLOAT );

  // Set Max and Minimal values
  slopeRasterParams.vmax_[0] = -TeMAXFLOAT;
  slopeRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  slopeRaster_ = new TeRaster( slopeRasterParams );

  // verify if slopeRaster created is valid
  if( !slopeRaster_->init() )
  {
    errorMessage_ = slopeRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  rasterDummy_ = demRaster_->params().dummy_[0];

  // STEP 1
  TeProgress::instance()->setMessage("Creating Slope Raster step 1 from 2.");  

  // compute slope matrix
  if( demRaster_->params().projection()->name().compare( "LatLong" ) == 0 )
  {
    if( !latLongSlope() )
    {
      return false;
    }
  }
  else
  {
    if( !metricSlope() )
    {
      return false;
    }
  }  

  // STEP 2
  TeProgress::instance()->setMessage("Creating Slope Raster step 2 from 2.");  
  
  // copy matrix values to raster.
  if( !copyPDIMatrix2Raster( slopeMatrix_, slopeRaster_ ) )
  {
    return cancel();
  }

  // Free Memory
  demMatrix_.clear();
  slopeMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool
HidroSlope::latLongSlope()
{
  // Formulas
  // http://www.soi.city.ac.uk/~jwo/phd/


  unsigned int nlines = slopeRaster_->params().nlines_;
  unsigned int ncolumns = slopeRaster_->params().ncols_;  

  // Set progress bar
  TeProgress::instance()->setTotalSteps( nlines );

  double A = slopeRaster_->params().projection()->datum().radius() / 1000.0;
  double F = slopeRaster_->params().projection()->datum().flattening();  

  //double A = 6378.137; //!COMPRIMENTO DO SEMI EIXO MAIOR DO ELIPSIDE (KM)
	//double B = 6356.752; //!COMPRIMENTO DO SEMI EIXO MENOR DO ELIPSIDE (KM)
  //double F = ( A - B ) / A; //!ACHATAMENTO DO ELIPSIDE
	
  double E2 = 2 * F - F * F; //!QUADRADO DA EXCENTRICIDADE

  double resx = slopeRaster_->params().resx_;
  double resy = slopeRaster_->params().resy_;

  slopeMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, slopeMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );

  // fill slope matrix with dummyes
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      slopeMatrix_[line][column] = -9999;
    }
  }

  //=B8*SEN(3/3600*PI()/180)*COS((0)*PI()/180)
  double DY = (A * sin( resy * M_PI/180 )) * 1000;  // em metros
  
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    // A rea varia somente na lina / y / latitude
    
    TeCoord2D index( 0, line );
    TeCoord2D coord = slopeRaster_->index2Coord( index );

    double YLAT = coord.y();        
    
    double RN =  A / sqrt( ( 1 - E2 * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) ); //!RAIO DE CURVATURA DA TERRA NA LATITUDE

    // =B8*SEN(3/3600*PI()/180)*COS((A8+1.5/3600)*PI()/180)
    double DX = (RN * sin( resx * M_PI/180 ) * cos( YLAT * M_PI/180 )) * 1000; // em metros

    for( unsigned int column=1; column<ncolumns-1; column++ )
    {      
      if( demMatrix_[line][column] != rasterDummy_ )
      {
        double z1 = 0, z2 = 0, z3 = 0, z4 = 0, z6 = 0, z7 = 0, z8 = 0, z9 = 0;

        if( demMatrix_[line-1][column-1] != rasterDummy_ )
        {
          z1 = demMatrix_[line-1][column-1];
        }

        if( demMatrix_[line-1][column] != rasterDummy_ )
        {
          z2 = demMatrix_[line-1][column];
        }

        if( demMatrix_[line-1][column+1] != rasterDummy_ )
        {
          z3 = demMatrix_[line-1][column+1];
        }

        if( demMatrix_[line][column-1] != rasterDummy_ )
        {
          z4 = demMatrix_[line][column-1];
        }

        if( demMatrix_[line][column+1] != rasterDummy_ )
        {
          z6 = demMatrix_[line][column+1];
        }

        if( demMatrix_[line+1][column-1] != rasterDummy_ )
        {
          z7 = demMatrix_[line+1][column-1];
        }

        if( demMatrix_[line+1][column] != rasterDummy_ )
        {
          z8 = demMatrix_[line+1][column];
        }

        if( demMatrix_[line+1][column+1] != rasterDummy_ )
        {
          z9 = demMatrix_[line+1][column+1];
        }

        double d = (z3+z6+z9 - z1-z4-z7) / (6*DX);

        double e = (z1+z2+z3 - z7-z8-z9) / (6*DY);

        slopeMatrix_[line][column] = (float) sqrt( d*d + e*e );// / 100;
      }      
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  return true;
}

bool
HidroSlope::metricSlope()
{
  // Formulas
  // http://www.soi.city.ac.uk/~jwo/phd/


  unsigned int nlines = slopeRaster_->params().nlines_;
  unsigned int ncolumns = slopeRaster_->params().ncols_;  

  // Set progress bar
  TeProgress::instance()->setTotalSteps( nlines );  

  double resx = slopeRaster_->params().resx_; //Gd
  double resy = slopeRaster_->params().resy_; //Ge

  slopeMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, slopeMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );

  // fill slope matrix with dummyes
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      slopeMatrix_[line][column] = -9999;
    }
  }  
  
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
    {      
      if( demMatrix_[line][column] != rasterDummy_ )
      {
        double z1 = 0, z2 = 0, z3 = 0, z4 = 0, z6 = 0, z7 = 0, z8 = 0, z9 = 0;

        if( demMatrix_[line-1][column-1] != rasterDummy_ )
        {
          z1 = demMatrix_[line-1][column-1];
        }

        if( demMatrix_[line-1][column] != rasterDummy_ )
        {
          z2 = demMatrix_[line-1][column];
        }

        if( demMatrix_[line-1][column+1] != rasterDummy_ )
        {
          z3 = demMatrix_[line-1][column+1];
        }

        if( demMatrix_[line][column-1] != rasterDummy_ )
        {
          z4 = demMatrix_[line][column-1];
        }

        if( demMatrix_[line][column+1] != rasterDummy_ )
        {
          z6 = demMatrix_[line][column+1];
        }

        if( demMatrix_[line+1][column-1] != rasterDummy_ )
        {
          z7 = demMatrix_[line+1][column-1];
        }

        if( demMatrix_[line+1][column] != rasterDummy_ )
        {
          z8 = demMatrix_[line+1][column];
        }

        if( demMatrix_[line+1][column+1] != rasterDummy_ )
        {
          z9 = demMatrix_[line+1][column+1];
        }

        double d = (z3+z6+z9 - z1-z4-z7) / (6*resx);

        double e = (z1+z2+z3 - z7-z8-z9) / (6*resy);

        slopeMatrix_[line][column] = (float) sqrt( d*d + e*e );
      }      
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  return true;
}

bool
HidroSlope::cancel()
{
  // Free Memory
  demMatrix_.clear();
  slopeMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}