#include <HidroSubwatershed.h>

#include <HidroFlowUtils.h>
#include <HidroMatrix.hpp>

HidroSubWatershed::HidroSubWatershed( 
  TeRaster* lddRaster,
  TeRaster* acuRaster,
  TePointSet& pointSet,
  TeRaster* &subWatershedRaster,
  bool minimizeBoundingBox ) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_( lddRaster ),
    acuRaster_(acuRaster),
    pointSet_( pointSet ),
    subWatershedRaster_( subWatershedRaster ),
     minimizeBoundingBox_( minimizeBoundingBox )
{  
}

bool
HidroSubWatershed::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load ACU
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load accumulated data.");
  if( !copyRaster2PDIMatrix( acuRaster_, accumulatedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output SubWatershed raster
  
  // subWatershedRaster params
  TeRasterParams subWatershedRasterParams = lddRaster_->params();
  
  // Set dummy
  subWatershedRasterParams.setDummy( -9999 );
  subWatershedRasterParams.useDummy_ = true;  

  // Change mode
  subWatershedRasterParams.setDataType( TeINTEGER );
  subWatershedRasterParams.mode_ = 'w';
  subWatershedRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  subWatershedRasterParams.vmax_[0] = -TeMAXFLOAT;
  subWatershedRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  subWatershedRaster_ = new TeRaster( subWatershedRasterParams );

  // verify if subWatershedRaster created is valid
  if( !subWatershedRaster_->init() )
  {
    errorMessage_ = subWatershedRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = subWatershedRaster_->params().nlines_;
  unsigned int ncolumns = subWatershedRaster_->params().ncols_;

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating SubWatersheds Raster step 1 from 5.");
  TeProgress::instance()->setTotalSteps( nlines );  

  // Init the subWatershede matrix
  subWatershedMatrix_.Reset( nlines, ncolumns, TePDIMatrix<int>::AutoMemPol, subWatershedMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( lddMatrix_[line][column] == 255 )
      {
        subWatershedMatrix_[line][column] = -9999;        
      }
      else
      {
      // flag until dummy
      subWatershedMatrix_[line][column] = -9998;
      }
      // refresh progress bar    
      TeProgress::instance()->setProgress( line );
      
      // check for cancel
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
    }
  }

  // mark outlet pixels in accumulated order
  for( unsigned int i=0; i<pointSet_.size(); ++i )
  {
    TeCoord2D coord = lddRaster_->coord2Index( pointSet_[i][0] );
    SubPoint SubPoint( (unsigned int)TeRound(coord.y_),
                             (unsigned int)TeRound(coord.x_),
                             accumulatedMatrix_[(unsigned int)TeRound(coord.y_)][(unsigned int)TeRound(coord.x_)] );
    SubPointsQueue_.push( SubPoint );
  }
  int subNumber = 1;
  while( !SubPointsQueue_.empty() )
  {
    SubPoint SubPoint = SubPointsQueue_.top();
    subWatershedMatrix_[SubPoint.line_][SubPoint.column_] = subNumber;
    SubPointsQueue_.pop();    
    subNumber++;
  }

  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating SubWatersheds Raster step 2 from 5.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  if( !initConnections() )
  {
    return false;
  }

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating SubWatersheds Raster step 3 from 5.");
  TeProgress::instance()->setProgress( 0 );

  // lao principal
  unsigned int lineTo;
  unsigned int columnTo;
  unsigned int lineFrom;
  unsigned int columnFrom;
  
  // paint this segment
  unsigned int lineStart;
  unsigned int columnStart;
  unsigned int lineStop;
  unsigned int columnStop;

  int subWatershedValue;

  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineTo = line;
        columnTo = column;
        
        lineStart = line;
        columnStart = column;

        subWatershedValue = -9999;

        bool find = false;

        // find the subwatershed point or dummy
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;

          lineStop = lineFrom;
          columnStop = columnFrom;

          if( subWatershedMatrix_[lineFrom][columnFrom] == -9999 )
          {
            subWatershedValue = -9999; // dummy
            paint( lineStart, columnStart, lineStop, columnStop, subWatershedValue );
            find = true;
            break;
          }

          // find one subwatershed
          if( subWatershedMatrix_[lineFrom][columnFrom] != -9998 )
          {
            find = true;
            subWatershedValue = subWatershedMatrix_[lineFrom][columnFrom];
            
            //paint
            paint( lineStart, columnStart, lineStop, columnStop, subWatershedValue );
            lddCode2LineColumn( lineTo, columnTo, lineStart, columnStart );
            if( subWatershedMatrix_[lineStart][columnStart] !=  -9998 )
            {
              break;
            }
          }
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );

        if( !find )
        {
          paint( lineStart, columnStart, lineStop, columnStop, -9999 );
        }
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao    
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory  
  lddMatrix_.clear(); 
  
  if( minimizeBoundingBox_ )
  {
    // STEP 4

    // compres the subWatershedMatrix
    TeProgress::instance()->setMessage("Creating SubWatershed Raster step 4 from 5.");
    if( !compres() )
    {
      return cancel();
    }

    // Free Memory 
    subWatershedMatrix_.clear();

    // STEP 5

    // copy matrix values to raster.
    TeProgress::instance()->setMessage("Creating SubWatershed Raster step 5 from 5.");
    if( !copyPDIMatrix2Raster( subWatershedMatrixCompressed_, subWatershedRaster_ ) )
    {
      return cancel();
    }

    // Free Memory  
    subWatershedMatrixCompressed_.clear();
  }
  else
  {
    // STEP 5

    // copy matrix values to raster.
    TeProgress::instance()->setMessage("Creating SubWatershed Raster step 5 from 5.");
    if( !copyPDIMatrix2Raster( subWatershedMatrix_, subWatershedRaster_ ) )
    {
      return cancel();
    }

    // Free Memory 
    subWatershedMatrix_.clear();
  }

  
  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;  
}

void
HidroSubWatershed::paint(const unsigned int &lineStart,
      const unsigned int &columnStart,
      const unsigned int &lineEnd,
      const unsigned int &columnEnd, 
      int subWatershedValue)
{
  // fill the path with subwatershed value
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo = lineStart;
  unsigned int columnTo = columnStart;
  do
  {
    lineFrom = lineTo;
    columnFrom = columnTo;

    subWatershedMatrix_[lineFrom][columnFrom]= subWatershedValue;    

    if( lineEnd == lineFrom && columnEnd == columnFrom )
    {
      break;
    }
  }
  while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );
}


bool
HidroSubWatershed::compres()
{
  unsigned int nlines = subWatershedMatrix_.GetLines();
  unsigned int ncolumns = subWatershedMatrix_.GetColumns();

  // barra de progresso  
  TeProgress::instance()->setTotalSteps( nlines * 2 );

  // Get Max and Min x and y
  unsigned int column2 = 0;
  unsigned int column1 = ncolumns;
  unsigned int line1 = 0;
  unsigned int line2 = nlines;  

  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( subWatershedMatrix_[line][column] != -9999 )
      {
        if( column2 < column) column2 = column;
        if( column1 > column) column1 = column;
        if( line1 < line) line1 = line;
        if( line2 > line) line2 = line;
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );
    
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // New line and column numbers
  unsigned int newlines = line1 - line2 + 1;
  unsigned int newcolumns = column2 - column1 + 1;

  // New bounding box
  TeBox boindingBox = lddRaster_->params().boundingBox();
  double resx = lddRaster_->params().resx_;
  double resy = lddRaster_->params().resy_;

  // x1'
  double x1 = boindingBox.x1_ + ( column1 * resx );

  // y1'
  double y1 = boindingBox.y2_ - ( ( line1 + 1 ) * resy );

  // x2'
  double x2 = boindingBox.x1_ + ( (column2 + 1) * resx );

  // y2'
  double y2 = boindingBox.y2_ - ( line2 * resy );

  // New matrix  
  subWatershedMatrixCompressed_.Reset( newlines, newcolumns, TePDIMatrix<int>::AutoMemPol, subWatershedMatrixCompressed_.getMaxTmpFileSize(),  MATRIX_MAX_PERCENT_USAGE );
  
  subWatershedRaster_->params().boundingBoxLinesColumns( x1, y1, x2, y2, newlines, newcolumns );

  // Copy Values
  for( unsigned int line=0; line<newlines; line++ )
  {
    for( unsigned int column=0; column<newcolumns; column++ )
    {
      subWatershedMatrixCompressed_[line][column] = subWatershedMatrix_[line+line2][column+column1];
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line + nlines_ );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // finaliza barra de progresso
  TeProgress::instance()->reset();
  return true;
}

bool HidroSubWatershed::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  subWatershedMatrix_.clear();
  subWatershedMatrixCompressed_.clear();

  return HidroFlowAlgorithm::cancel();
}

SubPoint::SubPoint()
{
}

SubPoint::SubPoint( unsigned int line, unsigned int column, float accumulated ) :
line_(line),
column_(column),
accumulated_(accumulated)
{
}

bool
SubPoint::operator>( const SubPoint &rightSide ) const
{
  return ( this->accumulated_ > rightSide.accumulated_ );
}
