#include <HidroBlocks.h>

#include <stdio.h>

#include <TeRaster.h>


HidroBlocks::HidroBlocks(  TeRaster* demRaster,
    TeRaster* soilsRaster,
    TeRaster* vegetationRaster,
    std::string cellFileName,
    TeRaster* &blocksGridRaster,
    std::string blocksFileName,
    std::string groupsFileName
    ) : 
  HidroFlowAlgorithm( demRaster ),
    demRaster_( demRaster ),
    soilsRaster_( soilsRaster ),
    vegetationRaster_( vegetationRaster ),
    cellFileName_( cellFileName ),
    blocksGridRaster_( blocksGridRaster ),
    blocksFileName_( blocksFileName ),
    groupsFileName_(groupsFileName)

{
}

HidroBlocks::~HidroBlocks()
{
  if( blocksFile_ != NULL )
  {
    fclose( blocksFile_ );
    blocksFile_ = NULL;
  }

  if( groupsFile_ != NULL )
  {
    fclose( groupsFile_ );
    groupsFile_ = NULL;
  }

  if( cellFile_ != NULL )
  {
    fclose( cellFile_ );
    cellFile_ = NULL;
  }
}

bool
HidroBlocks::execute()
{
  // records start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // create output blocks raster  
  // blocksRaster params
  TeRasterParams blocksRasterParams = demRaster_->params();
  
  // Set dummy
  blocksRasterParams.setDummy( -9999 );
  blocksRasterParams.useDummy_ = true;  

  // Change mode
  blocksRasterParams.mode_ = 'w';
  blocksRasterParams.decoderIdentifier_ = "SMARTMEM";
  blocksRasterParams.setDataType( TeINTEGER );

  // Set Max and Minimal values
  blocksRasterParams.vmax_[0] = -TeMAXFLOAT;
  blocksRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  blocksGridRaster_ = new TeRaster( blocksRasterParams );

  // verify if drainageRaster created is valid
  if( !blocksGridRaster_->init() )
  {
    errorMessage_ = blocksGridRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // create blocks Grid
  // used in for's and progress bar
  unsigned int nlines = blocksGridRaster_->params().nlines_;
  unsigned int ncolumns = blocksGridRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating blocks raster.");
  TeProgress::instance()->setTotalSteps( nlines );

  double soilsValue;
  double vegetationValue;
  int blockIdentifier;

  // The blocks group list
  BlocksGroups blocksGroup;

  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      // get the lat/long of the demRaster
      TeCoord2D demIndex( column, line );
      TeCoord2D centerDEMcoord = demRaster_->index2Coord( demIndex );      

      //centerDEMcoord.x_ = centerDEMcoord.x_ + demRaster_->params().resx_/2;
      //centerDEMcoord.y_ = centerDEMcoord.y_ + demRaster_->params().resy_/2;

      // get the index of soilsRaster
      TeCoord2D soilsIndex = soilsRaster_->coord2Index( centerDEMcoord );
      soilsIndex.x_ = (int)abs( soilsIndex.x_ );
      soilsIndex.y_ = (int)abs( soilsIndex.y_ );

      // get the index of vegetationRaster
      TeCoord2D vegetationIndex = vegetationRaster_->coord2Index( centerDEMcoord );
      vegetationIndex.x_ = (int)abs( vegetationIndex.x_ );
      vegetationIndex.y_ = (int)abs( vegetationIndex.y_ );

      if( soilsRaster_->getElement( soilsIndex.x(), soilsIndex.y(), soilsValue ) )
      {
        if( vegetationRaster_->getElement( vegetationIndex.x(), vegetationIndex.y(), vegetationValue ) )
        {
          // get the block identifier
          if(soilsValue!=0)
          {
            blockIdentifier = blocksGroup.insert( (int)soilsValue, (int)vegetationValue );
            blocksGridRaster_->setElement( column, line, blockIdentifier );
          }
        }
      }      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }


  return true;
}

bool
HidroBlocks::cancel()
{
  return HidroFlowAlgorithm::cancel();
}


Block::Block( int soils, int vegetation ) :
  soils_(soils),
  vegetation_(vegetation)
{
}

bool
Block::operator==( const Block &rightSide ) const
{
  if( this->soils_ == rightSide.soils_ )
    if( this->vegetation_ == rightSide.vegetation_ )
      return true;

  return false;
}

int
BlocksGroups::insert( int soils, int vegetation )
{
  int i;
  Block block( soils, vegetation );

  // the vector is empty
  if( blocksVector_.empty() )
  {
    // insert the first block
    blocksVector_.push_back( block );
    return 0;
  }
  else
  {
    for( i=0; i < blocksVector_.size(); ++i )
    {
      // alredy exists
      if( blocksVector_[i] == block )
      {
        return i;
      }
    }
    // not exists in vector yet
    blocksVector_.push_back( block );
    return i;
  }
}
