#include <HidroDrainage.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

HidroDrainage::HidroDrainage(
  TeRaster* accumulatedRaster,
  TeRaster* &drainageRaster,
  float threshold ) :
    HidroFlowAlgorithm( accumulatedRaster ),
    accumulatedRaster_(accumulatedRaster),
    drainageRaster_(drainageRaster),
    threshold_(threshold)
{  
}

HidroDrainage::~HidroDrainage()
{   
  accumulatedMatrix_.clear();
  drainageMatrix_.clear();
}


bool
HidroDrainage::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // load data  

  // load Accumulated
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Accumulated data.");    
  if( !copyRaster2PDIMatrix( accumulatedRaster_, accumulatedMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output drainage raster
  
  // drainageRaster params
  TeRasterParams drainageRasterParams = accumulatedRaster_->params();
  
  // Set dummy
  drainageRasterParams.setDummy( 255 );
  drainageRasterParams.useDummy_ = true;  

  // Change mode
  drainageRasterParams.mode_ = 'w';
  drainageRasterParams.decoderIdentifier_ = "SMARTMEM";
  drainageRasterParams.setDataType( TeUNSIGNEDCHAR );

  // Set Max and Minimal values
  drainageRasterParams.vmax_[0] = -TeMAXFLOAT;
  drainageRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  drainageRaster_ = new TeRaster( drainageRasterParams );

  // verify if drainageRaster created is valid
  if( !drainageRaster_->init() )
  {
    errorMessage_ = drainageRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = drainageRaster_->params().nlines_;
  unsigned int ncolumns = drainageRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");

  // STEP 1
  TeProgress::instance()->setMessage("Creating Drainage Raster step 1 from 3.");
  TeProgress::instance()->setTotalSteps( nlines );

  // incializa matrix de drenagem com dummys (255)
  drainageMatrix_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, drainageMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      drainageMatrix_[line][column] = (unsigned char) 255;     
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // STEP 2
  TeProgress::instance()->setMessage("Creating Drainage Raster step 2 from 3.");
  TeProgress::instance()->setTotalSteps( nlines );

  // find the drainage
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( accumulatedMatrix_[line][column] > threshold_ )
      {
        drainageMatrix_[line][column] = 1;
      }      
    }
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory  
  accumulatedMatrix_.clear();

  // STEP 4

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Creating Drainage Raster step 3 from 3.");
  if( !copyPDIMatrix2Raster( drainageMatrix_, drainageRaster_ ) )
  {
    return cancel();
  }

  // Free Memory 
  drainageMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroDrainage::cancel()
{
  // Free Memory  
  accumulatedMatrix_.clear();
  drainageMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}