#include <HidroFlowGrid.h>

#include <HidroFixPFS.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>

#include <qmessagebox.h>

HidroFlowGrid::HidroFlowGrid( TeRaster* demRaster,
    TeRaster* &lddRaster,    
    TeRaster* &fixedDemRaster,
    bool cavePlaneAreas,
    bool singlePits,
    bool partialResult,
    bool removeOutliers,
    double lowOutlierThreshold,
    std::string partialResultName,
    std::string outputDirectory ) :
      demRaster_(demRaster),      
      lddRaster_(lddRaster),
      fixedDemRaster_(fixedDemRaster),
      cavePlaneAreas_(cavePlaneAreas),
      singlePits_(singlePits),
      partialResult_(partialResult),
      removeOutliers_(removeOutliers),
      lowOutlierThreshold_(lowOutlierThreshold),
      partialResultName_(partialResultName),
      outputDirectory_(outputDirectory)
{
  if( demRaster_->params().useDummy_ )
  {
    hasDummy_ = true;
    demDummy_ = demRaster_->params().dummy_[0];
  }
  else
  {
    hasDummy_ = false;
    demDummy_ = -TeMAXFLOAT;
  }

  // monta bonitinho o nome dos resultados parciais  
  partialResultFullName_.clear();
  partialResultFullName_ = outputDirectory_;
  partialResultFullName_.append( "\\" );
  partialResultFullName_.append( partialResultName_ );
  partialResultFullName_.append(".tif");  
}

bool
HidroFlowGrid::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // load DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load DEM Grid data.");
  if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output LDD raster  
  // lddRaster params
  TeRasterParams lddRasterParams = demRaster_->params();
  
  // Set dummy
  lddRasterParams.setDummy( 255 );
  lddRasterParams.useDummy_ = true;  

  // Change mode
  lddRasterParams.setDataType( TeUNSIGNEDCHAR );
  lddRasterParams.mode_ = 'w';
  lddRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  lddRasterParams.vmax_[0] = -TeMAXFLOAT;
  lddRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  lddRaster_ = new TeRaster( lddRasterParams );

  // verify if lddRaster created is valid
  if( !lddRaster_->init() )
  {
    errorMessage_ = lddRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // create output DEM raster  
  // lddRaster params
  TeRasterParams fixedDemRasterParams = demRaster_->params();
  
  // Set dummy
  //fixedDemRasterParams.setDummy( -9999 );
  //fixedDemRasterParams.useDummy_ = true;  

  // Change mode
  fixedDemRasterParams.setDataType( TeFLOAT );
  fixedDemRasterParams.mode_ = 'w';
  fixedDemRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  fixedDemRasterParams.vmax_[0] = -TeMAXFLOAT;
  fixedDemRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  fixedDemRaster_ = new TeRaster( fixedDemRasterParams );

  // verify if fixedDemRaster created is valid
  if( !fixedDemRaster_->init() )
  {
    errorMessage_ = fixedDemRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // used in for's and progress bar
  unsigned int demLines = demRaster_->params().nlines_;
  unsigned int demColumns = demRaster_->params().ncols_;

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Flow Extraction step 1 from 4.");
  TeProgress::instance()->setTotalSteps( demLines );

  #ifdef GENERATE_FLOW_STATISTICS
    // Reset statistics of the flow generation
    HidroGenerateFlowStatistics::instance().reset();
  #endif

  //  LDD Directions Map
  //  032 | 064 | 128
  //  ----+-----+----
  //  016 | 000 | 001
  //  ----+-----+----
  //  008 | 004 | 002
  // the max value if was multi direction is 1+2+4+8+16+32+64+128 = 255 
  // the value 0 is a pit
  // the value 255 is dummy ??? the same of max value, we will need to change it if want to work if multi-directions  
  lddDirectionsMap_[0] = 32;
  lddDirectionsMap_[1] = 64;
  lddDirectionsMap_[2] = 128;
  lddDirectionsMap_[3] = 16;
  lddDirectionsMap_[4] = 1;
  lddDirectionsMap_[5] = 8;
  lddDirectionsMap_[6] = 4;
  lddDirectionsMap_[7] = 2;

  // init ldd matrix with dummys (255)
  lddMatrix_.Reset( demLines, demColumns,
    TePDIMatrix<unsigned char>::AutoMemPol
    , lddMatrix_.getMaxTmpFileSize(),
    MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line = 0; line < demLines; line++ )
	{
		for( unsigned int column = 0; column < demColumns; column++ )
		{
      lddMatrix_[line][column] = LDD_DUMMY;
    }
  }

  // For each cell compute the direction,
  // excluding first and last lines and columns
	for( unsigned int line = 1; line < demLines - 1; line++ )
	{
		for( unsigned int column = 1; column < demColumns - 1; column++ )
		{      
			if ( !d8( line, column ) )
			{
        QMessageBox::warning(0, "Warning", "Error in Flow Extraction.");
				
				return false;
			}      
		}

    // Update the progress windows
		if( TeProgress::instance() )
		{
			TeProgress::instance()->setProgress( line );
      
      if( TeProgress::instance()->wasCancelled() )
      {
        QMessageBox::warning(0, "Warning", "The operation Flow Extraction was cancelled by the user.");
			  
			  return false;
      }
		}
	}

  #ifdef GENERATE_FLOW_STATISTICS
    // Record the statistics of the flow generation in a file before correct the pits.
    std::string generateFlowStatisticsFileName = layerName + "_" + lddName + "_Moving_Means_Stats.txt";
    HidroGenerateFlowStatistics::instance().print( generateFlowStatisticsFileName.c_str() );
  #endif

  // Reset the progress windows
	if( TeProgress::instance() )
	{
		TeProgress::instance()->reset();
	}  

  // Fix the pits!
  HidroFixPFS hidroFixPFS( this, demMatrix_, lddMatrix_, this->demRaster_->params().fileName_, this->lddRaster_->params().fileName_, cavePlaneAreas_, singlePits_, partialResult_, removeOutliers_, lowOutlierThreshold_, outputDirectory_ );
  if( !hidroFixPFS.pfs() )
	{
    errorMessage_ = hidroFixPFS.errorMessage_;
    //QMessageBox::warning(0, "Warning", "Error while correcting LDD raster.");
		return false;
	}

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Flow Extraction step 3 from 4.");
  if( !copyPDIMatrix2Raster( lddMatrix_, lddRaster_ ) )
  {
    return cancel();
  }
  lddMatrix_.clear();

  // copy matrix values to raster.
  TeProgress::instance()->setMessage("Flow Extraction step 4 from 4.");
  if( !copyPDIMatrix2Raster( demMatrix_, fixedDemRaster_ ) )
  {
    return cancel();
  }
  demMatrix_.clear();

  // Time Statistics
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();
    
  return true;
}

// Deterministic 8 (D8) Unidirectional choose by Moving Means algorithm for each cell.
bool
HidroFlowGrid::d8( unsigned int line, unsigned int column )
{
  // Compute the neighborhood slopes  
  double slopeNeighbors[8];
  if( !this->slopeCompute( line, column, slopeNeighbors ) ) 
	{
    lddMatrix_[line][column] = LDD_DUMMY; // LDD Dummy value
    return true;
  }

  // find the maximum slope and maximum slope direction (last if there are mutiple directions).
	unsigned char maxSlopeDirection;
	double maxSlope = this->maxSlope( slopeNeighbors, maxSlopeDirection );

  // if there is a pit 
  if( maxSlope <= 0 )
  //if( maxSlope < 1e-10F ) try zero pits in float dem fixed
  {
		lddMatrix_[line][column] = 0;
    return true;
  }

  // find direction(s) number of max slope(s)	
	int ndirections = this->ndirectionCompute( slopeNeighbors, maxSlope );

  // if there are single direction
	if( ndirections == 1 )
  {
		lddMatrix_[line][column] = maxSlopeDirection;
    return true;
  }
  
  //if there are multiples directions choose by Moving Means
  if( ndirections > 1 )
  {
    // check if the cell is in boudary region of LDD ( -2 lines and columns of DEM )
		int currentNeighbor = 0; // used to know what neighbor we are looking			
    double altimetriaMean[8], minmedia;

    // Compute the start and end of window (3x3)
    int startColumn = column - 1;
    int endColumn = column + 2;
    int startLine = line - 1;
    int endLine = line + 2;

    // For each neighbor of center cell
		for( int currentNeighborLine = startLine; currentNeighborLine < endLine; currentNeighborLine++ )
		{			
      for( int currentNeighborColumn = startColumn; currentNeighborColumn < endColumn; currentNeighborColumn++ )
			{
        // its is need because the relation of slopeNeighbors[currentNeighbor] and currentNeighbor.
        if( currentNeighborLine != line || currentNeighborColumn != column )
        {          
          if( slopeNeighbors[currentNeighbor] == maxSlope )  // only for candidate cells to be maximum slope
					{
            // if the neighbor is boder it's the direction
            if( this->isBorder( currentNeighborLine, currentNeighborColumn ) )
            {
              maxSlopeDirection = lddDirectionsMap_[currentNeighbor];
              lddMatrix_[line][column] = maxSlopeDirection;

              return true;
            }

            // Compute the altimetria mean 3x3 for each neighbor and store in altimetriaMean
						altimetriaMean[currentNeighbor] = this->altimetriaMeanCompute( currentNeighborLine, currentNeighborColumn );
					}
					else
          {
						altimetriaMean[currentNeighbor] = TeMAXFLOAT; // dummy value for mean.
          }
					currentNeighbor++;
        }
			}
    }
			
    // find the minimum altimetria mean and this direction (that will be the maxSlopeDirection).
		minmedia = minSlope( altimetriaMean, maxSlopeDirection );
		
    lddMatrix_[line][column] = maxSlopeDirection;
    return true;
	}
  
  return false;
}

bool 
HidroFlowGrid::slopeCompute( unsigned int line, unsigned int column, double slopeNeighbors[8] )
{  
  double altimetriaCenter;  

  // altimetria in the center cell
  if( demMatrix_[line][column] == demDummy_ )
  {
    return false;
  }
  else
  {
    altimetriaCenter = demMatrix_[line][column];
  }

  //upper left
	if( demMatrix_[line-1][column-1] == demDummy_ )
	{		
		slopeNeighbors [0] = -1;
	}
	else
  {
    slopeNeighbors[0] = (altimetriaCenter - demMatrix_[line-1][column-1]) / M_SQRT2;
  }

  //up
	if( demMatrix_[line-1][column] == demDummy_ )
	{		
		slopeNeighbors[1] = -1;
	}
	else
  {
    slopeNeighbors [1] = altimetriaCenter - demMatrix_[line-1][column];
  }

  //upper right
	if( demMatrix_[line-1][column+1] == demDummy_ )
	{		
		slopeNeighbors[2] = -1;
	}
	else
  {
    slopeNeighbors[2] = (altimetriaCenter - demMatrix_[line-1][column+1]) / M_SQRT2;
  }

  //left
	if( demMatrix_[line][column-1] == demDummy_ )
	{		
		slopeNeighbors[3] = -1;
	}
	else
  {
    slopeNeighbors[3] = altimetriaCenter - demMatrix_[line][column-1];
  }

  //right
	if( demMatrix_[line][column+1] == demDummy_ )
	{		
		slopeNeighbors[4] = -1;
	}
	else
  {
    slopeNeighbors[4] = altimetriaCenter - demMatrix_[line][column+1];
  }

  //lower left
	if( demMatrix_[line+1][column-1] == demDummy_ )
	{
		slopeNeighbors[5] = -1;
	}
	else
  {
    slopeNeighbors[5] = (altimetriaCenter - demMatrix_[line+1][column-1]) / M_SQRT2;
  }

  //lower   
	if( demMatrix_[line+1][column] == demDummy_ )
	{
		slopeNeighbors[6] = -1;
	}
	else
  {
    slopeNeighbors[6] = altimetriaCenter - demMatrix_[line+1][column];
  }

  //lower right
	if( demMatrix_[line+1][column+1] == demDummy_ )
	{
		slopeNeighbors[7] = -1;
	}
	else
  {
    slopeNeighbors[7] = (altimetriaCenter - demMatrix_[line+1][column+1] ) / M_SQRT2;
  }

  return true;
}


double
HidroFlowGrid::altimetriaMeanCompute( unsigned int line, unsigned int column )
{
  double altimetria = 0.0;
  double media = 0.0;
	int ncell = 0;

  // Compute the start and end of window (3x3)
  unsigned int startColumn = column - 1;
  unsigned int endColumn = column + 2;
  unsigned int startLine = line - 1;
  unsigned int endLine = line + 2;

  // Verify Borders. The TePDIMatrix dont do it!
  if( startColumn < 0 )
    startColumn = 0;
  if( endColumn > demMatrix_.GetColumns() )
    endColumn = demMatrix_.GetColumns();
  if( startLine < 0 )
    startLine = 0;
  if( endLine > demMatrix_.GetLines() )
    endLine = demMatrix_.GetLines();

	for( unsigned int l = startLine; l < endLine; l++ )
	{		
		for( unsigned int c = startColumn; c < endColumn; c++ )
		{
      altimetria = demMatrix_[l][c];      
			if( altimetria != demDummy_ )
			{
				media += altimetria;
				ncell += 1; //number of cells in the grid
			}			
		}		
	}

	//Calculates media from grid
	media = media / ncell;

	return media;
}

double 
HidroFlowGrid::maxSlope( double slopeNeighbors[8], unsigned char& maxSlopeDirection )
{
	double maxSlope = slopeNeighbors[0];
	maxSlopeDirection = lddDirectionsMap_[0];

	for( int i = 1; i < 8; i++ )
	{
		if( slopeNeighbors[i] > maxSlope )
		{
			maxSlope = slopeNeighbors[i];
			maxSlopeDirection = lddDirectionsMap_[i];
		}
	}

  return maxSlope;
}

int
HidroFlowGrid::ndirectionCompute( double altimetriaCenter[8], double maxSlope )
{
  int ndirections = 0;

  for( int i = 0; i < 8; i++ )
  {
    if( maxSlope == altimetriaCenter[i] )
    {
      ndirections++;
    }
  }

  return ndirections;
}

bool
HidroFlowGrid::isBorder( unsigned int line, unsigned int column )
{
  if( column < 1 || column > (demMatrix_.GetColumns() - 2) ||
    line < 1 || line > (demMatrix_.GetLines() - 2) )
  {
    return true;
  }
  else if( hasDummy_ )
  {
    // Compute the start and end of window (3x3)
    unsigned int startColumn = column - 1;
    unsigned int endColumn = column +  2;
    unsigned int startLine = line - 1;
    unsigned int endLine = line + 2;    

	  for (unsigned int l = startLine; l < endLine; l++)
	  {				
		  for (unsigned int c = startColumn; c < endColumn; c++)
		  {
        if( demMatrix_[l][c] == demDummy_ )
        {          
          return true;
        }
      }
    }
  }

  return false;
}

double
HidroFlowGrid::minSlope( double slopeNeighbors[8], unsigned char& minSlopeDirection )
{
  double vmin = TeMAXFLOAT;
  minSlopeDirection = 0; //used if exist only one direction
	  
  for (int i = 0; i < 8; i++)
  {
	  if( slopeNeighbors[i] != -1 )
	  {
		  if( slopeNeighbors[i] < vmin)
		  {
			  vmin = slopeNeighbors[i];
			  minSlopeDirection = lddDirectionsMap_[i];
		  }
	  }
  }
  
  return vmin;
}

bool
HidroFlowGrid::saveTiff()
{
  TeRasterParams* pTiff = new TeRasterParams();
  pTiff->projection( demRaster_->params().projection() );  
  pTiff->fileName_ = partialResultFullName_;
	pTiff->nBands(1);
	pTiff->setDataType(TeFLOAT);
  pTiff->boxLinesColumns( demRaster_->params().box().x1(),
    demRaster_->params().box().y1(),
    demRaster_->params().box().x2(),
    demRaster_->params().box().y2(),
    demRaster_->params().nlines_,
    demRaster_->params().ncols_); 
  pTiff->mode_ = 'c';

  // Set Max and Minimal values
  pTiff->vmax_[0] = -TeMAXFLOAT;
  pTiff->vmin_[0] =  TeMAXFLOAT;

  if( hasDummy_ )
  {
    pTiff->useDummy_ = true;
    pTiff->setDummy( demDummy_ );    
  }
  else
  {
    pTiff->useDummy_ = false;
  }

  // transform in TeRaster
  unsigned int rasterColumns = demRaster_->params().ncols_; // number of columns from input raster
  unsigned int rasterLines = demRaster_->params().nlines_; // number of rows from input raster

  unsigned int blockW = rasterColumns;
	unsigned int blockH = rasterLines;
  if( rasterColumns > 512 )
	{
		blockW = 512;	
	}
	if( rasterLines > 512 )
  {
    blockH = 512;
  }  

  pTiff->blockHeight_ = blockH;
  pTiff->blockWidth_ = blockW;

  TeRaster* raster = new TeRaster( *pTiff );
  
  // verify if output raster created is valid
  if( !raster->init() )
  {
    //QMessageBox::warning(0, "Warning", "Error create TeRaster.");
    //cout << raster->errorMessage()<< endl;
    errorMessage_ = "Error to save partial DEM result.";
  	delete raster;
	  return false;
  }  

  double rasterValue;
  //double vmax = (double)-TeMAXFLOAT;
  //double vmin = (double) TeMAXFLOAT;
  
  for( unsigned int lin = 0; lin < rasterLines; ++lin )
	{
		for( unsigned int col = 0; col < rasterColumns; ++col )
    {
      rasterValue = demMatrix_[lin][col];
      //if(rasterValue > vmax) vmax = rasterValue;
      //if(rasterValue < vmin) vmin = rasterValue;
      raster->setElement( col, lin, rasterValue );    
    }
  }
  //raster->params().vmax_[0] = vmax;
  //raster->params().vmin_[0] = vmin;

  delete raster;
  raster = 0;
  delete pTiff;

  return true;
}

bool
HidroFlowGrid::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  demMatrix_.clear();

  errorMessage_= "The operation was canceled by user.";
  
  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();
  
  return false;
}

const std::string& 
HidroFlowGrid::getErrorMessage()
{
  return errorMessage_;
}

const std::string& 
HidroFlowGrid::getTimeStart()
{
  return timeStart_;
}

const std::string& 
HidroFlowGrid::getTimeEnd()
{
  return timeEnd_;
}

const std::string& 
HidroFlowGrid::getTimeTotal()
{
  return timeTotal_;
}