#include <HidroFlowPath.h>

#include <HidroFlowUtils.h>
#include <HidroPersister.h>

#include <TeRaster.h>
#include <TeGeometry.h>
#include <TeDatabase.h>
#include <TeLayer.h>

#include <TePDIMatrix.hpp>

bool
HidroFlowPath::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // STEP 1
  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Flow Path");	  
  TeProgress::instance()->setMessage("Step 1 from : load flow grid data");
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }  

  // cria o layer de saida
  TeProjection* proj = TeProjectionFactory::make( lddRaster_->projection()->params() );
	std::string layerName = outputName_;
	TeLayer* outputLayer = new TeLayer( layerName, database_, proj );

  // Cria a tabela
  TeTable table;
  TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.numChar_ = 16;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.name_ = "linkColumn";
	TeAttributeList attrList;
	attrList.push_back(at);  
  
  std::string tableName = outputName_;
  tableName.append("_attrTable");
	table.name( tableName );
	table.setAttributeList(attrList);  
	table.setLinkName("linkColumn");
	table.setUniqueName("linkColumn");
	database_->validTable(table);
	
  if(database_->tableExist(tableName))
	{
		errorMessage_ = "Attribute table already exist.";
		return false;
	}
  if(!database_->createTable(tableName, attrList))
	{
		errorMessage_ = database_->errorMessage();
		return false;
	}

   // STEP 2
  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Flow Path");	  
  TeProgress::instance()->setMessage("Step 2 from : draw the lines");
  TeProgress::instance()->setTotalSteps( pointSet_.size() );
  // mark outlet pixels in ac
  for( unsigned int i=0; i<pointSet_.size(); ++i )
  {
    TeCoord2D coord = lddRaster_->coord2Index( pointSet_[i][0] );
    unsigned int line = (unsigned int)TeRound(coord.y_);
    unsigned int column = (unsigned int)TeRound(coord.x_);
    
    if( lddMatrix_[line][column] != 255 && lddMatrix_[line][column] != 0 )
    {
      unsigned int lineFrom;
      unsigned int columnFrom;
      unsigned int lineTo;
      unsigned int columnTo;      
      TeCoord2D fromMatrix;
      TeCoord2D fromWorld;
      TeCoord2D toMatrix;
      TeCoord2D toWorld;
      // Conjunto de linhas
      TeLineSet lineSet;
      bool flag = false;

      lineFrom = line;
      columnFrom = column;
          
      fromMatrix.setXY(column, line);
      fromWorld = lddRaster_->index2Coord(fromMatrix);      
      
      // faz at achar a saida ou proximo segmento
      do
      {
        // verifica se  sada
        if( !lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {              
          break;
        }

        // Cria os pontos
        if( lddMatrix_[lineFrom][columnFrom] != lddMatrix_[lineTo][columnTo] )
        {              
          toMatrix.setXY(columnTo, lineTo);
          toWorld = lddRaster_->index2Coord(toMatrix);

          TeLine2D line2D;
          line2D.add( fromWorld );
          line2D.add( toWorld );                   

          lineSet.add( line2D );

          fromMatrix.setXY(columnTo, lineTo);
          fromWorld = lddRaster_->index2Coord(fromMatrix);
		      flag = true;
        }

        lddMatrix_[lineFrom][columnFrom] = 255;

        lineFrom = lineTo;
        columnFrom = columnTo;          
      }
      while( lddMatrix_[lineTo][columnTo] != 255 && lddMatrix_[lineTo][columnTo] != 0 );
      
      lineSet.objectId( Te2String( i, 0 ) );
      outputLayer->addLines( lineSet );
      TeTableRow row;
      row.push_back( lineSet.objectId() );
      row.push_back( lineSet.objectId() );
      table.add(row);
    }       

    TeProgress::instance()->setProgress( i );
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      errorMessage_= "Flow Paths was canceled by user.";
      // finaliza barra de progresso
      TeProgress::instance()->reset();
      // guarda o tempo decorrente do processo
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return false;
    }
  }  
  
  // Salva o Layer
  if(!database_->insertTable(table))
	{
		errorMessage_ = database_->errorMessage();
		return false;
	}
  HidroPersister persister(database_);
	if(!persister.saveLayer(outputLayer, table, TeLINES))
	{
		return false;
	}  

  // finaliza barra de progresso
  TeProgress::instance()->reset();

  // guarda a hora final do processo
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();  

  return true;
}

HidroFlowPath::HidroFlowPath( TeRaster* lddRaster,
    TePointSet& pointSet,
    std::string& outputName,
    TeDatabase* database) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    pointSet_(pointSet),
    outputName_(outputName),
    database_(database)
{
}


HidroFlowPath::~HidroFlowPath()
{  
  lddMatrix_.clear();
  connectionsMatrix_.clear();
}

bool
HidroFlowPath::cancel()
{
  // Free Memory  
  lddMatrix_.clear();
  connectionsMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}
