#include <HidroFlowUtils.h>

#include <TeDatabase.h>
#include <TeRaster.h>
#include <TeView.h>
#include <TeRasterRemap.h>
#include <TeDecoderMemoryMap.h>
#include <TeImportRaster.h>
#include <TeLayer.h>
#include <TeProgress.h>

#include <iostream>
#include <fstream>
#include <ctime>
#include <cstdio>
#include <cstdlib>

#include <HidroPersister.h>
#include <HidroFlowD8.h>
#include <HidroUtils.h>
#include <HidroMatrix.hpp>

#include <qmessagebox.h>
#include <qcombobox.h>
#include <qstring.h>

Cell::Cell()
{
}

Cell::Cell( unsigned int line, unsigned int column, float accumulated, int subwatershed ) :
line_(line),
column_(column),
accumulated_(accumulated),
subwatershed_(subwatershed)
{
}

bool
Cell::operator>( const Cell &rightSide ) const
{
  if ( this->subwatershed_ > rightSide.subwatershed_  )
  {
    return true;
  }

  if ( this->subwatershed_ == rightSide.subwatershed_  )
  {
    return (  this->accumulated_ > rightSide.accumulated_ );
  }

  return false;
}

int comboboxFind( QComboBox* combobox, QString text )
{
  int size = combobox->count();

  for( int i=0; i<size; ++i)
  {
    combobox->setCurrentItem( i );
    QString currentText = combobox->currentText();
    if( currentText.compare( text ) == 0 ) //find it!
    {
      return i;
    }
  }

  return 0;
}

bool convertLDD( TeDatabase* db, std::string lddInputName, int inputLDDmap[10], std::string lddOutputName, int outputLDDmap[10], int dataType, bool useDummy )
{
  // 1- open Input LDD
  TeLayer* inputLayer = NULL;
  HidroUtils utils( db );
  inputLayer = utils.getLayerByName( lddInputName );
  if( !inputLayer )
  {
    QMessageBox::warning( 0, "Warning", "Error getting layer from database." );
    return false;
  }
  if( !inputLayer->hasGeometry( TeRASTER ) )
  {
    QMessageBox::warning( 0, "Warning", "InputLayer nao tem teraster." );
    return false;
  }   
  TeRaster* inputLDDRaster = inputLayer->raster();
  TeRasterParams inputParams = inputLDDRaster->params();

  // 2- create Output LDD
  // check if layer name already exits in the database
  int i = 0;
  while( db->layerExist( lddOutputName ) )
  {
    i++;
    lddOutputName = lddOutputName + "_" + Te2String(i);
  }   
  TeRasterParams params( inputParams );
  params.decoderIdentifier_ = "SMARTMEM";   
  params.mode_ = 'w';
  if( useDummy )
  {
    params.useDummy_ = true;
    params.setDummy( outputLDDmap[9] );      
  }
  else
  {
    params.useDummy_ = false;
  }
  
  switch ( dataType )
  {
    case 1: // TerraHidro
      params.setDataType( TeUNSIGNEDCHAR ); 
      break;
    case 2: // Hande
      params.setDataType( TeUNSIGNEDCHAR ); 
      break;
    case 3: // MGB
      params.setDataType( TeSHORT );
      break;
    default: // como eu no sei os valores maximo e minimo
      params.setDataType( TeINTEGER );
      break;
  }  

  TeRaster* outputLDDRaster = new TeRaster( params );
  // verify if output raster created is valid
  if( !outputLDDRaster->init() )
  {
    QMessageBox::warning(0, "Warning", "Error create TeRaster.");
    return false;
  }

  // 3- Convert
  int rasterColumns = params.ncols_;
  int rasterLines = params.nlines_;

  double input;
  int output;

  // Progress Bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Convert the LDD.");
  TeProgress::instance()->setTotalSteps( rasterLines );

  for( int line = 0; line < rasterLines; line++ )
  {
    for( int column = 0; column < rasterColumns; column++ )
    {
        if( inputLDDRaster->getElement( column, line, input ) )
        {
          int i = 0;
          while( (int)input != inputLDDmap[i] && i < 9 )
          {               
              i++;
          }
          output = outputLDDmap[i];
        }
        else
        {
          output = outputLDDmap[9];            
        }
        outputLDDRaster->setElement( column, line, output );
    }
    TeProgress::instance()->setProgress( line );
  }

  // 4- Save the output raster in database
  TeLayer* layer = new TeLayer(lddOutputName, db, params.projection() );
  if (layer->id() <= 0)
  {
	  delete layer;
	  delete outputLDDRaster;
	  return false;
  }

  unsigned int blockW = params.blockWidth_;
  unsigned int blockH = params.blockHeight_;

  if (!TeImportRaster(layer, outputLDDRaster, blockW, blockH, outputLDDRaster->params().compression_[0], 
	  "", outputLDDRaster->params().dummy_[0], outputLDDRaster->params().useDummy_, outputLDDRaster->params().tiling_type_))
  {
	  delete layer;
	  delete outputLDDRaster;
	  return false;
  }

  // Finishi progress bar
  TeProgress::instance()->reset();

  return true;
}


double minValue(double slopeNeighbors[8], unsigned char lddDirectionsMap[8], unsigned char& minSlopeDirection)
{
  double vmin = TeMAXFLOAT;
  minSlopeDirection = 0; //used if exist only one direction
	  
  for (int i = 0; i < 8; i++)
  {
	  if( slopeNeighbors[i] != -1 )
	  {
		  if( slopeNeighbors[i] < vmin)
		  {
			  vmin = slopeNeighbors[i];
			  minSlopeDirection = lddDirectionsMap[i];
		  }
	  }
  }
  
  return vmin;
}


double maxValue(double slopeNeighbors[8], unsigned char lddDirectionsMap[8], unsigned char& maxSlopeDirection)
{
	double maxSlope = slopeNeighbors[0];
	maxSlopeDirection = lddDirectionsMap[0];

	for (int i = 1; i < 8; i++)
	{
		if (slopeNeighbors[i] > maxSlope)
		{
			maxSlope = slopeNeighbors[i];
			maxSlopeDirection = lddDirectionsMap[i];
		}
	}

  return maxSlope;
}


int ndirectionCompute( double altimetriaCenter[8], double maxSlope )
{
  int ndirections = 0;

  for( int i = 0; i < 8; i++ )
  {
    if( maxSlope == altimetriaCenter[i] )
		ndirections++;
  }

  return ndirections;
}


double altimetriaMeanCompute( HidroMatrix<double>& demMatrix, int gridlin, int gridcol)
{	
	double altimetria = 0.0, media = 0.0;
	int ncell = 0;

  // Compute the start and end of window (3x3)
  unsigned int startColumn = gridcol - 1;
  unsigned int endColumn = gridcol + 2;
  unsigned int startLine = gridlin - 1;
  unsigned int endLine = gridlin + 2;

  // Verify Borders. The HidroMatrix dont do it!
  if( startColumn < 0 )
    startColumn = 0;
  if( endColumn > demMatrix.GetColumns() )
    endColumn = demMatrix.GetColumns();
  if( startLine < 0 )
    startLine = 0;
  if( endLine > demMatrix.GetLines() )
    endLine = demMatrix.GetLines();

	for( unsigned int l = startLine; l < endLine; l++ )
	{		
		for( unsigned int c = startColumn; c < endColumn; c++ )
		{			
			if( demMatrix.getElement( c, l, altimetria ) )
			{
				media += altimetria;
				ncell += 1;//number of cells in the grid
			}			
		}		
	}

	//Calculates media from grid
	media = media / ncell;

	return media;
}


bool generatePtsPits(TeDatabase* db, TeView* view, TeRaster* raster, std::string layerName, std::string lddName)
{
	int i = 1;
	int nlines = raster->params().nlines_;
	int ncols = raster->params().ncols_;

	TePointSet ptset;

	for (int line = 2; line < nlines-2; line++)
	{
		for (int col = 2; col < ncols-2; col++)
		{
			double altimetriaCenter;
			raster->getElement(col, line, altimetriaCenter);

			if (altimetriaCenter == 0)
			{
				TePoint pt;
				TeCoord2D coord = raster->index2Coord(TeCoord2D(col, line));
				pt.add(coord);
				ptset.add(pt);
			}
		}
	}

  if( ptset.empty() )
    return false;

	TeLayer* layer = NULL;

	std::string layerPtsName = layerName + "_" + lddName + "_pits";
	TeProjection* proj = raster->projection();
	while (db->layerExist(layerPtsName) == true)
	{
		i++;
		layerPtsName += "_" + i;
	}

	layer= new TeLayer(layerPtsName, db, proj);

	if (!layer)
	{
		return false;
	}

	TeTable pointTable;
		
	if (!createTable(db, pointTable, layerPtsName))
	{
		return false;
	}

	if (!valuesTable(layer, pointTable, ptset))
	{
		return false;
	}

	if (!saveLayer(db, layer, pointTable))
	{
		return false;
	}

	HidroPersister hidro(db);

    hidro.createTheme(layer->name(), "", view, layer);

	return true;
}


bool createTable(TeDatabase* db, TeTable& pointTable, std::string layerName)
{
	std::string tablePointName = layerName + "_Table";

	TeAttributeList attrList;

	TeAttribute id;
	id.rep_.type_ = TeSTRING;
	id.rep_.numChar_ = 100;
	id.rep_.isPrimaryKey_ = true;
	id.rep_.name_ = "linkcolumn";
	attrList.push_back(id);

	TeAttribute x;
	x.rep_.type_ = TeREAL;
	x.rep_.isPrimaryKey_ = false;
	x.rep_.name_ = "x_coord";
	attrList.push_back(x);

	TeAttribute y;
	y.rep_.type_ = TeREAL;
	y.rep_.isPrimaryKey_ = false;
	y.rep_.name_ = "y_coord";
	attrList.push_back(y);

	pointTable.name(tablePointName);
	pointTable.setAttributeList(attrList); 
	pointTable.setLinkName("linkcolumn");
	pointTable.setUniqueName("linkcolumn");
	 
	//creates attributes table
	if (!db->createTable(tablePointName, attrList))
	{
		return false;
	}

	return true;
}


bool valuesTable(TeLayer* layer, TeTable& pointTable, TePointSet ptset)
{
	TePointSet ps;
	int itstep = 0, id = 0;

	if (TeProgress::instance())
	{
		TeProgress::instance()->setCaption("TerraHidro Plugin");
		TeProgress::instance()->setMessage("Inserting points of pits in the table...");
		TeProgress::instance()->setTotalSteps(ptset.size());
	}

	//insert attributes values in the table
	TePointSet::iterator itPS = ptset.begin();

	while(itPS != ptset.end())
	{
		std::string objId = Te2String(id, 0);

		double x = itPS->location().x();
		double y = itPS->location().y();

		TeTableRow row;
		row.push_back(objId);
		row.push_back(Te2String(x, 4));
		row.push_back(Te2String(y, 4));
		pointTable.add(row);

		TePoint pt (*itPS);
		pt.objectId(objId);

		ps.add(pt);

		//add attr table in layer
		if (!layer->saveAttributeTable(pointTable))
		{
			return false;
		}

		pointTable.clear();

		//add points in layer
		if (!layer->addPoints(ps))
		{
			return false;
		}

		ps.clear();
		
		if (TeProgress::instance())
		{
			TeProgress::instance()->setProgress(itstep);
		}

		++id;
		++itstep;
		++ itPS;
	}
	
	if (TeProgress::instance())
	{
		TeProgress::instance()->reset();
	}

	return true;
}


bool saveLayer(TeDatabase* db, TeLayer* layer, TeTable& pointTable)
{
	if (TeProgress::instance())
	{
		TeProgress::instance()->setCaption("TerraHidro Plugin");
		TeProgress::instance()->setMessage("Saving output layer...");
		TeProgress::instance()->setTotalSteps(2);
	} 

	if (!layer->database()->insertMetadata(layer->tableName(TePOINTS),layer->database()->getSpatialIdxColumn(TePOINTS), 0.0005,0.0005,layer->box()))
	{
		return false;
	}

	if (TeProgress::instance())
	{
		TeProgress::instance()->setProgress(1);
	}

	if (!layer->database()->createSpatialIndex(layer->tableName(TePOINTS),layer->database()->getSpatialIdxColumn(TePOINTS), (TeSpatialIndexType)TeRTREE))
	{
		return false;
	}

	if (TeProgress::instance())
	{
		TeProgress::instance()->setProgress(2);
	}

	if (!db->insertTableInfo(layer->id(), pointTable))
	{
		return false;
	}

	layer->addAttributeTable(pointTable);

	if (TeProgress::instance())
	{
		TeProgress::instance()->reset();
	}

	return true;
}


bool slopeCompute( HidroMatrix<double>& demMatrix, int col, int lin, double slopeNeighbors[8] )
{
  double altimetriaCenter;
  double altimetriaNeighbors;

  // altimetria in the center cell
  if( !demMatrix.getElement( col, lin, altimetriaCenter ) )
  {
    return false;
  }

  //upper left
	if( !demMatrix.getElement( col-1, lin-1, altimetriaNeighbors ) )
	{		
		slopeNeighbors [0] = -1;
	}
	else
		slopeNeighbors[0] = (altimetriaCenter - altimetriaNeighbors) / M_SQRT2;

  //up
	if( !demMatrix.getElement( col, lin-1,   altimetriaNeighbors ) )
	{		
		slopeNeighbors[1] = -1;
	}
	else
		slopeNeighbors [1] = altimetriaCenter - altimetriaNeighbors;

  //upper right
	if( !demMatrix.getElement( col+1, lin-1, altimetriaNeighbors ) )
	{		
		slopeNeighbors[2] = -1;
	}
	else
		slopeNeighbors[2] = (altimetriaCenter - altimetriaNeighbors) / M_SQRT2;

  //left
	if( !demMatrix.getElement(col-1,   lin, altimetriaNeighbors ) )
	{		
		slopeNeighbors[3] = -1;
	}
	else
		slopeNeighbors[3] = altimetriaCenter - altimetriaNeighbors;

  //right
	if( !demMatrix.getElement(col+1,   lin, altimetriaNeighbors ) )
	{		
		slopeNeighbors[4] = -1;
	}
	else
		slopeNeighbors[4] = altimetriaCenter - altimetriaNeighbors;

  //lower left
	if( !demMatrix.getElement( col-1, lin+1, altimetriaNeighbors ) )
	{
		slopeNeighbors[5] = -1;
	}
	else
		slopeNeighbors[5] = (altimetriaCenter - altimetriaNeighbors) / M_SQRT2;

  //lower   
	if( !demMatrix.getElement( col, lin+1,   altimetriaNeighbors ) )
	{
		slopeNeighbors[6] = -1;
	}
	else
		slopeNeighbors[6] = altimetriaCenter - altimetriaNeighbors;

  //lower right
	if( !demMatrix.getElement( col+1, lin+1, altimetriaNeighbors ) )
	{
		slopeNeighbors[7] = -1;
	}
	else
    slopeNeighbors[7] = (altimetriaCenter - altimetriaNeighbors) / M_SQRT2;

  return true;
}


bool isBorder( HidroMatrix<double>& demMatrix, unsigned int col, unsigned int lin )
{
  if( col < 1 || col > (demMatrix.GetColumns() - 2) ||
    lin < 1 || lin > (demMatrix.GetLines() - 2) )
  {
    return true;
  }
  else if( demMatrix.hasDummy() )
  {
    // Compute the start and end of window (3x3)
    unsigned int startColumn = col - 1;
    unsigned int endColumn = col +  2;
    unsigned int startLine = lin - 1;
    unsigned int endLine = lin + 2;
  	
    double altimetria;

	  for (unsigned int l = startLine; l < endLine; l++)
	  {				
		  for (unsigned int c = startColumn; c < endColumn; c++)
		  {
        if( !demMatrix.getElement(c, l, altimetria) )
        {
          // is dummy.
          return true;
        }
      }
    }
  }

  return false;
}
