#include <HidroHand.h>

#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

HidroHand::HidroHand(
  TeRaster* demRaster,
  TeRaster* lddRaster,
  TeRaster* drainageRaster,
  TeRaster* &handRaster ) :
    HidroFlowAlgorithm( lddRaster ),
    demRaster_(demRaster),
    lddRaster_(lddRaster),
    drainageRaster_(drainageRaster),
    handRaster_(handRaster)
{  
}

HidroHand::~HidroHand()
{
  demMatrix_.clear();  
  drainageMatrix_.clear();
  handMatrix_.clear();
}

bool
HidroHand::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( demRaster_->params().nlines_ != lddRaster_->params().nlines_ || demRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "DEM lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }

  // check line and columns number
  if( demRaster_->params().nlines_ != drainageRaster_->params().nlines_ || demRaster_->params().ncols_ != drainageRaster_->params().ncols_ )
  {
    errorMessage_= "DEM lines and columns are diferent from Drainage lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // load data

  // load DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load DEM data.");    
  if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load LDD
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // load Drainage
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Drainage data.");    
  if( !copyRaster2PDIMatrix( drainageRaster_, drainageMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // create output Hand raster
  
  // handRaster params
  TeRasterParams handRasterParams = demRaster_->params();
  
  // Set dummy
  handRasterParams.setDummy( -9999 );
  handRasterParams.useDummy_ = true;  

  // Change mode
  handRasterParams.mode_ = 'w';
  handRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  handRasterParams.vmax_[0] = -TeMAXFLOAT;
  handRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  handRaster_ = new TeRaster( handRasterParams );

  // verify if handRaster created is valid
  if( !handRaster_->init() )
  {
    errorMessage_ = handRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // used in for's and progress bar
  unsigned int nlines = handRaster_->params().nlines_;
  unsigned int ncolumns = handRaster_->params().ncols_;  

  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating Hand Raster step 1 from 4.");
  TeProgress::instance()->setTotalSteps( nlines );

  unsigned char drainageDummy = 255;
  if( drainageRaster_->params().useDummy_ )
  {
    drainageDummy = (unsigned char)drainageRaster_->params().dummy_[0];
  }

  // init handMatrix
  handMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, handMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( drainageMatrix_[line][column] != drainageDummy )
      {
        handMatrix_[line][column] = demMatrix_[line][column];        
      }
      else
      {
        // flag until dummy
        handMatrix_[line][column] = -9998;
      }
    }
    
    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  drainageMatrix_.clear();

  // STEP 2

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Hand Raster step 2 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init connections matrix
  if( !initConnections() )
  {
    return false;
  }  

  // STEP 3

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Hand Raster step 3 from 4.");
  TeProgress::instance()->setProgress( 0 );
  
  // init the handRaster with drainage altimetria values
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;

  unsigned int lineStop;
  unsigned int columnStop;

  float valDEM;
  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineTo = line;
        columnTo = column;

        valDEM = 0; // ocean or DEM border

        // find the drainage altimetria or handRasterDummy
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;

          lineStop = lineFrom;
          columnStop = columnFrom;

          if( handMatrix_[lineFrom][columnFrom] == -9999 )
          {
            valDEM = 0; // ocean or DEM border
            break;
          }

          if( handMatrix_[lineFrom][columnFrom] != -9998 )
          {
            valDEM = handMatrix_[lineFrom][columnFrom];
            break;
          }
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );

        // fill the path with drainage altimetria
        lineTo = line;
        columnTo = column;
        do
        {
          lineFrom = lineTo;
          columnFrom = columnTo;

          handMatrix_[lineFrom][columnFrom] = valDEM;

          if( lineStop == lineFrom && columnStop == columnFrom )
          {
            break;
          }          
        }
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) );
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Freee Memory
  lddMatrix_.clear();
  
  // STEP 4

  // refresh progress bar
	TeProgress::instance()->setMessage("Creating Hand Raster step 4 from 4.");
  TeProgress::instance()->setProgress( 0 );

  
  // DEM - HAND and dummys
  double valHand;  
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {
      valHand = handMatrix_[line][column];
      if( valHand != -9999 )
      {
        if( valHand != -9998 )
        {
          valHand = demMatrix_[line][column] - valHand;

          if( valHand < 0.0 )
          {
            valHand = 0.0;
          }

          //handMatrix_[line][column] =  (float) valHand;
          handRaster_->setElement( column, line, valHand );
        }
        else
        {
          //handMatrix_[line][column] = -9999;
          handRaster_->setElement( column, line, -9999 );
        }
      }
    }

    // refresh progress bar    
    TeProgress::instance()->setProgress( line );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  // Free Memory
  demMatrix_.clear();  
  handMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroHand::cancel()
{
  // Free Memory
  demMatrix_.clear();
  lddMatrix_.clear();
  drainageMatrix_.clear();
  handMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}