#include <HidroSegmenter2Vector.h>

#include <HidroFlowUtils.h>
#include <HidroPersister.h>

#include <TeGeometry.h>
#include <TeDatabase.h>
#include <TeLayer.h>

#include <qfile.h>
#include <qtextstream.h>

Segmenter2Vector::Segmenter2Vector( std::string& outputName, TeDatabase* database, TeRaster* segmentedRaster, HidroMatrix<unsigned char>& lddMatrix,
      HidroMatrix<int>& segmentedMatrix ) :
outputName_(outputName),
database_(database),
segmentedRaster_(segmentedRaster),
lddMatrix_(lddMatrix),
segmentedMatrix_(segmentedMatrix),
nlines_( lddMatrix.GetLines() ),
ncolumns_( lddMatrix.GetColumns() ),
errorMessage_( "" )
{  
}


bool
Segmenter2Vector::vectorizing()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // limpa a messagem de erro
  errorMessage_.clear();  

  // cria matriz de conexes inializada com 0
  HidroMatrix<unsigned char> connections( nlines_, ncolumns_ );
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if( lddMatrix_[line][column] != 255 )
      {
        connections[line][column] = 0;
      }
      else
      {
        connections[line][column] = 255;
      }
    }
  }

  // inicializa matriz de conexes
  // somente onde tem segmentos
  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if( segmentedMatrix_[line][column] > 0 )
      {
        if( lddCode2LineColumn( line, column, lineTo, columnTo ) )
        {
          connections[lineTo][columnTo]++;
        }
      }
    }
  }

  // cria o layer de saida
  TeProjection* proj = TeProjectionFactory::make( segmentedRaster_->projection()->params() );
	std::string layerName = outputName_;
	TeLayer* outputLayer = new TeLayer( layerName, database_, proj );

  // Cria a tabela
  TeTable table;
  TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.numChar_ = 16;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.name_ = "linkColumn";
	TeAttributeList attrList;
	attrList.push_back(at);
  TeAttribute at2;
	at2.rep_.type_ = TeINT;
	at2.rep_.name_ = "SegmentNumber";
  attrList.push_back(at2);
  TeAttribute at3;
	at3.rep_.type_ = TeINT;
	at3.rep_.name_ = "OrderIPH";
  attrList.push_back(at3);
  std::string tableName = outputName_;
  tableName.append("_attrTable");
	table.name( tableName );
	table.setAttributeList(attrList);  
	table.setLinkName("linkColumn");
	table.setUniqueName("linkColumn");
	database_->validTable(table);
	if(database_->tableExist(tableName))
	{
		errorMessage_ = "Attribute table already exist.";
		return false;
	}
  if(!database_->createTable(tableName, attrList))
	{
		errorMessage_ = database_->errorMessage();
		return false;
	}  

  // barra de progresso
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Vectorizing the segmental matrix.");
  TeProgress::instance()->setTotalSteps( nlines_ );

  // lao principal
  unsigned int lineFrom;
  unsigned int columnFrom;
  int count = 0;
  TeCoord2D fromMatrix;
  TeCoord2D fromWorld;
  TeCoord2D toMatrix;
  TeCoord2D toWorld;
  // Conjunto de linhas
  TeLineSet lineSet;
  bool flag = false;
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {      
      if( segmentedMatrix_[line][column] > 0 )
      {
        if( connections[line][column] == 0 )
        {          
          lineFrom = line;
          columnFrom = column;
          
          fromMatrix.setXY(column, line);
          fromWorld = segmentedRaster_->index2Coord(fromMatrix);
		      flag = false;

          // faz at achar a saida ou proximo segmento
          do
          {
            if( connections[lineTo][columnTo] > 1 && connections[lineTo][columnTo] < 9 )
            {
              connections[lineTo][columnTo] = 9;              
            }

            // verifica se  sada
            if( !lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
              if( flag )
              {
                toMatrix.setXY(columnFrom, lineFrom);
                toWorld = segmentedRaster_->index2Coord(toMatrix);

                TeLine2D line2D;
                line2D.add( fromWorld );
                line2D.add( toWorld );                   

                lineSet.add( line2D );
              }

              break;
            }

            // Cria os pontos
            if( lddMatrix_[lineFrom][columnFrom] != lddMatrix_[lineTo][columnTo] || connections[lineTo][columnTo] > 1 )
            {              
              toMatrix.setXY(columnTo, lineTo);
              toWorld = segmentedRaster_->index2Coord(toMatrix);

              TeLine2D line2D;
              line2D.add( fromWorld );
              line2D.add( toWorld );                   

              lineSet.add( line2D );

              fromMatrix.setXY(columnTo, lineTo);
              fromWorld = segmentedRaster_->index2Coord(fromMatrix);
			        flag = true;
            }

            if( connections[lineTo][columnTo] > 1 )
            {
              // Nova linha
              count = segmentedMatrix_[lineFrom][columnFrom];
              lineSet.objectId( Te2String( segmentedMatrix_[lineFrom][columnFrom], 0 ) );
              outputLayer->addLines( lineSet );
              TeTableRow row;
              row.push_back( lineSet.objectId() );
              row.push_back( lineSet.objectId() );
              row.push_back( "" ); // IPH order
              table.add(row);              
              lineSet.clear();
			        flag = false;
            }

            lineFrom = lineTo;
            columnFrom = columnTo;          
          }
          while( connections[lineTo][columnTo] > 0 && connections[lineTo][columnTo] < 9 );

          if( flag )
          {
            count = segmentedMatrix_[lineFrom][columnFrom];
            lineSet.objectId( Te2String( segmentedMatrix_[lineFrom][columnFrom], 0 ) );
            outputLayer->addLines( lineSet );
            TeTableRow row;              
            row.push_back( lineSet.objectId() );
            row.push_back( lineSet.objectId() ); 
            row.push_back( "" ); // IPH order
            table.add(row);
            lineSet.clear();
          }
        }     
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      errorMessage_= "Segmentation operation was canceled by user.";
      // finaliza barra de progresso
      TeProgress::instance()->reset();
      // guarda o tempo decorrente do processo
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return false;
    }
  }

  // Salva o Layer
  if(!database_->insertTable(table))
	{
		errorMessage_ = database_->errorMessage();
		return false;
	}
  HidroPersister persister(database_);
	if(!persister.saveLayer(outputLayer, table, TeLINES))
	{
		return false;
	}
  
  // barra de progresso
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Finding order.");
  TeProgress::instance()->setTotalSteps( nlines_ );

  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if( lddMatrix_[line][column] != 255 )
      {
        connections[line][column] = 0;
      }
      else
      {
        connections[line][column] = 255;
      }
    }
  }

  // inicializa matriz de conexes
  // somente onde tem segmentos  
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {
      if( segmentedMatrix_[line][column] > 0 )
      {
        if( lddCode2LineColumn( line, column, lineTo, columnTo ) )
        {
          connections[lineTo][columnTo]++;
        }
      }
    }
  }

  // lao principal
  int order = 1;
  std::vector<int> segmentsOrder( segmentedRaster_->params().vmax_[0], 0 );
  bool flagOut;
  for( unsigned int line=0; line<nlines_; line++ )
  {
    for( unsigned int column=0; column<ncolumns_; column++ )
    {      
      if( segmentedMatrix_[line][column] > 0 )
      {
        if( connections[line][column] == 0 )
        {          
          lineTo = line;
          columnTo = column;
          flagOut = false;
          order = 1;
          //segmentsOrder[ segmentedMatrix_[line][column]-1 ] = order;

          // faz at achar a saida ou proximo segmento
          do
          {
            lineFrom = lineTo;
            columnFrom = columnTo;

            // verifica se  sada
            if( !lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
              flagOut = true;
              break;
            }

            if( connections[lineTo][columnTo] > 1 )
            {
              segmentsOrder[ segmentedMatrix_[lineFrom][columnFrom]-1 ] = order;
              order++;
            }            
          }
          while( connections[lineTo][columnTo] > 0 &&
            order > segmentsOrder[ segmentedMatrix_[lineTo][columnTo]-1 ] );         

           if(flagOut)
            segmentsOrder[ segmentedMatrix_[lineFrom][columnFrom]-1 ] = order;
           else if( order > segmentsOrder[ segmentedMatrix_[lineTo][columnTo]-1 ] )
            segmentsOrder[ segmentedMatrix_[lineFrom][columnFrom]-1 ] = order;

         
        }     
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      errorMessage_= "Segmentation operation was canceled by user.";
      // finaliza barra de progresso
      TeProgress::instance()->reset();
      // guarda o tempo decorrente do processo
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return false;
    }
  }

  for( unsigned int i=0; i<segmentsOrder.size(); i++ )
  {
    //"UPDATE ord3_attrTable SET OrderIPH=1 WHERE SegmentNumber=1"
    std::string sql = "UPDATE ";
    sql.append( tableName );
    sql.append( " SET OrderIPH=" );
    sql.append( Te2String(segmentsOrder[i],0) );
    sql.append( " WHERE SegmentNumber=" );
    sql.append( Te2String(i+1,0) );              
    database_->execute( sql );
  }

  // finaliza barra de progresso
  TeProgress::instance()->reset();

  // guarda a hora final do processo
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}


bool
Segmenter2Vector::lddCode2LineColumn( const unsigned int &lineFrom, const unsigned int &columnFrom,
    unsigned int &lineTo, unsigned int &columnTo )
{
  const unsigned char lddCode = lddMatrix_[lineFrom][columnFrom];

  if( lddCode == 0 || lddCode == 255 )
  {
    return false;
  }
  // rigth
  else if( lddCode == 1 )
  {
    if( columnFrom < ncolumns_-1 )
    
    {
      lineTo = lineFrom;
      columnTo = columnFrom + 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // down rigth
  else if( lddCode == 2 )
  {
    if( columnFrom < ncolumns_-1 && lineFrom < nlines_-1 )
    
    {
      lineTo = lineFrom + 1;
      columnTo = columnFrom + 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // down
  else if( lddCode == 4 )
  {
    if( lineFrom < nlines_-1 )
    
    {
      lineTo = lineFrom + 1;
      columnTo = columnFrom;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // down left
  else if( lddCode == 8 )
  {
    if( columnFrom > 0 && lineFrom < nlines_-1 )
    
    {
      lineTo = lineFrom + 1;
      columnTo = columnFrom - 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // left
  else if( lddCode == 16 )
  {
    if( columnFrom > 0 )
    
    {
      lineTo = lineFrom;
      columnTo = columnFrom - 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // up left
  else if( lddCode == 32 )
  {
    if( columnFrom > 0 && lineFrom > 0 )
    
    {
      lineTo = lineFrom - 1;
      columnTo = columnFrom - 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // up
  else if( lddCode == 64 )
  {
    if( lineFrom > 0 )
    
    {
      lineTo = lineFrom - 1;
      columnTo = columnFrom;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  // up rigth
  else if( lddCode == 128 )
  {
    if( columnFrom < ncolumns_-1 && lineFrom > 0 )
    
    {
      lineTo = lineFrom - 1;
      columnTo = columnFrom + 1;
      if( lddMatrix_[lineTo][columnTo] != 255 )
      {
        return true;
      }
    }    
  }
  
  return false;
}

const std::string&
Segmenter2Vector::getErrorMessage()
{
  return errorMessage_;
}

void
Segmenter2Vector::print( std::string fileName )
{
  QFile segmenterStatisticsFile( fileName.c_str() );  

  if( segmenterStatisticsFile.open( IO_ReadWrite  ) )
  {
    QTextStream segmenterStatisticsStream;
    segmenterStatisticsStream.setDevice( &segmenterStatisticsFile );

    segmenterStatisticsStream << fileName.c_str() << "\n";
    segmenterStatisticsStream << "Image size: " << Te2String(nlines_*ncolumns_).c_str() << "\n";
    segmenterStatisticsStream << "Lines: " << Te2String(nlines_).c_str() << "\n";
    segmenterStatisticsStream << "Columns: " << Te2String(ncolumns_).c_str() << "\n \n";

    segmenterStatisticsStream << "Start Time: " << timeStart_.c_str();

    if( !errorMessage_.empty() )
    {
      segmenterStatisticsStream << errorMessage_.c_str();
    }

    segmenterStatisticsStream << "End Time: " << timeEnd_.c_str();
    segmenterStatisticsStream << "Processing time: " << timeTotal_.c_str() << "\n \n";    
  }

  segmenterStatisticsFile.close();
}