#include <HidroCrossSectionBottomLevel.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>

#include <vector>

HidroCrossSectionBottomLevel::HidroCrossSectionBottomLevel(TeRaster* lddRaster,
    TeRaster* demRaster,
    TeRaster* contributingAreaRaster,
    TeRaster* pathsRaster,
    TeRaster* &outputDEMRaster,
    unsigned int line,
    unsigned int column,
    float outletContributingArea,
    unsigned int interactions,
    float alpha,
    float beta,
    float theta,
    float thetam,
    float thetac ) :
    HidroFlowAlgorithm( lddRaster ),
    lddRaster_(lddRaster),
    demRaster_(demRaster),
    contributingAreaRaster_(contributingAreaRaster),
    pathsRaster_(pathsRaster),
    outputDEMRaster_(outputDEMRaster),
    line_(line),
    column_(column),
    outletContributingArea_(outletContributingArea),
    interactions_(interactions),
    alpha_(alpha),
    beta_(beta),
    theta_(theta),
    thetam_(thetam),
    thetac_(thetac)
{
}

HidroCrossSectionBottomLevel::~HidroCrossSectionBottomLevel()
{
  //demMatrix_.clear();
  //contributingAreaMatrix_.clear();
  pathsMatrix_.clear();
  //outputDEMMatrix_.clear();
}

bool HidroCrossSectionBottomLevel::cancel()
{
  // Free Memory
  //demMatrix_.clear();
  //contributingAreaMatrix_.clear();
  pathsMatrix_.clear();
  //outputDEMMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}

bool
HidroCrossSectionBottomLevel::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // check line and columns number
  if( demRaster_->params().nlines_ != lddRaster_->params().nlines_ || demRaster_->params().ncols_ != lddRaster_->params().ncols_ )
  {
    errorMessage_= "DEM lines and columns are diferent from LDD lines and columns.";    

    // save processing time
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  // load data

  // load Paths
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load Paths  data.");    
  if( !copyRaster2PDIMatrix( pathsRaster_, pathsMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  unsigned int nlines = pathsRaster_->params().nlines_;
  unsigned int ncolumns = pathsRaster_->params().ncols_;
  
  // STEP 1
  // start the progress bar
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Cross Section Bottom Level");	  
  TeProgress::instance()->setMessage("Step 1 from 5: load data");
  TeProgress::instance()->setTotalSteps( nlines );

  int size = 0;
  int k = 0;
  std::vector<int> icell;
  std::vector<float> aacum_v;
  std::vector<float> mnt_v;  
  double contributingArea;
  double altimetria;
  double aux;
  
  for( unsigned int line = 0; line < nlines; line++ )
  {
    pathsMatrix_[line][0] = -1;
    pathsMatrix_[line][1] = -1;
    pathsMatrix_[line][ncolumns-1] = -1;
    pathsMatrix_[line][ncolumns-2] = -1;
  }

  for( unsigned int column = 0; column < ncolumns; column++ )
  {
    pathsMatrix_[0][column] = -1;
    pathsMatrix_[1][column] = -1;
    pathsMatrix_[nlines-1][column] = -1;
    pathsMatrix_[nlines-2][column] = -1;
  }

  for( unsigned int line = 0; line < nlines; line++ )
  {
    for( unsigned int column = 0; column < ncolumns; column++ )
    {
      if( pathsMatrix_[line][column] == 255 )
      {
        pathsMatrix_[line][column] = -1;
      }
    }
  }

  //! cria um vetor com os caminhos e aloca nele as areas acumuladas
  for( unsigned int line = 2; line < nlines-2; line++ )
  {
    for( unsigned int column = 2; column < ncolumns-2; column++ )
    {
      if( pathsMatrix_[line][column] >-1 )
      {        
        contributingAreaRaster_->getElement( column, line, contributingArea );
        aacum_v.push_back( (float)contributingArea );
        
        icell.push_back( line*ncolumns + column );
        pathsMatrix_[line][column] = k;

        demRaster_->getElement( column, line, altimetria );
        //! substrai a profundidade do curso de agua em funcao da area acumulada
        altimetria = -alpha_ *  powf(contributingArea, beta_);
        mnt_v.push_back( (float)altimetria );

        size++;
        k++;
      }
      else
      {
         pathsMatrix_[line][column] = -1;
      }
    }
    TeProgress::instance()->setProgress( line );
  }  

  // Step 2
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Cross Section Bottom Level");	  
  TeProgress::instance()->setMessage("Step 2 from 5: determining the downstream pixels");
  TeProgress::instance()->setTotalSteps( size );

  //! determina a celula a jusante de cada pixel com base no ldd
  std::vector<int> celljus;
  std::vector<unsigned char> ncellmon;
  // id, cellmon1, cellmon2, cellmon3  
  std::vector< int*> cellmon;
  
  unsigned int lline = 0;
  unsigned int ccolumn = 0;

  unsigned int l = 0;
  unsigned int c = 0;
     
  for( unsigned int k = 0; k < size; k++ )
  {
    lline = (unsigned int)( icell[k] / ncolumns );
    ccolumn = icell[k]-(lline)*ncolumns;

    lddRaster_->getElement( ccolumn, lline, aux );
    l = lline + lineIncrement[(unsigned char)aux];
    c = ccolumn+ columnIncrement[(unsigned char)aux];

    if( (unsigned char)aux != 0 && (unsigned char)aux != 255 && pathsMatrix_[l][c] > -1 )
    {      
      celljus.push_back( pathsMatrix_[l][c] );
    }
    else
    {
       celljus.push_back( -1 );
    }

    ncellmon.push_back( 0 );
    int* a = new  int[3];
    a[0] = -1;
    a[1] = -1;
    a[2] = -1;
    cellmon.push_back( a );

    TeProgress::instance()->setProgress( k );
  }

  // Free Memory
  pathsMatrix_.clear();

  // Step 3
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Cross Section Bottom Level");	  
  TeProgress::instance()->setMessage("Step 3 from 5: determining the upstream pixels");
  TeProgress::instance()->setTotalSteps( size );
  //! cria topologia de montante e jusante
    
  for( unsigned int k = 0; k < size; k++ )
  {   
    if( celljus[k] > -1 )
    {
      if( ncellmon[celljus[k]] < 3 )
      {        
        cellmon[celljus[k]][ncellmon[celljus[k]]] = k;
        ncellmon[celljus[k]] = ncellmon[celljus[k]] + 1;
      }
    }
    TeProgress::instance()->setProgress( k );
  }


  // Step 4
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Cross Section Bottom Level");	  
  TeProgress::instance()->setMessage("Step 4 from 5: removing random noise");
  TeProgress::instance()->setTotalSteps( interactions_ );
  
  //! itera
  unsigned int iter = 0;
  std::vector<float> mnt_v2;
  mnt_v2.resize( size );

  while ( iter <= interactions_ )
  {
    iter++;
    for( unsigned int k = 0; k < size; k++ )
    {
       if( ncellmon[k] == 0 ) //! nao ha celulas de montante, nascente
       {
         if( celljus[k] > 0 )
         {
           mnt_v2[k] = thetam_ * mnt_v[k] + (1-thetam_) * mnt_v[celljus[k]];
         }
         else
         {
           mnt_v2[k] = thetam_ * mnt_v[k];
         }
       }
       else
       {
         if( ncellmon[k] == 1 ) //!existe uma unica celula de montante
         {
           if( celljus[k] == -1) //! nao ha celulas a jusante, foz
           {
             mnt_v2[k]= thetam_ * mnt_v[k] + (1-thetam_) * mnt_v[cellmon[k][0]];
           }
           else //sequencia normal
           {
             mnt_v2[k] = theta_ * mnt_v[cellmon[k][0]] + (1-2*theta_) * mnt_v[k]+ theta_ * mnt_v[celljus[k]];
           }
         }
         else //! confluencia de dois ou mais rios
         {
           float area = 0.0;
           for( int i=0; i<ncellmon[k]; i++ )
           {
             if( cellmon[k][i] > -1 )
             {
               area = area + aacum_v[cellmon[k][i]]; // ! acumula area a montante
             }
           }
           if( celljus[k] == -1 ) // nao ha celulas a jusante (foz) mas ha confluencia a montante
           {
             mnt_v2[k] = thetac_ * mnt_v[k];
           }
           else
           {
             area = area + aacum_v[celljus[k]];// ! area a jusante
             mnt_v2[k]=mnt_v[celljus[k]] * (1-thetac_) * aacum_v[celljus[k]]/area + thetac_ * mnt_v[k];
           }
           for( int i=0; i<ncellmon[k]; i++ )
           {
             mnt_v2[k]=mnt_v2[k] + mnt_v[cellmon[k][i]] * (1-thetac_) * aacum_v[cellmon[k][i]]/area;
           }            
         }
       }
    }
    mnt_v = mnt_v2;
    TeProgress::instance()->setProgress( iter );
  }
  mnt_v2.clear();        
       
  
  // Step 5
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("Cross Section Bottom Level");	  
  TeProgress::instance()->setMessage("Step 5 from 5: building the new DEM");
  TeProgress::instance()->setTotalSteps( nlines + size );

  // create output DEM raster
  
  // output DEM params
  TeRasterParams outputDEMRasterParams = demRaster_->params();

  // Set data type
  outputDEMRasterParams.setDataType( TeFLOAT );
  
  // Set dummy
  outputDEMRasterParams.setDummy( -9999 );
  outputDEMRasterParams.useDummy_ = true;  

  // Change mode
  outputDEMRasterParams.mode_ = 'w';
  outputDEMRasterParams.decoderIdentifier_ = "SMARTMEM";

  // Set Max and Minimal values
  outputDEMRasterParams.vmax_[0] = -TeMAXFLOAT;
  outputDEMRasterParams.vmin_[0] =  TeMAXFLOAT;

  // create the raster
  outputDEMRaster_ = new TeRaster( outputDEMRasterParams );

  // verify if outputDEMRaster created is valid
  if( !outputDEMRaster_->init() )
  {
    errorMessage_ = outputDEMRaster_->errorMessage();    
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return false;
  }  

  for( unsigned int line = 0; line < nlines; line++ )
  {
    for( unsigned int column = 0; column < ncolumns; column++ )
    {
       demRaster_->getElement( column, line, altimetria );
       outputDEMRaster_->setElement(  column, line, altimetria );
    }
    TeProgress::instance()->setProgress( line );
  }

  for( unsigned int k = 0; k < size; k++ )
  {
    lline = (unsigned int)( icell[k] / ncolumns );
    ccolumn = icell[k]-(lline)*ncolumns;    

    altimetria = mnt_v[k];

    outputDEMRaster_->setElement(  ccolumn, lline, altimetria );
    
    TeProgress::instance()->setProgress( k );
  }
  
  
  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}
