#include <HidroTopographicIndexMDInfinite.h>

#include <HidroFlowUtils.h>

#include <stdio.h>
#include <cmath>

#define M_PI       3.14159265358979323846
#define M_SQRT2    1.41421356237309504880

#include <TeRaster.h>

HidroTopographicIndexMDInfinite::HidroTopographicIndexMDInfinite(
  TeRaster* demRaster,
  TeRaster* lddRaster,
  TeRaster* highSubwatershedRaster,
  TeRaster* lowSubwatershedRaster,
  TeRaster* lowAccumulatedRaster,
  std::string modelFileName,
  std::string outFileName,
  float factor,
  float cit,
  int nac ) :
demRaster_(demRaster),
lddRaster_(lddRaster),
highSubwatershedRaster_(highSubwatershedRaster),
lowSubwatershedRaster_(lowSubwatershedRaster),
lowAccumulatedRaster_(lowAccumulatedRaster),
modelFileName_(modelFileName),
outFileName_(outFileName),
factor_(factor),
cit_(cit),
nac_(nac)
{
  minSlope_ = 0.0001;

  topographicDummy_ = -9999.0;

  if( demRaster->params().useDummy_ )
  {
    demDummy_ = (float)demRaster->params().dummy_[0];
  }
  else
  {
    demDummy_ = -9999;
  }

  lineIncrement_[0] = 1;
  columnIncrement_[0] = 0;

  lineIncrement_[1] = 1;
  columnIncrement_[1] = 1;

  lineIncrement_[2] = 0;
  columnIncrement_[2] = 1;

  lineIncrement_[3] = -1;
  columnIncrement_[3] = 1;

  lineIncrement_[4] = -1;
  columnIncrement_[4] = 0;

  lineIncrement_[5] = -1;
  columnIncrement_[5] = -1;

  lineIncrement_[6] = 0;
  columnIncrement_[6] = -1;

  lineIncrement_[7] = 1;
  columnIncrement_[7] = -1;

  // right
  lineLDDIncrement_[1] = 0;
  columnLDDIncrement_[1] = 1;

  // down right
  lineLDDIncrement_[2] = 1;
  columnLDDIncrement_[2] = 1;

  // down
  lineLDDIncrement_[4] = 1;
  columnLDDIncrement_[4] = 0;

  // down left
  lineLDDIncrement_[8] = 1;
  columnLDDIncrement_[8] = -1;

  // left
  lineLDDIncrement_[16] = 0;
  columnLDDIncrement_[16] = -1;

  // up left
  lineLDDIncrement_[32] = -1;
  columnLDDIncrement_[32] = -1;

  // up
  lineLDDIncrement_[64] = -1;
  columnLDDIncrement_[64] = 0;

  // up right
  lineLDDIncrement_[128] = -1;
  columnLDDIncrement_[128] = 1;
}

HidroTopographicIndexMDInfinite::~HidroTopographicIndexMDInfinite()
{
  if( outFile_ != NULL )
  {
    fclose( outFile_ );
    outFile_ = NULL;
  }  
}

bool
HidroTopographicIndexMDInfinite::execute()
{
  // guarda a hora inicial do processo
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();

  // abre o arquivo de saida
  outFile_ = NULL;
  outFile_ = fopen( outFileName_.c_str(), "w" );

  if( outFile_ == NULL )
  {
    errorMessage_ = "Error - Can't create output file!";
    return error();
  }

  // ler dados da celulas do modelo
  if( !readCell() )
  {
    errorMessage_ = "Error - Can't read cells file!";
    return error();
  }

  double demResolution = demRaster_->params().resx_;
  pixelsNumber_ = factor_;//(int)(modelResolution_ / demResolution + 0.5) ;

  demMatrix_.Reset( factor_, factor_, TePDIMatrix<float>::AutoMemPol, demMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  lddMatrix_.Reset( factor_, factor_, TePDIMatrix<unsigned char>::AutoMemPol, lddMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  accumulatedMatrix_.Reset( factor_, factor_, TePDIMatrix<float>::AutoMemPol, accumulatedMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  topographicIndexMatrix_.Reset( factor_, factor_, TePDIMatrix<float>::AutoMemPol, topographicIndexMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
    
  maxLines_ = demMatrix_.GetLines();
  maxColumns_ = demMatrix_.GetColumns();

  TeProgress::instance()->setMessage("Creating Topographic Index.");
  TeProgress::instance()->setTotalSteps( (int)cellsVector_.size() );
  TeProgress::instance()->setProgress( 0 );

  // para cada celula do modelo
  for( unsigned int i=0; i<cellsVector_.size(); i++ )
  {
    // imprime no arquivo
    fprintf( outFile_, "%5d %15.3f %15.3f", cellsVector_[i].id_, cellsVector_[i].lat_, cellsVector_[i].long_ );    
    fflush( outFile_ );

    // encotra a linha e coluna do pixel central da celula do modelo
    // processo diferente para pixelNumber PAR e IMPAR
    int lineBegin;
    int lineEnd;
    int columnBegin;
    int columnEnd;

    if( pixelsNumber_ % 2 > 0 ) //IMPAR
    {
      TeCoord2D centerPixelIndex = demRaster_->coord2Index( 
      TeCoord2D( cellsVector_[i].long_,
      cellsVector_[i].lat_ ) );
    
      lineBegin = (int)(centerPixelIndex.y_ + 0.5 - pixelsNumber_ / 2);
      lineEnd = 1 + (int)(centerPixelIndex.y_ + 0.5 + pixelsNumber_ / 2);

      columnBegin = (int)(centerPixelIndex.x_ + 0.5 - pixelsNumber_ / 2);
      columnEnd = 1 + (int)(centerPixelIndex.x_ + 0.5 + pixelsNumber_ / 2);

    }
    else // PAR
    {
      TeCoord2D centerPixelIndex = demRaster_->coord2Index( 
      TeCoord2D( cellsVector_[i].long_ + demResolution/2,
      cellsVector_[i].lat_ - demResolution/2 ) );
    
      lineBegin = (int)(centerPixelIndex.y_ + 0.5 - pixelsNumber_ / 2);
      lineEnd = (int)(centerPixelIndex.y_ + 0.5 + pixelsNumber_ / 2);

      columnBegin = (int)(centerPixelIndex.x_ + 0.5 - pixelsNumber_ / 2);
      columnEnd = (int)(centerPixelIndex.x_ + 0.5 + pixelsNumber_ / 2);
    }    

    natb_ = 0;

    // encontra a rea acumulada com MDinfinito
    accumulated( lineBegin, lineEnd, columnBegin, columnEnd );

    // refresh progress bar    
    TeProgress::instance()->setProgress( i );
    
    // check for cancel
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

  if( outFile_ != NULL )
  {
    fclose( outFile_ );    
    outFile_ = NULL;
  }

  // Finish progress bar
  TeProgress::instance()->reset();

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}  

bool
HidroTopographicIndexMDInfinite::readCell()
{
  // ler dados da celulas do modelo
  FILE *f;
  char line[300];  

	f = fopen( modelFileName_.c_str(), "r" );

  if( f != NULL )
	{
    while( !feof( f ) )
    {
      cellModel cell;
      fgets( line, 300, f );
      sscanf( line, "%d %f %f", &cell.id_, &cell.long_, &cell.lat_  );
      cellsVector_.push_back( cell );
    }
    cellsVector_.pop_back(); // retira a ultima celula pois le 2 vezes a ultima linha do arquivo.
    fclose(f);
	}
  else
  {
    return false;
  }

  return true;
}

void
HidroTopographicIndexMDInfinite::accumulated( unsigned int lineBegin, unsigned int lineEnd,
    unsigned int columnBegin, unsigned int columnEnd )
{ 
  double demVal;
  double subwatershedVal;
  double lddVal;
  float l;
  
  double demResolution = demRaster_->params().resx_;

  if( demRaster_->params().projection()->name().compare( "LatLong" ) == 0 )
  {
    l = (float)(110000 * demResolution);
  }
  else
  {
    l = (float)demResolution;
  }
  
  float areaPixel = (float)(l*l);  

  // Inicializa as matrizes dem, accululated e topographicIndex
  for( unsigned int line=lineBegin; line<lineEnd; line++ )
  {
    for( unsigned int column=columnBegin; column<columnEnd; column++ )
    {
      if( !demRaster_->getElement( column, line, demVal ) )
      {
        demVal = demDummy_;
      }

      // so para ficar igual ao fortran.
      if( demVal < -9999 )
      {
        demVal = demDummy_;
      }

      if( lddRaster_->getElement( column, line, lddVal ) )
      {
        lddMatrix_[line-lineBegin][column-columnBegin] = (unsigned char)lddVal;
      }
      else
      {
        lddMatrix_[line-lineBegin][column-columnBegin] = 255;
      }

      highSubwatershedRaster_->getElement( column, line, subwatershedVal );

      if( subwatershedVal <= 0 )
      {
        demMatrix_[line-lineBegin][column-columnBegin] = demDummy_;
      }
      else
      {
        demMatrix_[line-lineBegin][column-columnBegin] = (float)demVal;
      }      

      // SET ALL NON-CATCHMENT A/TANB VALUES TO UNDEF AND ALL CATCHMENT VALUES        
      // TO -9.9.  SET ALL A VALUES TO DX*DX 
      if( demMatrix_[line-lineBegin][column-columnBegin] == demDummy_ )
      {
        accumulatedMatrix_[line-lineBegin][column-columnBegin] = demDummy_;
        topographicIndexMatrix_[line-lineBegin][column-columnBegin] = topographicDummy_;
        natb_++;
      }
      else
      {
        accumulatedMatrix_[line-lineBegin][column-columnBegin] = areaPixel;
        //topographicIndexMatrix_[line-lineBegin][column-columnBegin] = -8888.0;
        topographicIndexMatrix_[line-lineBegin][column-columnBegin] = 0.0;
        HillTop hilltop(line-lineBegin, column-columnBegin, demMatrix_[line-lineBegin][column-columnBegin] );
        hillTopQueue_.push( hilltop );
      }
    }
  }

  // acha os topos de morros
  //findHillTop();

  gridAverageSlope_ = 0;
  gridNSlope_ = 0;

  // faz para todos topos de morro
  while( !hillTopQueue_.empty() )
  {
    HillTop hilltop = hillTopQueue_.top();
    hillTopQueue_.pop();
    
    // calcula a direo e declividade
    //if( topographicIndexMatrix_[hilltop.line_][hilltop.column_] == -8888.0 )
    {
      md_inf( hilltop.line_, hilltop.column_ );
      //includeNeighbors( hilltop.line_, hilltop.column_ );
    }    
  }

  gridAverageSlope_ = gridAverageSlope_ / gridNSlope_;
  fprintf( outFile_, " %15.7f", gridAverageSlope_/10 );

  // calcula o histograma
  histogram();
}

void
HidroTopographicIndexMDInfinite::histogram()
{
  float atbMax = (float)-TeMAXFLOAT;
  float atbMin = (float)TeMAXFLOAT;
  //float atbSum = 0.0;
  int atbNumber = 0;

  // Find max, min and sum
  for( int line=0; line<maxLines_; line++ )
  {
    for( int column=0; column<maxColumns_; column++ )
    {
      if( topographicIndexMatrix_[line][column] != topographicDummy_ )
      {
        if( topographicIndexMatrix_[line][column] > atbMax )
        {
          atbMax = topographicIndexMatrix_[line][column];
        }

        if( topographicIndexMatrix_[line][column] < atbMin )
        {
          atbMin = topographicIndexMatrix_[line][column];
        }        

        //atbSum +=  (float)exp( topographicIndexMatrix_[line][column] );        
        atbNumber++;
      }
    }
  }

  //float atbAVG = atbSum / atbNumber;  
  float atbDiff = ( atbMax - atbMin ) / ( nac_ );

  int ind = 0;
  vector<double> y;
  y.resize( nac_ );
  
  vector<double> ac;
  ac.resize( nac_ );
  
  vector<double> st;
  st.resize( nac_ );

  for( int i=0; i<nac_; i++ )
  {
    y[i] = 0;
    ac[i] = 0.0;
    st[i] = 0.0;
  }

  for( int line=0; line<maxLines_; line++ )
  {
    for( int column=0; column<maxColumns_; column++ )
    {
      if( topographicIndexMatrix_[line][column] != topographicDummy_ )
      {
        ind = (int)( ((topographicIndexMatrix_[line][column] - atbMin) / atbDiff) + 1 );

        if( ind > (nac_-1) )
        {
          ind = nac_ - 1;
        }

        y[ind] = y[ind] + 1;
      }
    }
  }

  ac[0] = 0.0;
  //ac[0] = y[0]/atbNumber;
  st[0] =  ( atbMax );
  
  for( int i=1; i<nac_; i++ )
  {    
    ac[i]= (y[nac_-i]/atbNumber) + ac[i-1];
	  st[i]= ( (atbMax-(i)*atbDiff) );   
  }

  for( int i=0; i<nac_; i++ )
  {
    fprintf( outFile_, " %15.3e", ac[i] );
  }
  fflush(outFile_);

  for( int i=0; i<nac_; i++ )
  {
    fprintf( outFile_, " %20.3f", st[i] );
  }
  fflush(outFile_);

  fprintf( outFile_, "\n");
}

void
HidroTopographicIndexMDInfinite::includeNeighbors( unsigned int line, unsigned int column )
{
  int neighborLine;
  int neighborColumn;

  for( int k = 0; k < 8; ++k )
  {
    neighborLine = line + lineIncrement_[k];    

    if( neighborLine > -1 && neighborLine < maxLines_ )
    {
      neighborColumn = column + columnIncrement_[k];
      if( neighborColumn > -1 && neighborColumn < maxColumns_ )
      {
        if( topographicIndexMatrix_[neighborLine][neighborColumn] == -8888.0 )
        {
          topographicIndexMatrix_[neighborLine][neighborColumn] = topographicDummy_;
          HillTop hilltop( neighborLine, neighborColumn, demMatrix_[neighborLine][neighborColumn] );
          hillTopQueue_.push( hilltop );
        }
      }
    }
  }
}

void
HidroTopographicIndexMDInfinite::md_inf( unsigned int line, unsigned int column )
{
  unsigned char q[8];
  float fangle[16], fslope[16], ftotal[16], z[8];
  int neighbor[16];
  float sumSlope = 0;
  int nd = 0;
  int m;
  float nx, ny, nz;
  float alfa;

  float resx = (float) demRaster_->params().resx_ * 110000;
  float resy = (float) demRaster_->params().resy_ * 110000;

  double demResolution = demRaster_->params().resx_;
  float l = (float)(110000 * demResolution);
  float areaPixel = (float)(l*l);
  float radii = (float)(sqrt( areaPixel ) / sqrt( M_PI ));
  float contour = 0;  

  for( int i=0; i<8; ++i )
  {
    q[i] = 0;
    z[i] = (float)TeMAXFLOAT;
  }  

  // line = i
  // column = j  

  // z0
  if( (int)line + 1 < maxLines_ )
  {
    z[0] = demMatrix_[line + 1][column] - demMatrix_[line][column];
  }

  // z1
  if( (int)line + 1 < maxLines_ && (int)column + 1 < maxColumns_ )
  {
    z[1] = demMatrix_[line + 1][column + 1] - demMatrix_[line][column];
  }

  // z2
  if( (int)column + 1 < maxColumns_ )
  {
    z[2] = demMatrix_[line][column + 1] - demMatrix_[line][column];
  }

  // z3
  if( (int)line - 1 > 0 && (int)column + 1 < maxColumns_ )
  {
    z[3] = demMatrix_[line - 1][column + 1] - demMatrix_[line][column];
  }

  // z4
  if( (int)line - 1 > 0 )
  {
    z[4] = demMatrix_[line - 1][column] - demMatrix_[line][column];
  }

  // z5
  if( (int)line - 1 > 0 && (int)column - 1 > 0 )
  {
    z[5] = demMatrix_[line - 1][column - 1] - demMatrix_[line][column];
  }

  // z6
  if( (int)column - 1 > 0  )
  {
    z[6] = demMatrix_[line][column - 1] - demMatrix_[line][column];
  }

  // z7
  if( (int)line + 1 < maxLines_ && (int)column - 1 > 0  )
  {
    z[7] = demMatrix_[line + 1][column - 1] - demMatrix_[line][column];
  }
  
  //contour = (float)( radii * M_PI / 4);
  // super looping Joo Ricardo
  for( int k = 0; k < 8; k+=2 )
  {
    m = (k + 2) % 8; // artificio para dar volta no looping

    if( z[k] < 0 || z[k+1] < 0 )
    {
      nx = -z[k] * resx;
      ny = (z[k] - z[k+1]) * resy;
      nz = resy * resx;

      if( ny <= 0 )
      {
        q[k]++;
        if( q[k] == 2 )
        {
          fangle[nd] = 1.0;
          fslope[nd] = abs( nx ) / nz;
          contour += (float)( radii * M_PI / 4);
          sumSlope += fslope[nd];
          neighbor[nd] = k;
          nd++;
        }
      }
      else if( nx > 0 && abs( nx ) > abs( ny ) )
      {
        alfa = atan( ny / nx );
        fangle[nd] = (float)(1 - 4 * alfa / M_PI);
        fslope[nd] = sqrt( nx*nx + ny*ny ) / nz;
        neighbor[nd] = k;
        nd++;
        fangle[nd] = (float)(4 * alfa / M_PI);
        fslope[nd] = fslope[nd-1];
        contour += (float)( radii * M_PI / 4);        
        sumSlope += fslope[nd];
        neighbor[nd] = k + 1;
        nd++;
      }
      else
      {
        q[k+1]++;
        if( q[k+1] == 2 )
        {
          fangle[nd] = 1.0;
          fslope[nd] = (float)(M_SQRT2 * abs( ny )) / nz;
          contour += (float)( radii * M_PI / 4);          
          sumSlope += fslope[nd];
          neighbor[nd] = k + 1;
          nd++;
        }
      }
    }
    // outro triangulo
    if( z[k+1] < 0 || z[m] < 0 )
    {
      nx = (z[m] - z[k+1]) * resx;
      ny = -z[m] * resy;
      nz = resx * resy;

      if( nx <= 0 )
      {
        q[m]++;
        if( q[m] == 2 )
        {
          fangle[nd] = 1.0;
          fslope[nd] = abs( ny ) / nz;
          contour += (float)( radii * M_PI / 4);          
          sumSlope += fslope[nd];
          neighbor[nd] = m;
          nd++;
        }
      }
      else if( ny > 0 && abs( ny ) > abs( nx ) )
      {
        alfa = atan( nx / ny );
        fangle[nd] = (float)(1 - 4 * alfa / M_PI);
        fslope[nd] = sqrt( nx*nx + ny*ny ) / nz;
        neighbor[nd] = m;
        nd++;
        fangle[nd] = (float)(4 * alfa / M_PI);
        fslope[nd] = fslope[nd-1];
        contour += (float)( radii * M_PI / 4);        
        sumSlope += fslope[nd];
        neighbor[nd] = k + 1;
        nd++;
      }
      else
      {
        q[k+1]++;
        if( q[k+1] == 2 )
        {
          fangle[nd] = 1.0;
          fslope[nd] = (float)(M_SQRT2 * abs( nx )) / nz;
          contour += (float)( radii * M_PI / 4);          
          sumSlope += fslope[nd];
          neighbor[nd] = k + 1;
          nd++;
        }
      }
    }    
  }

  if( nd > 0 ) // verifica se  fosso ou borda
  {
    // verifica o valor do cit (tipo limiar de drenagem)
    if( accumulatedMatrix_[line][column] > cit_ )
    {
      float maxSlope = (float)-TeMAXFLOAT;
      int neighborMax;

      for( int k = 0; k < nd; ++k )
      {
        if( fslope[k] > maxSlope )
        {
          maxSlope = fslope[k];
          neighborMax = neighbor[k];
        }
        ftotal[0] = maxSlope;
        neighbor[0] = neighborMax;
        contour = (float)( radii * M_PI / 4);        
        nd = 1;
      }
    }
    else
    {
      // calcula as propores
      for( int k = 0; k < nd; ++k )
      {
        fslope[k] = fslope[k] / sumSlope;    
        ftotal[k] = fslope[k] * fangle[k];
      }
    }

    gridAverageSlope_ = gridAverageSlope_ + sumSlope;
    gridNSlope_ = gridNSlope_ + nd;

    // calcula o Indice Topografico    
    topographicIndexMatrix_[line][column] = 100 * (float)log( accumulatedMatrix_[line][column] / contour );
    natb_++;
    
    // distribui a rea para os vizinhos
    unsigned int neighborLine;
    unsigned int neighborColumn;
    for( int k = 0; k < nd; ++k )
    {
      neighborLine = line + lineIncrement_[neighbor[k]];
      neighborColumn = column + columnIncrement_[neighbor[k]];      

      accumulatedMatrix_[neighborLine][neighborColumn] =
      accumulatedMatrix_[neighborLine][neighborColumn] +
      accumulatedMatrix_[line][column] * ftotal[k];
    }    
  }
  else// fosso ou borda
  {
    // verificar se existe uma direo para dentro do dominio
    // no  fosso, somente impreciso no DEM
    if( lddMatrix_[line][column] != 255 )
    {
      unsigned int  lineTo = line + lineLDDIncrement_[lddMatrix_[line][column]];
      unsigned int  columnTo = column + columnLDDIncrement_[lddMatrix_[line][column]];

      if( lineTo < maxLines_ && columnTo < maxColumns_ && demMatrix_ [line][column] != demDummy_ )
      {
        //topographicIndexMatrix_[line][column] = (float)log( accumulatedMatrix_[line][column] / minSlope_*radii*M_PI/4 );
        //no codigo do javier esta assim (sem o minSlope_)
        topographicIndexMatrix_[line][column] = (float)log( accumulatedMatrix_[line][column] / radii*M_PI/4 );        

        accumulatedMatrix_[lineTo][columnTo] =
        accumulatedMatrix_[lineTo][columnTo] +
        accumulatedMatrix_[line][column];
      }
      else // realmente eh fosso ou borda
      {
        topographicIndexMatrix_[line][column] = topographicDummy_;
        natb_++;
      }
    }
    else // realmente eh fosso ou borda
    {
      topographicIndexMatrix_[line][column] = topographicDummy_;
      natb_++;
    }


    /*
    sumSlope = 0;    
    float length;
    int numSlope = 0;
    for( int k = 0; k < nd; ++k )
    {
      if( z[k] != (float)TeMAXFLOAT )
      {
        if( k%2 == 0 )
        {
          length = l;
        }
        else
        {
          length = (float)(l * M_SQRT2);
        }

        sumSlope += z[k]*length;
        numSlope++;
      }      
    }
    float meanSlope = sumSlope / numSlope;
    if( meanSlope > 0.000001 )
    {      
      topographicIndexMatrix_[line][column] = (float)log( accumulatedMatrix_[line][column] / meanSlope*radii*M_PI/4 );
      natb_++;
    }
    else
    {
      topographicIndexMatrix_[line][column] = topographicDummy_;
      natb_++;
    }*/    
  }

  
}

void
HidroTopographicIndexMDInfinite::findHillTop()
{
  // procurando por topo de morro
  // tem que ser feito depois da matriz dem inicializada por completo

  // excluindo as bordas
  for( unsigned int line=1; line<demMatrix_.GetLines()-1; line++ )
  {
    for( unsigned int column=1; column<demMatrix_.GetColumns()-1; column++ )
    {
      if( isHillTop( line, column ) && topographicIndexMatrix_[line][column] == -8888.0 )
      {
        HillTop hilltop( line, column, demMatrix_[line][column] );
        hillTopQueue_.push( hilltop );
        topographicIndexMatrix_[line][column] = topographicDummy_;
      }
    }
  }

  bool flag;

  // columns -------------------------------------------------------------------
  for( unsigned int column=1; column<demMatrix_.GetColumns()-1; column++ )
  {
    // top
    if( demMatrix_[0][column] != demDummy_ )
    {
      flag = true;
      if( demMatrix_[0][column] < demMatrix_[0][column-1] )
      {
        flag = false;
      }
      if( demMatrix_[0][column] < demMatrix_[1][column-1] )
      {
        flag = false;
      }
      if( demMatrix_[0][column] < demMatrix_[1][column] )
      {
        flag = false;
      }
      if( demMatrix_[0][column] < demMatrix_[1][column+1] )
      {
        flag = false;
      }
      if( demMatrix_[0][column] < demMatrix_[0][column+1] )
      {
        flag = false;
      }
      if( flag && topographicIndexMatrix_[0][column] == -8888.0 )
      {
        HillTop hilltop( 0, column, demMatrix_[0][column] );
        hillTopQueue_.push( hilltop );
        topographicIndexMatrix_[0][column] = topographicDummy_;
      }
    }

    // bottom
    if( demMatrix_[demMatrix_.GetLines()-1][column] != demDummy_ )
    {
      flag = true;
      if( demMatrix_[demMatrix_.GetLines()-1][column] < demMatrix_[demMatrix_.GetLines()-1][column-1] )
      {
        flag = false;
      }
      if( demMatrix_[demMatrix_.GetLines()-1][column] < demMatrix_[demMatrix_.GetLines()-2][column-1] )
      {
        flag = false;
      }
      if( demMatrix_[demMatrix_.GetLines()-1][column] < demMatrix_[demMatrix_.GetLines()-2][column] )
      {
        flag = false;
      }
      if( demMatrix_[demMatrix_.GetLines()-1][column] < demMatrix_[demMatrix_.GetLines()-2][column+1] )
      {
        flag = false;
      }
      if( demMatrix_[demMatrix_.GetLines()-1][column] < demMatrix_[demMatrix_.GetLines()-1][column+1] )
      {
        flag = false;
      }
      if( flag && topographicIndexMatrix_[demMatrix_.GetLines()-1][column] == -8888.0 )
      {
        HillTop hilltop( demMatrix_.GetLines()-1, column, demMatrix_[demMatrix_.GetLines()-1][column] );
        hillTopQueue_.push( hilltop );
        topographicIndexMatrix_[demMatrix_.GetLines()-1][column] = topographicDummy_;
      }
    }
  }

  // lines -------------------------------------------------------------------
  for( unsigned int line=1; line<demMatrix_.GetLines()-1; line++ )
  {
    // left
    if( demMatrix_[line][0] != demDummy_ )
    {
      flag = true;
      if( demMatrix_[line][0] < demMatrix_[line-1][0] )
      {
        flag = false;
      }
      if( demMatrix_[line][0] < demMatrix_[line-1][1] )
      {
        flag = false;
      }
      if( demMatrix_[line][0] < demMatrix_[line][1] )
      {
        flag = false;
      }
      if( demMatrix_[line][0] < demMatrix_[line+1][1] )
      {
        flag = false;
      }
      if( demMatrix_[line][0] < demMatrix_[line+1][0] )
      {
        flag = false;
      }
      if( flag && topographicIndexMatrix_[line][0] == -8888.0 )
      {
        HillTop hilltop( line, 0, demMatrix_[line][0] );
        hillTopQueue_.push( hilltop );
        topographicIndexMatrix_[line][0] == topographicDummy_;
      }
    }

    // rigth
    if( demMatrix_[line][demMatrix_.GetColumns()-1] != demDummy_ )
    {
      flag = true;
      if( demMatrix_[line][demMatrix_.GetColumns()-1] < demMatrix_[line-1][demMatrix_.GetColumns()-1] )
      {
        flag = false;
      }
      if( demMatrix_[line][demMatrix_.GetColumns()-1] < demMatrix_[line-1][demMatrix_.GetColumns()-2] )
      {
        flag = false;
      }
      if( demMatrix_[line][demMatrix_.GetColumns()-1] < demMatrix_[line][demMatrix_.GetColumns()-2] )
      {
        flag = false;
      }
      if( demMatrix_[line][demMatrix_.GetColumns()-1] < demMatrix_[line+1][demMatrix_.GetColumns()-2] )
      {
        flag = false;
      }
      if( demMatrix_[line][demMatrix_.GetColumns()-1] < demMatrix_[line+1][demMatrix_.GetColumns()-1] )
      {
        flag = false;
      }
      if( flag && topographicIndexMatrix_[line][demMatrix_.GetColumns()-1] == -8888.0 )
      {
        HillTop hilltop( line, demMatrix_.GetColumns()-1, demMatrix_[line][demMatrix_.GetColumns()-1] );
        hillTopQueue_.push( hilltop );
        topographicIndexMatrix_[line][demMatrix_.GetColumns()-1] == topographicDummy_;
      }
    }
  }

  // diagonais -------------------------------------------------------

  // up left
  if( demMatrix_[0][0] != demDummy_ )
  {
    flag = true;  
    if( demMatrix_[0][0] < demMatrix_[0][1] )
    {
      flag = false;
    }
    if( demMatrix_[0][0] < demMatrix_[1][0] )
    {
      flag = false;
    }
    if( demMatrix_[0][0] < demMatrix_[1][1] )
    {
      flag = false;
    }
    if( flag == true && topographicIndexMatrix_[0][0] == -8888.0 )
    {
      HillTop hilltop( 0, 0, demMatrix_[0][0] );
      hillTopQueue_.push( hilltop );
      topographicIndexMatrix_[0][0] == topographicDummy_;
    }
  }

  // up right
  if( demMatrix_[0][demMatrix_.GetLines()-1] != demDummy_ )
  {
    flag = true;  
    if( demMatrix_[0][demMatrix_.GetLines()-1] < demMatrix_[0][demMatrix_.GetColumns()-2] )
    {
      flag = false;
    }
    if( demMatrix_[0][demMatrix_.GetLines()-1] < demMatrix_[1][demMatrix_.GetColumns()-2] )
    {
      flag = false;
    }
    if( demMatrix_[0][demMatrix_.GetLines()-1] < demMatrix_[1][demMatrix_.GetColumns()-1] )
    {
      flag = false;
    }
    if( flag == true  && topographicIndexMatrix_[0][demMatrix_.GetLines()-1] == -8888.0 )
    {
      HillTop hilltop( 0, demMatrix_.GetColumns()-1, demMatrix_[0][demMatrix_.GetColumns()-1] );
      hillTopQueue_.push( hilltop );
      topographicIndexMatrix_[0][demMatrix_.GetLines()-1] == topographicDummy_;
    }
  }

    // down left
  if( demMatrix_[demMatrix_.GetLines()-1][0] != demDummy_ )
  {
    flag = true;  
    if( demMatrix_[demMatrix_.GetLines()-1][0] < demMatrix_[demMatrix_.GetLines()-2][0] )
    {
      flag = false;
    }
    if( demMatrix_[demMatrix_.GetLines()-1][0] < demMatrix_[demMatrix_.GetLines()-2][1] )
    {
      flag = false;
    }
    if( demMatrix_[demMatrix_.GetLines()-1][0] < demMatrix_[demMatrix_.GetLines()-1][1] )
    {
      flag = false;
    }
    if( flag == true  && topographicIndexMatrix_[demMatrix_.GetLines()-1][0] == -8888.0 )
    {
      HillTop hilltop( demMatrix_.GetLines()-1, 0, demMatrix_[demMatrix_.GetLines()-1][0] );
      hillTopQueue_.push( hilltop );
      topographicIndexMatrix_[demMatrix_.GetLines()-1][0] == topographicDummy_;
    }
  }

    // down rigth
  if( demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] != demDummy_ )
  {
    flag = true;  
    if( demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] < demMatrix_[demMatrix_.GetLines()-2][demMatrix_.GetColumns()-2] )
    {
      flag = false;
    }
    if( demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] < demMatrix_[demMatrix_.GetLines()-2][demMatrix_.GetColumns()-1] )
    {
      flag = false;
    }
    if( demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] < demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-2] )
    {
      flag = false;
    }
    if( flag == true  && topographicIndexMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] == -8888.0 )
    {
      HillTop hilltop( demMatrix_.GetLines()-1, demMatrix_.GetColumns()-1, demMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] );
      hillTopQueue_.push( hilltop );
      topographicIndexMatrix_[demMatrix_.GetLines()-1][demMatrix_.GetColumns()-1] == topographicDummy_;
    }
  }
}

bool
HidroTopographicIndexMDInfinite::isHillTop( unsigned int line, unsigned int column )
{
  if( demMatrix_[line][column] == demDummy_ )
  {
    return false;
  }

  // top left 
  if( demMatrix_[line][column] < demMatrix_[line - 1][column - 1] )
  {
    return false;
  }

  // top 
  if( demMatrix_[line][column] < demMatrix_[line - 1][column] )
  {
    return false;
  }

  // top right
  if( demMatrix_[line][column] < demMatrix_[line - 1][column + 1] )
  {
    return false;
  }

  // letf
  if( demMatrix_[line][column] < demMatrix_[line][column - 1] )
  {
    return false;
  }

  // rigth
  if( demMatrix_[line][column] < demMatrix_[line][column + 1] )
  {
    return false;
  }

  // bottom left
  if( demMatrix_[line][column] < demMatrix_[line + 1][column - 1] )
  {
    return false;
  }

  // bottom 
  if( demMatrix_[line][column] < demMatrix_[line + 1][column] )
  {
    return false;
  }
  
  // bottom right
  if( demMatrix_[line][column] < demMatrix_[line + 1][column + 1] )
  {
    return false;
  }
  
  return true;
}

HillTop::HillTop()
{
}

HillTop::HillTop( unsigned int line, unsigned int column, float altimetria ) :
line_(line),
column_(column),
altimetria_(altimetria)
{
}

bool
HillTop::operator>( const HillTop &rightSide ) const
{
  return ( this->altimetria_ < rightSide.altimetria_ );
}
