#include <HidroHydrologicalVariables.h>
#include <HidroFlowUtils.h>

#include <HidroUtils.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>
#include <qstring.h>
#include <math.h>

#define D2R(a) (a*M_PI/180)

HidroHydrologicalVariables::HidroHydrologicalVariables( TeRaster* lddRaster, TeRaster* demRaster, TeRaster* drainRaster, TeRaster* *rasterpoint, bool *flagVariables, unsigned int* options ) :
HidroFlowAlgorithm( lddRaster ),
lddRaster_(lddRaster),
demRaster_(demRaster),
drainRaster_(drainRaster),
rasterpoint_(rasterpoint),
flagVariables_(flagVariables),
options_(options)
{
}

HidroHydrologicalVariables::~HidroHydrologicalVariables()
{
  // Free Memory
  demMatrix_.clear();
  drainMatrix_.clear();
  varMatrix_.clear();
}

bool HidroHydrologicalVariables::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();  

  // load data
  TeProgress::instance()->reset();
  TeProgress::instance()->setCaption("TerraHidro");
  // load LDD
  TeProgress::instance()->setMessage("Load LDD data.");    
  if( !copyRaster2PDIMatrix( lddRaster_, lddMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    timeEnd_ = Time::instance().actualTimeString();
    timeTotal_ = Time::instance().partialString();
    return cancel();
  }
  // load DEM
  if(demRaster_ != NULL)
  {
	TeProgress::instance()->setMessage("Load DEM data.");    
    if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  // load Drainage
  if(drainRaster_ != NULL)
  {
	TeProgress::instance()->setMessage("Load Drainage data.");    
    if( !copyRaster2PDIMatrix( drainRaster_, drainMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }

  // create output raster
  
  // Raster params
  TeRasterParams RasterParams = lddRaster_->params();
  
  // Set dummy
  RasterParams.setDummy( -9999 );
  RasterParams.useDummy_ = true;  

  // Change mode
  RasterParams.mode_ = 'w';
  RasterParams.decoderIdentifier_ = "SMARTMEM";
  RasterParams.setDataType( TeFLOAT );

  // Set Max and Minimal values
  RasterParams.vmax_[0] = -TeMAXFLOAT;
  RasterParams.vmin_[0] =  TeMAXFLOAT;

  // Iniciate rasterpoint_
  for( unsigned int i=0; i<6; i++ )
  {
    if(flagVariables_[i] == true)
    {
      rasterpoint_[i] = new TeRaster( RasterParams );
      // verify if Raster created is valid
      if( !rasterpoint_[i]->init() )
      {
        errorMessage_ = rasterpoint_[i]->errorMessage();    
        timeEnd_ = Time::instance().actualTimeString();
        timeTotal_ = Time::instance().partialString();
        return cancel();
      }
	}
  }

  // compute basic matrix
  unsigned int nlines = lddRaster_->params().nlines_;
  unsigned int ncolumns = lddRaster_->params().ncols_;  
  varMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, varMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  if(flagVariables_[5]) //HAND
  {
    varMatrixb_.Reset( nlines, ncolumns, TePDIMatrix<unsigned char>::AutoMemPol, varMatrixb_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }

  if(flagVariables_[0]) //Contributing Area
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Contributing Area Raster 1/2...");
    TeProgress::instance()->setProgress( 0 );
    TeProgress::instance()->setTotalSteps( nlines );
    AreaCalcMatrix();
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Contributing Area Raster 2/2...");
    TeProgress::instance()->setProgress( 0 );
    if( !Accumulation() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[0] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;
  if(flagVariables_[1]) //Level Difference from the Top
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    if( !copyRaster2PDIMatrix( demRaster_, varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Level Difference from the Top Raster 1/2...");
    TeProgress::instance()->setProgress( 0 );
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {      
        if( connectionsMatrix_[line][column] == 0 )
        {
          lineFrom = line;
          columnFrom = column;
          while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
          {
            if( lddMatrix_[lineTo][columnTo] == 255 )
            {
              break;
            }
            varMatrix_[lineTo][columnTo] = max(varMatrix_[lineFrom][columnFrom],varMatrix_[lineTo][columnTo]);         
            // if connections bigger than one
            if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
            {
              connectionsMatrix_[lineTo][columnTo]--;
              break;
            }
            lineFrom = lineTo;
            columnFrom = columnTo;
          }        
        }
      }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Level Difference from the Top Raster 2/2...");
    TeProgress::instance()->setProgress( 0 );
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
		varMatrix_[line][column] -= demMatrix_[line][column];
		if(varMatrix_[line][column] < 0)
		{
		  varMatrix_[line][column] = 0;
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
    }
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[1] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  if(flagVariables_[2]) //Level Difference in Flow Direction
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    if( !copyRaster2PDIMatrix( demRaster_, varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Level Difference in Flow Direction Raster...");
    TeProgress::instance()->setProgress( 0 );
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {      
        lineFrom = line;
        columnFrom = column;
		bool flag = lddCode2LineColumnSteps( lineFrom, columnFrom, lineTo, columnTo, options_[1] );
 	  	varMatrix_[line][column] -= demMatrix_[lineTo][columnTo];
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[2] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  if(flagVariables_[3]) //Slope in Flow Direction
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    if( !copyRaster2PDIMatrix( demRaster_, varMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Slope in Flow Direction Raster...");
    TeProgress::instance()->setProgress( 0 );
    float resx = (float) lddRaster_->params().resx_;
    float resy = (float) lddRaster_->params().resy_;
    double A = lddRaster_->params().projection()->datum().radius() / 1000.0;
    double F = lddRaster_->params().projection()->datum().flattening();  
    double E2 = 2 * F - F * F; //!QUADRADO DA EXCENTRICIDADE
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
		if(varMatrix_[line][column] != -9999)
		{
          lineFrom = line;
          columnFrom = column;
		  vector <unsigned int> linesVec, columnsVec;
		  bool flag = HidroHydrologicalVariables::lddCode2LineColumnVec( lineFrom, columnFrom, options_[2], linesVec, columnsVec);
//		  bool flag = lddCode2LineColumnSteps( lineFrom, columnFrom, lineTo, columnTo, options_[1] );
		  unsigned int nvec = linesVec.size();
		  varMatrix_[line][column] -= demMatrix_[linesVec[nvec-1]][columnsVec[nvec-1]];
		  varMatrix_[line][column] = max(varMatrix_[line][column],0);
		  if(varMatrix_[line][column] != 0)
		  {
			vector <float> dx (nvec,resx);
			vector <float> dy (nvec,resy);
            if( demRaster_->params().projection()->name().compare( "LatLong" ) == 0 )
			{
              dy[0] = (float) (A * sin( resy * M_PI/180 )) * 1000;  // em metros
			  for( unsigned int i=0; i<nvec; i++ )
			  {
				TeCoord2D index( columnsVec[i], linesVec[i] );
			    TeCoord2D coord = lddRaster_->index2Coord( index );
                double YLAT = coord.y();        
                double RN =  A / sqrt( ( 1 - E2 * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) ); //!RAIO DE CURVATURA DA TERRA NA LATITUDE
				dx[i] = (float) (RN * sin( resx * M_PI/180 ) * cos( YLAT * M_PI/180 )) * 1000; // em metros;
				dy[i] = dy[0];
			  }
			}
			float length = 0;
			for( unsigned int i=1; i<nvec; i++ )
			{
			  length += (sqrt(dx[i-1]*dx[i-1]*((columnsVec[i-1] == columnsVec[i])?0:1)+dy[i-1]*dy[i-1]*((linesVec[i-1] == linesVec[i])?0:1))+sqrt(dx[i]*dx[i]*((columnsVec[i-1] == columnsVec[i])?0:1)+dy[i]*dy[i]*((linesVec[i-1] == linesVec[i])?0:1)))/2;
			}
			dx.~vector();
			dy.~vector();
			varMatrix_[line][column] = 100*varMatrix_[line][column]/length;
		  }
  		  linesVec.~vector();
		  columnsVec.~vector();
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[3] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  if(flagVariables_[4]) //Maximum Drainage Order
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating Maximum Order Raster...");
    TeProgress::instance()->setProgress( 0 );
	bool flag = MaximumOrderCalcMatrix( options_[3] );
	if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[4] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }
  if(flagVariables_[5]) //HAND
  {
    // init connections matrix
    if( !initConnections() )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
    // refresh progress bar
    TeProgress::instance()->setMessage("Creating HAND Raster 1/3...");
    TeProgress::instance()->setProgress( 0 );
	if(options_[4] != 2)
	{
	  bool flag = MaximumOrderCalcMatrix( options_[4] );
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {
		  varMatrix_[line][column] = (varMatrix_[line][column] > (options_[5]-1))?1:0;
		}
	  }
	}
	else
	{
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {
		  varMatrix_[line][column] = ((drainMatrix_[line][column] != 0) && (drainMatrix_[line][column] != 255))?1:0;
		}
	  }
	}
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {      
		if((varMatrix_[line][column] > 0))
		{
		  varMatrixb_[line][column] = 1;
		  varMatrix_[line][column] = demMatrix_[line][column];
		}
		else if((demMatrix_[line][column] == -9999) || (lddMatrix_[line][column] == 0))
		{
		  varMatrixb_[line][column] = 1;
		  varMatrix_[line][column] = -9999;
		}
		else
		{
		  varMatrixb_[line][column] = 0;
		  varMatrix_[line][column] = -9999;
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
    TeProgress::instance()->setMessage("Creating HAND Raster 2/3...");
    TeProgress::instance()->setProgress( 0 );
	float tmp;
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
		if(varMatrixb_[line][column] == 0)
		{
	      tmp = varMatrix_[line][column];
          lineFrom = line;
          columnFrom = column;
	      while ((varMatrixb_[lineFrom][columnFrom] != 1) && lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ))
		  {
		    tmp = varMatrix_[lineTo][columnTo];
            lineFrom = lineTo;
            columnFrom = columnTo;
		  }
          lineFrom = line;
          columnFrom = column;
	      while ((varMatrixb_[lineFrom][columnFrom] != 1) && lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ))
		  {
		    varMatrixb_[lineFrom][columnFrom] = 1;
		    varMatrix_[lineFrom][columnFrom] = tmp;
            lineFrom = lineTo;
            columnFrom = columnTo;
		  }
		  varMatrixb_[lineFrom][columnFrom] = 1;
		  varMatrix_[lineFrom][columnFrom] = tmp;
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
    TeProgress::instance()->setMessage("Creating HAND Raster 3/3...");
    TeProgress::instance()->setProgress( 0 );
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
		if(varMatrix_[line][column] != -9999)
		{
		  varMatrix_[line][column] = max(demMatrix_[line][column]-varMatrix_[line][column],0);
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[5] ) )
    {
      timeEnd_ = Time::instance().actualTimeString();
      timeTotal_ = Time::instance().partialString();
      return cancel();
    }
  }

  // Free Memory
  demMatrix_.clear();
  lddMatrix_.clear();
  varMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroHydrologicalVariables::cancel()
{
  // Free Memory
  demMatrix_.clear();
  lddMatrix_.clear();
  varMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}

void HidroHydrologicalVariables::AreaCalcMatrix()
{
  unsigned int nlines = lddRaster_->params().nlines_;
  unsigned int ncolumns = lddRaster_->params().ncols_;  
  float resx = (float) lddRaster_->params().resx_;
  float resy = (float) lddRaster_->params().resy_;
  if(options_[0] == 1)
  {
	resx /= 1000;
	resy /= 1000;
  }
  else if(options_[0] == 2)
  {
	resx = 1;
	resy = 1;
  }
  if( (lddRaster_->params().projection()->name().compare( "LatLong" ) == 0 ) && (options_[0] != 2))
  {
    double A = lddRaster_->params().projection()->datum().radius() / 1000.0;
    double F = lddRaster_->params().projection()->datum().flattening();  
    double E2 = 2 * F - F * F; //!QUADRADO DA EXCENTRICIDADE
    double DY = (A * sin( resy * M_PI/180 )) * 1000;  // em metros
  
    for( unsigned int line=0; line<nlines; line++ )
    {
      // A largura varia somente na linha / y / latitude
      TeCoord2D index( 0, line );
      TeCoord2D coord = lddRaster_->index2Coord( index );

      double YLAT = coord.y();        
      double RN =  A / sqrt( ( 1 - E2 * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) ); //!RAIO DE CURVATURA DA TERRA NA LATITUDE
      double DX = (RN * sin( resx * M_PI/180 ) * cos( YLAT * M_PI/180 )) * 1000; // em metros

      for( unsigned int column=0; column<ncolumns; column++ )
      {      
	    varMatrix_[line][column] = DX * DY;
      }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return;
      }
	}
  }
  else
  {
    for( unsigned int line=0; line<nlines; line++ )
    {
      for( unsigned int column=0; column<ncolumns; column++ )
      {
	    varMatrix_[line][column] = resx * resy;
	  }
	}
  }
}

bool HidroHydrologicalVariables::Accumulation()
{
  // Calculate the accumulated matrix
  unsigned int nlines = lddRaster_->params().nlines_;
  unsigned int ncolumns = lddRaster_->params().ncols_;  
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;
  for( unsigned int line=0; line<nlines; line++ )
  {
    for( unsigned int column=0; column<ncolumns; column++ )
    {      
      if( connectionsMatrix_[line][column] == 0 )
      {
        lineFrom = line;
        columnFrom = column;
        while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
        {
          if( lddMatrix_[lineTo][columnTo] == 255 )
          {
            break;
          }

          varMatrix_[lineTo][columnTo] = varMatrix_[lineTo][columnTo] + varMatrix_[lineFrom][columnFrom];          

          // if connections bigger than one
          if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
          {
            connectionsMatrix_[lineTo][columnTo]--;
            break;
          }
          
          lineFrom = lineTo;
          columnFrom = columnTo;
        }        
      }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }
  return true;
}

bool HidroHydrologicalVariables::lddCode2LineColumnSteps( const unsigned int &lineFrom, const unsigned int &columnFrom, unsigned int &lineTo, unsigned int &columnTo, const unsigned int &steps)
{
  bool flag;
  unsigned int line;
  unsigned int column;
  line = lineFrom;
  column = columnFrom;
  for( unsigned int i=0; i<steps; i++ )
  {
	flag = lddCode2LineColumn( line, column, lineTo, columnTo );
    if( !flag )
    {
      lineTo = line;
      columnTo = column;
	  break;
    }
    line = lineTo;
    column = columnTo;
  }
  return flag;
}

unsigned int HidroHydrologicalVariables::lddCode2LineColumnVec( const unsigned int &lineIni, const unsigned int &columnIni, const unsigned int &steps, vector <unsigned int> &linesVec, vector <unsigned int> &columnsVec)
{
  bool flag;
  unsigned int line, lineTo;
  unsigned int column, columnTo;
  line = lineIni;
  column = columnIni;
  linesVec.push_back(line);
  columnsVec.push_back(column);
  for( unsigned int i=0; i<steps; i++ )
  {
	flag = lddCode2LineColumn( line, column, lineTo, columnTo );
    if( !flag )
    {
//      lineTo = line;
//      columnTo = column;
	  break;
    }
    line = lineTo;
    column = columnTo;
    linesVec.push_back(line);
    columnsVec.push_back(column);
  }
//  if(flag && (linesVec.size() > 1))
//  {
//	  bool b = flag;
//  }
  return flag;
}

bool HidroHydrologicalVariables::MaximumOrderCalcMatrix( const unsigned int &option)
{
  unsigned int nlines = lddRaster_->params().nlines_;
  unsigned int ncolumns = lddRaster_->params().ncols_;  
  unsigned int lineFrom;
  unsigned int columnFrom;
  unsigned int lineTo;
  unsigned int columnTo;
	if(option == 0) //Strahler
	{
      // refresh progress bar
      TeProgress::instance()->setMessage("Creating Maximum Strahler Order Raster 1/2...");
      TeProgress::instance()->setProgress( 0 );
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {
	      varMatrix_[line][column] = 0;
        }
        // atualiza barra de progresso
        TeProgress::instance()->setProgress( line );
        // verifica se usurio cancelou a operao
        if( TeProgress::instance()->wasCancelled() )
        {
          return false;
        }
      }
      TeProgress::instance()->setMessage("Creating Maximum Strahler Order Raster 2/2...");
      TeProgress::instance()->setProgress( 0 );
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {      
          if( connectionsMatrix_[line][column] == 0 )
          {
	  	    varMatrix_[line][column] = 1;
            lineFrom = line;
            columnFrom = column;
            while( lddCode2LineColumn( lineFrom, columnFrom, lineTo, columnTo ) )
            {
              if( lddMatrix_[lineTo][columnTo] == 255 )
              {
                break;
              }
			  if( varMatrix_[lineTo][columnTo] == varMatrix_[lineFrom][columnFrom] )
			  {
			    varMatrix_[lineTo][columnTo]++;
			  }
			  else if( varMatrix_[lineTo][columnTo] < varMatrix_[lineFrom][columnFrom] )
			  {
                varMatrix_[lineTo][columnTo] = varMatrix_[lineFrom][columnFrom]; 
			  }
              // if connections bigger than one
              if( ((int)connectionsMatrix_[lineTo][columnTo]) > 1 )
              {
                connectionsMatrix_[lineTo][columnTo]--;
                break;
              }
              lineFrom = lineTo;
              columnFrom = columnTo;
            }        
          }
        }
        // atualiza barra de progresso
        TeProgress::instance()->setProgress( line );
        // verifica se usurio cancelou a operao
        if( TeProgress::instance()->wasCancelled() )
        {
          return false;
        }
      }
	}
	else //Shreve
	{
      // refresh progress bar
      TeProgress::instance()->setMessage("Creating Maximum Shreve Order Raster 1/2...");
      TeProgress::instance()->setProgress( 0 );
      for( unsigned int line=0; line<nlines; line++ )
      {
        for( unsigned int column=0; column<ncolumns; column++ )
        {
          if( connectionsMatrix_[line][column] == 0 )
          {
	        varMatrix_[line][column] = 1;
		  }
		  else
          {
	        varMatrix_[line][column] = 0;
		  }
        }
        // atualiza barra de progresso
        TeProgress::instance()->setProgress( line );
        // verifica se usurio cancelou a operao
        if( TeProgress::instance()->wasCancelled() )
        {
          return false;
        }
      }
      TeProgress::instance()->setMessage("Creating Maximum Shreve Order Raster 2/2...");
      TeProgress::instance()->setProgress( 0 );
      if( !Accumulation() )
      {
        timeEnd_ = Time::instance().actualTimeString();
        timeTotal_ = Time::instance().partialString();
        return false;
      }
	}
  return true;
}
