// Formulas
// http://www.soi.city.ac.uk/~jwo/phd/

#include <HidroMorphometricVariables.h>

#include <HidroUtils.h>
#include <HidroFlowUtils.h>

#include <TeRaster.h>
#include <TePDIMatrix.hpp>
#include <qstring.h>
#include <math.h>

HidroMorphometricVariables::HidroMorphometricVariables( TeRaster* demRaster, TeRaster* *rasterpoint, bool *flagVariables ) :
HidroFlowAlgorithm( demRaster ),
demRaster_(demRaster),
rasterpoint_(rasterpoint),
flagVariables_(flagVariables)
{
}

HidroMorphometricVariables::~HidroMorphometricVariables()
{
  // Free Memory
  demMatrix_.clear();
  pMatrix_.clear();
  qMatrix_.clear();
  rMatrix_.clear();
  sMatrix_.clear();
  tMatrix_.clear();
  varMatrix_.clear();
  meancurvMatrix_.clear();
  unspherMatrix_.clear();
  horcurvMatrix_.clear();
  diffcurvMatrix_.clear();
}

bool HidroMorphometricVariables::execute()
{
  // save start time
  Time::instance().start();
  timeStart_ = Time::instance().actualTimeString();  

  // load data

  // load DEM
  TeProgress::instance()->reset();
	TeProgress::instance()->setCaption("TerraHidro");
	TeProgress::instance()->setMessage("Load DEM data.");    
  if( !copyRaster2PDIMatrix( demRaster_, demMatrix_, MATRIX_MAX_PERCENT_USAGE ) )
  {
    return cancel();
  }

  // Raster params
  TeRasterParams RasterParams = demRaster_->params();
  
  // Set dummy
  RasterParams.setDummy( -9999 );
  RasterParams.useDummy_ = true;  

  // Change mode
  RasterParams.mode_ = 'w';
  RasterParams.decoderIdentifier_ = "SMARTMEM";
  RasterParams.setDataType( TeFLOAT );

  // Set Max and Minimal values
  RasterParams.vmax_[0] = -TeMAXFLOAT;
  RasterParams.vmin_[0] =  TeMAXFLOAT;

  // start the progress bar
  TeProgress::instance()->reset();
  TeProgress::instance()->setCaption("TerraHidro");	

  // STEP 1
  TeProgress::instance()->setMessage("Creating Basic Raster...");  

  // compute basic matrix
  unsigned int nlines = demRaster_->params().nlines_;
  unsigned int ncolumns = demRaster_->params().ncols_;  
  varMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, varMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  for( unsigned int line=0; line<nlines; line++ )
  {
	varMatrix_[line][0] = -9999;
	varMatrix_[line][ncolumns-1] = -9999;
  }
  for( unsigned int column=0; column<ncolumns; column++ )
  {
    varMatrix_[0][column] = -9999;
	varMatrix_[nlines-1][column] = -9999;
  }

  pMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, pMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );

  qMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, qMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );

  bool rstflag = false;
  unsigned int i=2;
  while ((rstflag == false) && (i<18))
  {
	  rstflag = flagVariables_[i];
	  i++;
  }

  if(rstflag)
  {
	rMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, rMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
	sMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, sMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
	tMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, tMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }
  double resx = 1;
  double resy = 1;
  if( demRaster_->params().projection()->name().compare( "LatLong" ) != 0 )
  {
	double resx = demRaster_->params().resx_;
	double resy = demRaster_->params().resy_;
  }  

  double z1, z2, z3, z4, z5, z6, z7, z8, z9;

  TeProgress::instance()->setTotalSteps( nlines );
  for( unsigned int line=1; line<nlines-1; line++ )
  {
    for( unsigned int column=1; column<ncolumns-1; column++ )
    {      
      if( demMatrix_[line][column] != -9999 )
      {
        if( demMatrix_[line-1][column-1] != -9999 )
        {
			z1 = demMatrix_[line-1][column-1];
        }
		else
		{
			z1 = demMatrix_[line][column];
		}
        if( demMatrix_[line-1][column] != -9999 )
        {
			z2 = demMatrix_[line-1][column];
        }
		else
		{
			z2 = demMatrix_[line][column];
		}
        if( demMatrix_[line-1][column+1] != -9999 )
        {
			z3 = demMatrix_[line-1][column+1];
        }
		else
		{
			z3 = demMatrix_[line][column];
		}
        if( demMatrix_[line][column-1] != -9999 )
        {
			z4 = demMatrix_[line][column-1];
        }
		else
		{
			z4 = demMatrix_[line][column];
		}
        if( demMatrix_[line][column+1] != -9999 )
        {
			z6 = demMatrix_[line][column+1];
        }
		else
		{
			z6 = demMatrix_[line][column];
		}
        if( demMatrix_[line+1][column-1] != -9999 )
        {
			z7 = demMatrix_[line+1][column-1];
        }
		else
		{
			z7 = demMatrix_[line][column];
		}
        if( demMatrix_[line+1][column] != -9999 )
        {
			z8 = demMatrix_[line+1][column];
        }
		else
		{
			z8 = demMatrix_[line][column];
		}
        if( demMatrix_[line+1][column+1] != -9999 )
        {
			z9 = demMatrix_[line+1][column+1];
        }
		else
		{
			z9 = demMatrix_[line][column];
		}
		pMatrix_[line][column] = (z3+z6+z9-z1-z4-z7) / (6*resx);
		qMatrix_[line][column] = (z1+z2+z3-z7-z8-z9) / (6*resy);
		if( rstflag == true)
		{
		  z5 = demMatrix_[line][column];
		  rMatrix_[line][column] = (z1+z3+z4+z6+z7+z9 - 2*(z2+z5+z8)) / (3*resx*resx);
		  sMatrix_[line][column] = (z3+z7-z1-z9) / (4*resx*resy);
		  tMatrix_[line][column] = (z1+z2+z3+z7+z8+z9 - 2*(z4+z5+z6)) / (3*resy*resy);
		}

      }
	  else
	  {
		varMatrix_[line][column] = -9999;
	  }
    }
    // atualiza barra de progresso
    TeProgress::instance()->setProgress( line );

    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
  }

   // corrige resolucao caso a projecao seja LatLong
  if( demRaster_->params().projection()->name().compare( "LatLong" ) == 0 )
  {
    // STEP 2
	TeProgress::instance()->setMessage("Calculating spatial resolution...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );

    double A = demRaster_->params().projection()->datum().radius() / 1000.0;
    double F = demRaster_->params().projection()->datum().flattening();  
    double E2 = 2 * F - F * F; //!QUADRADO DA EXCENTRICIDADE
    double resx = demRaster_->params().resx_;
    double resy = demRaster_->params().resy_;
    double DY = (A * sin( resy * M_PI/180 )) * 1000;  // em metros
  
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      // A largura varia somente na linha / y / latitude
      TeCoord2D index( 0, line );
      TeCoord2D coord = demRaster_->index2Coord( index );

      double YLAT = coord.y();        
      double RN =  A / sqrt( ( 1 - E2 * sin(YLAT * M_PI/180)*sin(YLAT * M_PI/180) ) ); //!RAIO DE CURVATURA DA TERRA NA LATITUDE
      double DX = (RN * sin( resx * M_PI/180 ) * cos( YLAT * M_PI/180 )) * 1000; // em metros

      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
		  pMatrix_[line][column] = pMatrix_[line][column] / DX;
		  qMatrix_[line][column] = qMatrix_[line][column] / DY;
		  if( rstflag == true)
		  {
		    rMatrix_[line][column] = rMatrix_[line][column] / (DX*DX);
		    sMatrix_[line][column] = sMatrix_[line][column] / (DX*DY);
		    tMatrix_[line][column] = tMatrix_[line][column] / (DY*DY);
		  }
        }      
      }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );

      // verifica se usurio cancelou a operao
      if( TeProgress::instance()->wasCancelled() )
      {
        return cancel();
      }
	}
  }
  
  // Free Memory
  demMatrix_.clear();

  bool meancurvflag = flagVariables_[5] && (flagVariables_[9] || flagVariables_[10] || flagVariables_[11] || flagVariables_[17]);
  if(meancurvflag)
  {
	meancurvMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, meancurvMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }
  bool unspherflag = flagVariables_[8] && (flagVariables_[9] || flagVariables_[10] || flagVariables_[12] || flagVariables_[13]);
  if(unspherflag)
  {
	unspherMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, unspherMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }
  bool horcurvflag = flagVariables_[2] && flagVariables_[11];
  if(horcurvflag)
  {
	horcurvMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, horcurvMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }
  bool diffcurvflag = flagVariables_[11] && (flagVariables_[12] || flagVariables_[13] || flagVariables_[17]);
  if(diffcurvflag)
  {
	diffcurvMatrix_.Reset( nlines, ncolumns, TePDIMatrix<float>::AutoMemPol, diffcurvMatrix_.getMaxTmpFileSize(), MATRIX_MAX_PERCENT_USAGE );
  }

  for( unsigned int i=0; i<18; i++ )
  {
    if(flagVariables_[i])
    {
      rasterpoint_[i] = new TeRaster( RasterParams );
      // verify if Raster created is valid
      if( !rasterpoint_[i]->init() )
      {
        errorMessage_ = rasterpoint_[i]->errorMessage();    
        timeEnd_ = Time::instance().actualTimeString();
        timeTotal_ = Time::instance().partialString();
        return false;
      }
	}
  }

  float cte, cte2;
  if(flagVariables_[0]) //compute Slope
  {
    TeProgress::instance()->setMessage("Computing Slope 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
          varMatrix_[line][column] = (float) 100*sqrt( pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column] );
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Slope 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[0] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[1]) //compute Gradient Factor
  {
    TeProgress::instance()->setMessage("Computing Gradient Factor 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
		  cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  varMatrix_[line][column] = (float) 100*sqrt( cte )/sqrt( 1+cte );
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Gradient Factor 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[1] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[2]) //compute Horizontal Curvature
  {
    TeProgress::instance()->setMessage("Computing Horizontal Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(cte*sqrt(1+cte));
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Horizontal Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[2] ) )
    {
      return cancel();
    }
	if(horcurvflag) //persistindo o valor de Horizontal Curvature para ser usado por outra variavel
	{
      for( unsigned int line=1; line<nlines-1; line++ )
      {
        for( unsigned int column=1; column<ncolumns-1; column++ )
        {
		  horcurvMatrix_[line][column] = varMatrix_[line][column];
		}
	  }
	}
  }
  if(flagVariables_[3]) //compute Plan Curvature
  {
    TeProgress::instance()->setMessage("Computing Plan Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(pow(cte,(float) 1.5));
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Plan Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[3] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[4]) //compute Vertical Curvature
  {
    TeProgress::instance()->setMessage("Computing Vertical Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column] + 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column])/(cte*pow(1+cte,(float) 1.5));
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Vertical Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[4] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[5]) //compute Mean Curvature
  {
    TeProgress::instance()->setMessage("Computing Mean Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Mean Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[5] ) )
    {
      return cancel();
    }
	if(meancurvflag) //persistindo o valor de Mean Curvature para ser usado por outra variavel
	{
      for( unsigned int line=1; line<nlines-1; line++ )
      {
        for( unsigned int column=1; column<ncolumns-1; column++ )
        {
		  meancurvMatrix_[line][column] = varMatrix_[line][column];
		}
	  }
	}
  }
  if(flagVariables_[6]) //compute Longitudinal Curvature
  {
    TeProgress::instance()->setMessage("Computing Longitudinal Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column] + 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column])/cte;
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Longitudinal Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[6] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[7]) //compute Cross-Sectional Curvature
  {
    TeProgress::instance()->setMessage("Computing Cross-Sectional Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) -1000*(tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column])/cte;
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Cross-Sectional Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[7] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[8]) //compute Unsphericity
  {
    TeProgress::instance()->setMessage("Computing Unsphericity 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
		    cte2 = sqrt((1+qMatrix_[line][column]*qMatrix_[line][column])/(1+pMatrix_[line][column]*pMatrix_[line][column]));
			varMatrix_[line][column] = (float) 1000*sqrt((pow(rMatrix_[line][column]*cte2 - tMatrix_[line][column]/cte2,(float) 2))/(1+cte) + pow(pMatrix_[line][column]*qMatrix_[line][column]*rMatrix_[line][column]*cte2 - 2*sMatrix_[line][column]*cte2 + pMatrix_[line][column]*qMatrix_[line][column]*tMatrix_[line][column]/cte2,(float) 2))/(2*pow(1+cte,(float) 1.5));
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Unsphericity 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[8] ) )
    {
      return cancel();
    }
	if(unspherflag) //persistindo o valor de Unsphericity para ser usado por outra variavel
	{
      for( unsigned int line=1; line<nlines-1; line++ )
      {
        for( unsigned int column=1; column<ncolumns-1; column++ )
        {
		  unspherMatrix_[line][column] = varMatrix_[line][column];
		}
	  }
	}
  }
  if(flagVariables_[9]) //compute Minimal Curvature
  {
    TeProgress::instance()->setMessage("Computing Minimal Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(meancurvflag)
		  {
			 varMatrix_[line][column] = meancurvMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] = (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			}
			else
			{
			  varMatrix_[line][column] = 0;
			}
		  }
		  if(unspherflag)
		  {
		    varMatrix_[line][column] -= unspherMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
		      cte2 = sqrt((1+qMatrix_[line][column]*qMatrix_[line][column])/(1+pMatrix_[line][column]*pMatrix_[line][column]));
			  varMatrix_[line][column] -= (float) (1000*sqrt((pow(rMatrix_[line][column]*cte2 - tMatrix_[line][column]/cte2,(float) 2))/(1+cte) + pow(pMatrix_[line][column]*qMatrix_[line][column]*rMatrix_[line][column]*cte2 - 2*sMatrix_[line][column]*cte2 + pMatrix_[line][column]*qMatrix_[line][column]*tMatrix_[line][column]/cte2,(float) 2))/(2*pow(1+cte,(float) 1.5)));
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Minimal Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[9] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[10]) //compute Maximal Curvature
  {
    TeProgress::instance()->setMessage("Computing Maximal Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(meancurvflag)
		  {
			 varMatrix_[line][column] = meancurvMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] = (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			}
			else
			{
			   varMatrix_[line][column] = 0;
			}
		  }
		  if(unspherflag)
		  {
		    varMatrix_[line][column] += unspherMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
		      cte2 = sqrt((1+qMatrix_[line][column]*qMatrix_[line][column])/(1+pMatrix_[line][column]*pMatrix_[line][column]));
			  varMatrix_[line][column] += (float) (1000*sqrt((pow(rMatrix_[line][column]*cte2 - tMatrix_[line][column]/cte2,(float) 2))/(1+cte) + pow(pMatrix_[line][column]*qMatrix_[line][column]*rMatrix_[line][column]*cte2 - 2*sMatrix_[line][column]*cte2 + pMatrix_[line][column]*qMatrix_[line][column]*tMatrix_[line][column]/cte2,(float) 2))/(2*pow(1+cte,(float) 1.5)));
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Maximal Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[10] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[11]) //compute Difference Curvature
  {
    TeProgress::instance()->setMessage("Computing Difference Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(horcurvflag)
		  {
		    varMatrix_[line][column] = horcurvMatrix_[line][column];
		  }
		  else
		  {
		    if(cte != 0)
		    {
			  varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(cte*sqrt(1+cte));
			}
			else
			{
			   varMatrix_[line][column] = 0;
			}
		  }
		  if(meancurvflag)
		  {
		    varMatrix_[line][column] -= meancurvMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] -= (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Difference Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[11] ) )
    {
      return cancel();
    }
	if(diffcurvflag) //persistindo o valor de Difference Curvature para ser usado por outra variavel
	{
      for( unsigned int line=1; line<nlines-1; line++ )
      {
        for( unsigned int column=1; column<ncolumns-1; column++ )
        {
		  diffcurvMatrix_[line][column] = varMatrix_[line][column];
		}
	  }
	}
  }
  if(flagVariables_[12]) //compute Horizontal Excess Curvature
  {
    TeProgress::instance()->setMessage("Computing Horizontal Excess Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(unspherflag)
		  {
		    varMatrix_[line][column] = unspherMatrix_[line][column];
		  }
		  else
		  {
		    if(cte != 0)
		    {
			  cte2 = sqrt((1+qMatrix_[line][column]*qMatrix_[line][column])/(1+pMatrix_[line][column]*pMatrix_[line][column]));
			  varMatrix_[line][column] = (float) (1000*sqrt((pow(rMatrix_[line][column]*cte2 - tMatrix_[line][column]/cte2,(float) 2))/(1+cte) + pow(pMatrix_[line][column]*qMatrix_[line][column]*rMatrix_[line][column]*cte2 - 2*sMatrix_[line][column]*cte2 + pMatrix_[line][column]*qMatrix_[line][column]*tMatrix_[line][column]/cte2,(float) 2))/(2*pow(1+cte,(float) 1.5)));
			}
			else
			{
			   varMatrix_[line][column] = 0;
			}
		  }
		  if(diffcurvflag)
		  {
		    varMatrix_[line][column] -= diffcurvMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] -= (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(cte*sqrt(1+cte));
			  varMatrix_[line][column] += (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Horizontal Excess Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[12] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[13]) //compute Vertical Excess Curvature
  {
    TeProgress::instance()->setMessage("Computing Vertical Excess Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(unspherflag)
		  {
		    varMatrix_[line][column] = unspherMatrix_[line][column];
		  }
		  else
		  {
		    if(cte != 0)
		    {
			  cte2 = sqrt((1+qMatrix_[line][column]*qMatrix_[line][column])/(1+pMatrix_[line][column]*pMatrix_[line][column]));
			  varMatrix_[line][column] = (float) (1000*sqrt((pow(rMatrix_[line][column]*cte2 - tMatrix_[line][column]/cte2,(float) 2))/(1+cte) + pow(pMatrix_[line][column]*qMatrix_[line][column]*rMatrix_[line][column]*cte2 - 2*sMatrix_[line][column]*cte2 + pMatrix_[line][column]*qMatrix_[line][column]*tMatrix_[line][column]/cte2,(float) 2))/(2*pow(1+cte,(float) 1.5)));
			}
			else
			{
			   varMatrix_[line][column] = 0;
			}
		  }
		  if(diffcurvflag)
		  {
		    varMatrix_[line][column] += diffcurvMatrix_[line][column];
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] += (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(cte*sqrt(1+cte));
			  varMatrix_[line][column] -= (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Vertical Excess Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[13] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[14]) //compute Rotor
  {
    TeProgress::instance()->setMessage("Computing Rotor 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) 1000*(cte*sMatrix_[line][column]-pMatrix_[line][column]*qMatrix_[line][column]*(rMatrix_[line][column]-tMatrix_[line][column]))/pow(cte,(float) 1.5);
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Rotor 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[14] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[15]) //compute Total Gaussian Curvature
  {
    TeProgress::instance()->setMessage("Computing Total Gaussian Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      varMatrix_[line][column] = (float) 1000000*(rMatrix_[line][column]*tMatrix_[line][column]-sMatrix_[line][column]*sMatrix_[line][column])/((1+cte)*(1+cte));
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Total Gaussian Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[15] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[16]) //compute Total Ring Curvature
  {
    TeProgress::instance()->setMessage("Computing Total Ring Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(cte != 0)
		  {
			varMatrix_[line][column] = (float) 1000000*pow((cte*sMatrix_[line][column]-pMatrix_[line][column]*qMatrix_[line][column]*(rMatrix_[line][column]-tMatrix_[line][column]))/(cte*(1+cte)),(float) 2);
		  }
		  else
		  {
			varMatrix_[line][column] = 0;
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Total Ring Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[16] ) )
    {
      return cancel();
    }
  }
  if(flagVariables_[17]) //compute Total Accumulation Curvature
  {
    TeProgress::instance()->setMessage("Computing Total Accumulation Curvature 1/2...");  
    // Set progress bar
    TeProgress::instance()->setTotalSteps( nlines );
    for( unsigned int line=1; line<nlines-1; line++ )
    {
      for( unsigned int column=1; column<ncolumns-1; column++ )
      {      
        if( varMatrix_[line][column] != -9999 )
        {
	      cte = pMatrix_[line][column]*pMatrix_[line][column] + qMatrix_[line][column]*qMatrix_[line][column];
		  if(diffcurvflag)
		  {
		    varMatrix_[line][column] = -diffcurvMatrix_[line][column]*diffcurvMatrix_[line][column];
		  }
		  else
		  {
		    if(cte != 0)
		    {
			  varMatrix_[line][column] = (float) -1000*(rMatrix_[line][column]*qMatrix_[line][column]*qMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + tMatrix_[line][column]*pMatrix_[line][column]*pMatrix_[line][column])/(cte*sqrt(1+cte));
			  varMatrix_[line][column] -= (float) -1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5));
			  varMatrix_[line][column] = -varMatrix_[line][column]*varMatrix_[line][column];
			}
			else
			{
			   varMatrix_[line][column] = 0;
			}
		  }
		  if(meancurvflag)
		  {
		    varMatrix_[line][column] += (meancurvMatrix_[line][column]*meancurvMatrix_[line][column]);
		  }
		  else
		  {
			if(cte != 0)
			{
			  varMatrix_[line][column] += pow(-1000*((1+qMatrix_[line][column]*qMatrix_[line][column])*rMatrix_[line][column] - 2*pMatrix_[line][column]*qMatrix_[line][column]*sMatrix_[line][column] + (1+pMatrix_[line][column]*pMatrix_[line][column])*tMatrix_[line][column])/(2*pow(1+cte,(float) 1.5)),2);
			}
		  }
		}
	  }
      // atualiza barra de progresso
      TeProgress::instance()->setProgress( line );
	}
    // verifica se usurio cancelou a operao
    if( TeProgress::instance()->wasCancelled() )
    {
      return cancel();
    }
    TeProgress::instance()->setMessage("Computing Total Accumulation Curvature 2/2...");  
    // copy matrix values to raster.
    if( !copyPDIMatrix2Raster( varMatrix_, rasterpoint_[17] ) )
    {
      return cancel();
    }
  }

  // Free Memory
  pMatrix_.clear();
  qMatrix_.clear();
  rMatrix_.clear();
  sMatrix_.clear();
  tMatrix_.clear();
  varMatrix_.clear();
  meancurvMatrix_.clear();
  unspherMatrix_.clear();
  horcurvMatrix_.clear();
  diffcurvMatrix_.clear();

  // Finish progress bar
  TeProgress::instance()->reset();  

  // save processing time
  timeEnd_ = Time::instance().actualTimeString();
  timeTotal_ = Time::instance().partialString();

  return true;
}

bool HidroMorphometricVariables::cancel()
{
  // Free Memory
  demMatrix_.clear();
  pMatrix_.clear();
  qMatrix_.clear();
  rMatrix_.clear();
  sMatrix_.clear();
  tMatrix_.clear();
  varMatrix_.clear();
  meancurvMatrix_.clear();
  unspherMatrix_.clear();
  horcurvMatrix_.clear();
  diffcurvMatrix_.clear();

  return HidroFlowAlgorithm::cancel();
}